import {
    createSlateEditor,
    GetAboveNodeOptions,
    getBlockAbove,
    getEndPoint,
    getNodeEntries,
    getPointAfter,
    getPointBefore,
    getStartPoint,
    isCollapsed,
    moveSelection,
    removeNodes,
    SlateEditor,
    TDescendant,
    TNode,
    TNodeOperation,
    TTextOperation
} from '@udecode/plate-common';
import {createPlatePlugin} from '@udecode/plate-common/react';
import {HorizontalRulePlugin} from '@udecode/plate-horizontal-rule/react';
import {BaseElement, Element, Path, Point, Range} from 'slate';

/**
 * Return true if:
 * - at start/end of a margin.
 * - next to a margin. Move selection to the page break.<br>
 * Adapted from withDeleteTable.ts in `@udecode/plate-table`.
 */
const preventDeleteMargin = (
    editor: SlateEditor, {unit, reverse}: { unit?: 'character' | 'word' | 'line' | 'block'; reverse?: boolean }
): boolean => {
    const {selection} = editor,

        getPoint = reverse ? getEndPoint : getStartPoint,
        getNextPoint = reverse ? getPointAfter : getPointBefore;

    if (isCollapsed(selection)) {
        const marginEntry = getBlockAbove(editor, {match: isHeaderOrFooter});
        if (marginEntry) {
            // Prevent deleting margin at the start or end of a margin
            const [, marginPath] = marginEntry,
                start = getPoint(editor, marginPath);
            if (selection && Point.equals(selection.anchor, start)) return true;
        } else {
            // Prevent deleting margin when selection is before or after a page break
            const nextPoint = getNextPoint(editor, selection!, {unit}),
                nextMarginEntry = getBlockAbove(editor, {
                    match: isHeaderOrFooter, at: nextPoint
                });
            if (nextMarginEntry) {
                const nextBreakEntry = getBlockAbove(editor, {
                    match: {type: PageBreakPlugin.key}, at: nextPoint
                });
                if (nextBreakEntry && !(nextBreakEntry[0].attributes as { auto? })?.auto) {
                    removeNodes(editor, {at: nextBreakEntry[1]});
                    return true;
                }
                moveSelection(editor, {reverse: !reverse});
                return true;
            }
        }
    }

    return false;
};

/**
 * Update headers and footers based on the section nodes' header and footer attributes.
 * @param editor - the editor to update headers and footers for.
 * @param operation - the operation that triggered the update.
 */
export const updateHeadersAndFooters = (
    editor: SlateEditor, operation?: TNodeOperation | TTextOperation
): void => {
    let pageNumber = 0;
    (editor.children[0].children as TElement[]).forEach((section, sectionIdx) => {
        let len = section.children.length;
        if (section.children[0].type !== PageBreakPlugin.key) {// the first header
            editor.apply({
                fromPaged: true,
                type: 'insert_node',
                path: [0, sectionIdx, 0],
                node: {type: PageBreakPlugin.key, attributes: {auto: true}, children: []}
            });
            len++;
        }
        if (section.children[section.children.length - 1].type !== PageBreakPlugin.key) {// the last footer
            editor.apply({
                fromPaged: true,
                type: 'insert_node',
                path: [0, sectionIdx, len],
                node: {type: PageBreakPlugin.key, attributes: {auto: true}, children: []}
            });
        }
        const {defaultHeader, evenHeader, firstHeader, defaultFooter, evenFooter, firstFooter} = section,
            pageBreaks = Array.from(getNodeEntries(editor, {
                at: [0, sectionIdx], match: {type: PageBreakPlugin.key}
            }));
        for (let i = 0, j = 0; i < pageBreaks.length; i++) {
            const [pageBreak, pageBreakPath] = pageBreaks[i],
                children = [] as TElement[];
            if (i > 0) {
                let footer = defaultFooter;
                if (j % 2 && evenFooter) footer = evenFooter;
                if (j++ === 0 && firstFooter) footer = firstFooter;
                children.push(updatePageNumber(JSON.parse(JSON.stringify(footer)) as TElement, pageNumber));
            }
            if (operation?.path && Path.isAncestor(pageBreakPath, operation.path)) continue;
            if (i < pageBreaks.length - 1) {
                if (i > 0) children.push({type: HorizontalRulePlugin.key, children: [{text: ''}]});
                let header = defaultHeader;
                if (j % 2 && evenHeader) header = evenHeader;
                if (j === 0 && firstHeader) header = firstHeader;
                children.push(updatePageNumber(JSON.parse(JSON.stringify(header)) as TElement, ++pageNumber));
            }
            editor.apply({
                fromPaged: true, type: 'remove_node', path: pageBreakPath, node: pageBreak as TDescendant
            });
            editor.apply({
                fromPaged: true, type: 'insert_node', path: pageBreakPath, node: {type: PageBreakPlugin.key, children}
            });
        }
    });
};

export const RootPlugin = createPlatePlugin({
    key: 'root',
    node: {isElement: true},
});

export const SectionPlugin = createPlatePlugin({
    key: 'section',
    node: {isElement: true},
});

export const PageBreakPlugin = createPlatePlugin({
    key: 'page-break',
    node: {isElement: true},
});

export const HeaderPlugin = createPlatePlugin({
    key: 'header',
    node: {isElement: true},
});

export const FooterPlugin = createPlatePlugin({
    key: 'footer',
    node: {isElement: true},
});

export const isHeaderOrFooter = (node: TNode): boolean =>
    Element.isElementType(node, HeaderPlugin.key) || Element.isElementType(node, FooterPlugin.key);

export const getHeaderOrFooterAbove = <E extends SlateEditor>(
    editor: E, options?: GetAboveNodeOptions<E>
): ReturnType<typeof getBlockAbove> => getBlockAbove(editor, {match: isHeaderOrFooter, ...options});

export const PageNumberPlugin = createPlatePlugin({
    key: 'page-number',
    node: {isElement: true, isInline: true},
});

const updatePageNumber = (node: TElement, pageNumber: number): TElement => {
    const editor = createSlateEditor({value: [node]});
    Array.from(getNodeEntries(editor, {at: [], match: {type: PageNumberPlugin.key}}))
        .forEach(([pageNumberNode]) => pageNumberNode.children = [{text: `${pageNumber}`}]);
    return editor.children[0];
};

export const PaginationPlugin = createPlatePlugin({
    key: 'pagination',
    plugins: [RootPlugin, SectionPlugin,
        PageBreakPlugin, HeaderPlugin, FooterPlugin, HorizontalRulePlugin, PageNumberPlugin],
    extendEditor: ({editor}) => {
        // Prevent auto-inserted page breaks from being deleted, adapted from withDeleteTable.ts
        const {deleteBackward, deleteForward} = editor;
        editor.deleteBackward = (unit) => {
            if (preventDeleteMargin(editor, {unit})) return;
            deleteBackward(unit);
        };
        editor.deleteForward = (unit) => {
            if (preventDeleteMargin(editor, {unit, reverse: true})) return;
            deleteForward(unit);
        };

        // Move selection to the content before/after the page break when navigating with arrow keys, TODO: up/down keys
        const {isElementReadOnly, isSelectable} = editor;
        editor.isElementReadOnly = (element) =>
            Element.isElementType(element, PageNumberPlugin.key) || isElementReadOnly(element);

        const isSelectableOutsideMargin = (element: BaseElement): boolean => !isHeaderOrFooter(element as TElement)
                && !Element.isElementType(element, PageBreakPlugin.key)
                && isSelectable(element),
            {apply} = editor;
        editor.apply = (operation) => {
            const {type} = operation;
            if (type === 'set_selection') {
                const anchor = operation.properties?.anchor,
                    newAnchor = operation.newProperties?.anchor;
                if ((!anchor || getHeaderOrFooterAbove(editor, {at: anchor}))
                    && newAnchor && !getHeaderOrFooterAbove(editor, {at: newAnchor})) {
                    editor.isSelectable = isSelectableOutsideMargin;
                }
            }
            apply(operation);
        };

        // Prevent moving out of page margin when navigating with arrow keys, adapted from move.ts, TODO: up/down keys
        const {setSelection, move} = editor;
        editor.setSelection = (props) => {
            if (props.anchor && getHeaderOrFooterAbove(editor, {at: props.anchor})) {
                editor.isSelectable = isSelectable;
            }
            setSelection(props);
        };
        editor.move = (options = {}) => {
            const {selection} = editor,
                {distance = 1, unit = 'character', reverse = false} = options;
            let {edge = undefined} = options;
            if (!selection) return;
            if (edge === 'start') {
                edge = Range.isBackward(selection) ? 'focus' : 'anchor';
            }
            if (edge === 'end') {
                edge = Range.isBackward(selection) ? 'anchor' : 'focus';
            }

            const {anchor, focus} = selection,
                opts = {distance, unit, ignoreNonSelectable: true},
                props: Partial<Range> = {};

            if (edge == null || edge === 'anchor') {
                const point = reverse ? editor.before(anchor, opts) : editor.after(anchor, opts);
                if (point) {
                    props.anchor = point;
                }
            }
            if (edge == null || edge === 'focus') {
                const point = reverse ? editor.before(focus, opts) : editor.after(focus, opts);
                if (point) {
                    props.focus = point;
                }
            }

            // prevent moving out of the margin using arrow keys
            if (props.anchor && getHeaderOrFooterAbove(editor, {at: selection.anchor})
                && !getHeaderOrFooterAbove(editor, {at: props.anchor})) return;

            move(options);
        };

        return editor;
    },
});
