import {
    getAboveNode,
    getEndPoint,
    getPointAfter,
    getPointBefore,
    getStartPoint,
    isCollapsed,
    moveSelection,
    SlateEditor
} from '@udecode/plate-common';
import {createPlatePlugin} from '@udecode/plate-common/react';
import {BaseEditor, Editor, Element, Node, Point, Text, Transforms} from 'slate';

const preventDeleteComponent = (
        editor: SlateEditor, {unit, reverse}: { unit?: 'character' | 'word' | 'line' | 'block'; reverse?: boolean }
    ): boolean => {
        const {selection} = editor,
            getPoint = reverse ? getEndPoint : getStartPoint,
            getNextPoint = reverse ? getPointAfter : getPointBefore;

        if (isCollapsed(selection)) {
            const entry = getAboveNode(editor, {match: {type: ComponentPlugin.key}});
            if (entry) {
                if (entry[0].deletable) return false;
                const [, path] = entry,
                    start = getPoint(editor, path);
                if (selection && Point.equals(selection.anchor, start)) return true;
            } else {
                const nextPoint = getNextPoint(editor, selection!, {unit}),
                    nextEntry = getAboveNode(editor, {at: nextPoint, match: {type: ComponentPlugin.key}});
                if (nextEntry) {
                    if (nextEntry[0].deletable) return false;
                    moveSelection(editor, {reverse: !reverse});
                    return true;
                }
            }
        }

        return false;
    };

export const ComponentPlugin = createPlatePlugin({
    key: 'component',
    node: {isElement: true, isInline: true, isVoid: true},
    extendEditor: ({editor}) => {
        // Prevent not deletable components from being deleted
        const {deleteBackward, deleteForward} = editor;
        editor.deleteBackward = (unit) => {
            if (preventDeleteComponent(editor, {unit})) return;
            deleteBackward(unit);
        };
        editor.deleteForward = (unit) => {
            if (preventDeleteComponent(editor, {unit, reverse: true})) return;
            deleteForward(unit);
        };
        // Prevent auto spacing components with text children
        editor.normalizeNode = (entry) => {
            const [node, path] = entry;

            // There are no core normalizations for text nodes.
            if (Text.isText(node)) return;

            // Ensure that block and inline nodes have at least one text child.
            if (Element.isElement(node) && node.children.length === 0) {
                const child = {text: ''};
                Transforms.insertNodes(editor as BaseEditor, child, {
                    at: path.concat(0),
                    voids: true,
                });
                return;
            }

            // Determine whether the node should have block or inline children.
            const shouldHaveInlines = Editor.isEditor(node)
                ? false
                : Element.isElement(node) &&
                (editor.isInline(node as TElement) ||
                    node.children.length === 0 ||
                    Text.isText(node.children[0]) ||
                    editor.isInline(node.children[0]));

            // Since we'll be applying operations while iterating, keep track of an
            // index that accounts for any added/removed nodes.
            let n = 0;

            for (let i = 0; i < node.children.length; i++, n++) {
                const currentNode = Node.get(editor, path);
                if (Text.isText(currentNode)) continue;
                const child = currentNode.children[n],
                    prev = currentNode.children[n - 1],
                    isLast = i === node.children.length - 1,
                    isInlineOrText =
                        Text.isText(child) || (Element.isElement(child) && editor.isInline(child as TElement));

                // Only allow block nodes in the top-level children and parent blocks
                // that only contain block nodes. Similarly, only allow inline nodes in
                // other inline nodes, or parent blocks that only contain inlines and
                // text.
                if (isInlineOrText !== shouldHaveInlines) {
                    Transforms.removeNodes(editor as BaseEditor, {at: path.concat(n), voids: true});
                    n--;
                } else if (Element.isElement(child)) {
                    // Ensure that inline nodes are surrounded by text nodes, except for components.
                    if (editor.isInline(child as TElement) && !Element.isElementType(child, ComponentPlugin.key)) {
                        if (prev == null || !Text.isText(prev) && !Element.isElementType(prev, ComponentPlugin.key)) {
                            const newChild = {text: ''};
                            Transforms.insertNodes(editor as BaseEditor, newChild, {
                                at: path.concat(n),
                                voids: true,
                            });
                            n++;
                        } else if (isLast) {
                            const newChild = {text: ''};
                            Transforms.insertNodes(editor as BaseEditor, newChild, {
                                at: path.concat(n + 1),
                                voids: true,
                            });
                            n++;
                        }
                    }
                } else {
                    // Merge adjacent text nodes that are empty or match.
                    if (prev != null && Text.isText(prev)) {
                        if (Text.equals(child, prev, {loose: true})) {
                            Transforms.mergeNodes(editor as BaseEditor, {at: path.concat(n), voids: true});
                            n--;
                        } else if (prev.text === '') {
                            Transforms.removeNodes(editor as BaseEditor, {
                                at: path.concat(n - 1),
                                voids: true,
                            });
                            n--;
                        } else if (child.text === '') {
                            Transforms.removeNodes(editor as BaseEditor, {
                                at: path.concat(n),
                                voids: true,
                            });
                            n--;
                        }
                    }
                }
            }
        };
        return editor;
    },
});
