import {getBlockAbove, setNodes} from '@udecode/plate-common';
import {findNodePath, PlateRenderElementProps, toDOMNode, useEditorRef, useElement} from '@udecode/plate-common/react';
import * as React from 'react';
import {useSelected} from 'slate-react';

const getPositionStyles = (position: string): React.CSSProperties => {
    switch (position) {
        case 'before_start':
            return {left: 0, top: 0, transform: 'translateY(-100%)'};
        case 'before_center':
            return {left: '50%', top: 0, transform: 'translate(-50%, -100%)'};
        case 'before_end':
            return {right: 0, top: 0, transform: 'translateY(-100%)'};
        case 'after_start':
            return {left: 0, bottom: 0, transform: 'translateY(100%)'};
        case 'after_center':
            return {left: '50%', bottom: 0, transform: 'translate(-50%, 100%)'};
        case 'after_end':
            return {right: 0, bottom: 0, transform: 'translateY(100%)'};
        case 'start_before':
            return {left: 0, top: 0, transform: 'translateX(-100%)'};
        case 'start_center':
            return {left: 0, top: '50%', transform: 'translate(-100%, -50%)'};
        case 'start_after':
            return {left: 0, bottom: 0, transform: 'translateX(-100%)'};
        case 'end_before':
            return {right: 0, top: 0, transform: 'translateX(100%)'};
        case 'end_center':
            return {right: 0, top: '50%', transform: 'translate(100%, -50%)'};
        case 'end_after':
            return {right: 0, bottom: 0, transform: 'translateX(100%)'};
        case 'top_left':
            return {left: 0, top: 0};
        case 'top_center':
            return {left: '50%', top: 0, transform: 'translateX(-50%)'};
        case 'top_right':
            return {right: 0, top: 0};
        case 'middle_left':
            return {left: 0, top: '50%', transform: 'translateY(-50%)'};
        case 'middle_center':
            return {left: '50%', top: '50%', transform: 'translate(-50%, -50%)'};
        case 'middle_right':
            return {right: 0, top: '50%', transform: 'translateY(-50%)'};
        case 'bottom_left':
            return {left: 0, bottom: 0};
        case 'bottom_center':
            return {left: '50%', bottom: 0, transform: 'translateX(-50%)'};
        case 'bottom_right':
            return {right: 0, bottom: 0};
        default:
            return {};
    }
};

export const ComponentElement = (({attributes, children}: PlateRenderElementProps): React.ReactElement => {
        const editor = useEditorRef(),
            element = useElement(),
            {containerUuid, textWrap, offsetParentType, position, left, top, width, height} = element,
            container = document.getElementById(containerUuid as string)!,
            inlineRef = React.createRef<HTMLSpanElement>(),
            isSelected = useSelected(),
            inline = textWrap === 'inline',
            breakText = textWrap === 'break_text',
            behindText = textWrap === 'behind_text';

        React.useEffect(() => {
            container.style.border = isSelected ? '1px solid blue' : '';
        }, [isSelected]);

        React.useEffect(() => {
            if (inline || breakText) {
                inlineRef.current?.appendChild(container);
                container.style.position = 'static';
                container.style.transform = 'none';
                container.style.overflow = 'auto';
            } else {
                const path = findNodePath(editor, element),
                    offsetParentNode = getBlockAbove(editor, {
                        at: path, match: offsetParentType ? {type: offsetParentType} : undefined
                    }),
                    offsetParent = offsetParentNode && toDOMNode(editor, offsetParentNode[0]);
                if (offsetParent) {
                    jq(offsetParent).children('.ghost').append(container);
                    jq(container).css({
                        position: 'absolute',
                        zIndex: behindText ? -999 : 999,
                        left: left as number, top: top as number,
                        ...getPositionStyles(position as string),
                        overflow: 'hidden',
                    });
                } else throw new Error(`Component at ${path?.toString()} has no offset parent of type ${offsetParentType as string}`);
            }
        }, [containerUuid, textWrap, offsetParentType, position, left, top]);

        React.useEffect(() => {
            container.style.overflow = inline || breakText ? 'auto' : 'hidden';
            container.style.width = width === -1 ? 'fit-content' : `${width as number}px`;
            container.style.height = height === -1 ? 'fit-content' : `${height as number}px`;
            if (width === -1 || height === -1) {
                setNodes(editor, {
                    ...(width === -1 && {width: Math.ceil(zk(container).offsetWidth())}),
                    ...(height === -1 && {height: Math.ceil(zk(container).offsetHeight())}),
                }, {at: findNodePath(editor, element)});
            }
        }, [width, height]);

        return (
            <div contentEditable={false}
                 style={{display: inline ? 'inline-block' : undefined, verticalAlign: 'text-bottom'}}>
                {(inline || breakText) && <span ref={inlineRef}/>}
                <span {...attributes}>{children}</span>
            </div>
        );
    }
);