import {Editor} from '@plate/components/editor';
import {plugins} from '@plate/create-plate-plugins';
import {createPlateUI} from '@plate/create-plate-ui';
import {
    FooterPlugin,
    HeaderPlugin,
    isHeaderOrFooter,
    PageBreakPlugin,
    RootPlugin,
    SectionPlugin,
    updateHeadersAndFooters
} from '@plate/plugins/plate-pagination';
import {
    deserializeHtml,
    getBlockAbove,
    isText,
    parseHtmlDocument,
    TDescendant,
    TOperation,
    Value
} from '@udecode/plate-common';
import {createPlateEditor, focusEditor, Plate, TPlateEditor} from '@udecode/plate-common/react';
import {TablePlugin} from '@udecode/plate-table/react';
import * as React from 'react';
import {createRoot, Root} from 'react-dom/client';
import {Element, Path} from 'slate';
import {TypedEmitter} from 'tiny-typed-emitter';

function EditorDefault({zclass, emitter, initialValue}: {
    zclass: string;
    emitter: TypedEmitter;
    initialValue: TElement[];
}): React.ReactElement {
    const editor = React.useMemo(() => {
            const editor = createPlateEditor({
                value: initialValue,
                plugins, override: {components: createPlateUI(zclass), plugins: {history: {enabled: false}}},
            }), {apply, onChange, getFragment, setFragmentData, insertData, insertFragment} = editor;
            editor.apply = operation => {
                apply(operation);
                const {type, path} = operation;
                if (operation.fromPaged || type === 'set_selection') return;
                const margin = getHeaderOrFooter(editor, type === 'set_node' ? path : path.slice(0, -1));
                if (margin) { // update headers and footers if operates on a header or footer
                    const capitalizedType = margin.type.charAt(0).toUpperCase() + margin.type.slice(1);
                    editor.node(path.slice(0, 2))[0][`${margin.pageType as string}${capitalizedType}`] = margin;
                    updateHeadersAndFooters(editor, operation);
                }
                if ((type === 'insert_node' || type === 'remove_node') && operation.node.type === PageBreakPlugin.key)
                    updateHeadersAndFooters(editor); // if inserts or removes a page break
            };
            editor.onChange = options => {
                const {operations} = editor;
                if (operations[0] && !operations[0]['fromPaged'] && !operations[0]['fromServer']) {
                    const kOperations: KOperation[] = [];
                    operations.forEach(operation => {
                        if (operation.fromPaged || operation.fromServer || operation.type === 'set_selection') return;
                        kOperations.push(toKOperation(editor, operation));
                    });
                    if (editor.selection) {
                        const start = editor.start(editor.selection),
                            end = editor.end(editor.selection);
                        kOperations.push({
                            type: 'set_selection', properties: undefined, newProperties: {
                                anchor: {path: toKPath(editor, start.path), offset: start.offset},
                                focus: {path: toKPath(editor, end.path), offset: end.offset},
                            }
                        });
                    }
                    emitter.emit('onChange', kOperations);
                }
                onChange(options);
            };
            editor.getFragment = () => {
                let fragment = getFragment();
                while (fragment[0] && (
                    [RootPlugin.key, SectionPlugin.key, HeaderPlugin.key, FooterPlugin.key] as string[]
                ).includes(fragment[0].type as string))
                    fragment = fragment.flatMap(node => node.children) as TDescendant[];
                return fragment;
            };
            editor.setFragmentData = (data, originEvent) => {
                setFragmentData(data, originEvent);
                if (originEvent) // emit onCopy, onCut, or onDrag
                    emitter.emit(`on${originEvent[0].toUpperCase()}${originEvent.slice(1)}`, editor.getFragment());
            };
            const normalizePastedTable = (fragment: TDescendant[]): TDescendant[] => {
                // wrap cell children in a paragraph if not already
                const dummyEditor = createPlateEditor({plugins: [TablePlugin]});
                return fragment.map(node => {
                    if (isText(node)) return node;
                    dummyEditor.children = [node];
                    dummyEditor.normalize({force: true});
                    return dummyEditor.children[0];
                });
            };
            editor.insertFragment = fragment => insertFragment(normalizePastedTable(fragment));
            // eslint-disable-next-line zk/noMixedHtml
            editor.insertData = data => {
                insertData(data);
                // eslint-disable-next-line zk/noMixedHtml
                const html = data.getData('text/html');
                if (html) {
                    // eslint-disable-next-line zk/noMixedHtml
                    const document = parseHtmlDocument(html),
                        deserialized = deserializeHtml(editor, {element: document.body}),
                        normalized = normalizePastedTable(deserialized);
                    emitter.emit('onPaste', normalized);
                }
            };
            emitter.on('setValue', (value: TElement[]) => {
                // editor.tf.setValue(value) without firing back to the server
                editor.apply({fromServer: true, type: 'remove_node', path: [0], node: editor.children[0]});
                editor.apply({fromServer: true, type: 'insert_node', path: [0], node: value[0]});
            });
            emitter.on('apply', (operations: KOperation[]) => {
                const normalizing = editor.isNormalizing();
                editor.setNormalizing(false);
                try {
                    let shouldUpdateHeadersAndFooters = false;
                    operations.forEach((kOperation: KOperation) => {
                        const pOperation = toPOperation(editor, kOperation),
                            {type, path} = pOperation;
                        if (type === 'merge_node' && path.length === 2) { // merge two sections
                            (editor.node(Path.previous(path))[0] as TElement).children.pop(); // remove the last footer
                            (editor.node(path)[0] as TElement).children.shift(); // remove the first header
                        }
                        pOperation['fromServer'] = true;
                        editor.apply(pOperation);
                        // if updates a section or inserts a section break
                        if ((type === 'set_node' || type === 'split_node')
                            && Element.isElementType(editor.node(path)[0], SectionPlugin.key))
                            updateHeadersAndFooters(editor);
                        shouldUpdateHeadersAndFooters ||= (type === 'insert_node' || type === 'remove_node')
                            && (pOperation.node.type === RootPlugin.key || pOperation.node.type === PageBreakPlugin.key);
                    });
                    if (shouldUpdateHeadersAndFooters) updateHeadersAndFooters(editor);
                } finally {
                    editor.setNormalizing(normalizing);
                }
            });
            emitter.on('focus', () => focusEditor(editor));
            return editor;
        }, []),
        onMouseUpHandler = React.useCallback(() => {
            // wait for the onChange (cause by onMouseUp) finishes,
            // to make sure the selection state is up-to-date,
            // then fire the onMouseUp
            setTimeout(() => emitter.emit('onMouseUp'), 10);
        }, []);
    return (
        <Plate editor={editor}>
            <Editor onMouseUp={onMouseUpHandler}/>
        </Plate>
    );
}

/**
 * Converts a path in the KKDoc model to a path in the Plate editor.
 * @param editor - the Plate editor
 * @param kPath - the path in the KKDoc model
 */
const toPPath = (editor: TPlateEditor, kPath: KPath): Path => {
        const {sectionIndex, inHeader, pageType, relativePath} = kPath;
        if (pageType) {
            const [_, headerPath] = Array.from(editor.nodes({
                at: [sectionIndex],
                match: node => (node as TElement).type === (inHeader ? HeaderPlugin.key : FooterPlugin.key)
                    && (node as TElement).pageType === pageType
            }))[0];
            return [...headerPath, ...relativePath];
        }
        if (relativePath.length) relativePath[0]++; // for the first header not in the model
        return sectionIndex === -1 ? [0] : [0, sectionIndex, ...relativePath];
    },
    /**
     * Converts a KKDoc operation to a Plate operation.
     * @param editor - the Plate editor
     * @param kOperation - the KKDoc operation
     */
    toPOperation = (editor: TPlateEditor, kOperation: KOperation): TOperation => {
        const {type} = kOperation, {path, newPath, ...pOperation} = kOperation;
        if (type === 'set_selection') {
            const {newProperties} = kOperation;
            if (newProperties) {
                const {anchor, focus} = newProperties as KRange;
                pOperation.newProperties = {
                    anchor: {path: toPPath(editor, anchor.path), offset: anchor.offset},
                    focus: {path: toPPath(editor, focus.path), offset: focus.offset},
                };
            }
        }
        if (path) pOperation.path = toPPath(editor, path as KPath);
        if (newPath) pOperation.newPath = toPPath(editor, newPath as KPath);
        if (type === 'split_node' && (pOperation.path as Path).length === 2)
            (pOperation.position as number)++; // for the first header not in the model
        pOperation['fromServer'] = true;
        return pOperation as TOperation;
    },
    /**
     * Converts a path in the Plate editor to a path in the KKDoc model.
     * @param editor - the Plate editor
     * @param pPath - the path in the Plate editor
     */
    toKPath = (editor: TPlateEditor, pPath: Path): KPath => {
        const [_, sectionIndex] = pPath,
            parentPath = pPath.slice(0, -1),
            margin = getHeaderOrFooter(editor, parentPath),
            relativePath = pPath.slice(2); // skip [root, section]
        if (margin) {
            return {
                sectionIndex,
                inHeader: margin.type === HeaderPlugin.key,
                inFooter: margin.type === FooterPlugin.key,
                pageType: margin.pageType as string | undefined,
                relativePath: relativePath.slice(2) // skip [page-break, header/footer]
            };
        }
        relativePath[0]--; // for the first header not in the model
        return {sectionIndex, inHeader: false, inFooter: false, relativePath};
    },
    /**
     * Converts a Plate operation to a KKDoc operation.
     * @param editor - the Plate editor
     * @param pOperation - the Plate operation
     */
    toKOperation = (editor: TPlateEditor, pOperation: TOperation): KOperation => {
        if (pOperation.type === 'set_selection') {
            const pAnchor = pOperation.newProperties?.anchor!,
                pFocus = pOperation.newProperties?.focus!;
            return {
                type: 'set_selection', properties: undefined, newProperties: {
                    anchor: {path: toKPath(editor, pAnchor.path), offset: pAnchor.offset},
                    focus: {path: toKPath(editor, pFocus.path), offset: pFocus.offset},
                }
            };
        }
        return {
            ...pOperation,
            path: toKPath(editor, pOperation.path),
            newPath: pOperation.newPath ? toKPath(editor, pOperation.newPath as Path) : undefined,
        };
    },
    getHeaderOrFooter = (editor: TPlateEditor, path: Path): TElement | undefined => {
        if (editor.hasPath(path)) {
            const node = editor.node(path)[0] as TElement;
            return isHeaderOrFooter(node) ? node
                : getBlockAbove(editor, {at: path, match: isHeaderOrFooter})?.[0] as TElement;
        }
        return undefined;
    };

@zk.WrapClass('kkdoc.PlateObject')
export class PlateObject extends zk.Object {
    /** @internal */
    declare _zclass: string;
    /** @internal */
    declare _emitter: TypedEmitter;
    /** @internal */
    declare _root: Root;

    constructor(uuid: string, zclass: string, initialValue: TElement, onChange: (operations: KOperation[]) => void,
                onCopy: (nodes: TElement[]) => void, onCut: (nodes: TElement[]) => void, onPaste: (nodes: TElement[]) => void,
                onMouseUp: () => void) {
        super();
        this._zclass = zclass;
        this._emitter = new TypedEmitter();
        this._emitter.on('onChange', onChange);
        this._emitter.on('onCopy', onCopy);
        this._emitter.on('onCut', onCut);
        this._emitter.on('onPaste', onPaste);
        this._emitter.on('onMouseUp', onMouseUp);
        this._root = createRoot(document.getElementById(uuid)!);
        this._root.render(<EditorDefault zclass={this._zclass} emitter={this._emitter}
                                         initialValue={this._beforeSetValue(initialValue)}/>);
    }

    setValue(value: TElement): this {
        this._emitter.emit('setValue', this._beforeSetValue(value));
        return this;
    }

    /** @internal */
    _beforeSetValue(value: TElement): TElement[] {
        const dummyEditor = createPlateEditor();
        dummyEditor.children = [value];
        updateHeadersAndFooters(dummyEditor);
        return dummyEditor.children;
    }

    applyOperation(operations: KOperation[]): void {
        this._emitter.emit('apply', operations);
    }

    focus(): void {
        this._emitter.emit('focus');
    }
}
