import {
    createPluginFactory,
    getBlockAbove,
    getEndPoint,
    getPointAfter,
    getPointBefore,
    getStartPoint,
    isCollapsed,
    moveSelection,
    PlateEditor,
    removeNodes,
    TElement,
    TNode,
    Value
} from '@udecode/plate-common';
import {Ancestor, Element, Location, NodeEntry, NodeMatch, Point, Range} from 'slate';
import {ELEMENT_HR} from '@udecode/plate-horizontal-rule';

export const ELEMENT_ROOT = 'root',
    ELEMENT_SECTION = 'section',
    ELEMENT_PAGE_BREAK = 'page-break',
    ELEMENT_HEADER = 'header',
    ELEMENT_FOOTER = 'footer',
    ELEMENT_PAGE_NUMBER = 'page-number',
    matchPageBreak = (node: Element): boolean => Element.isElementType(node, ELEMENT_PAGE_BREAK),
    matchHeaderOrFooter = (node: Element): boolean => Element.isElementType(node, ELEMENT_HEADER)
        || Element.isElementType(node, ELEMENT_FOOTER),
    matchPageNumber = (node: Element): boolean => Element.isElementType(node, ELEMENT_PAGE_NUMBER);

/**
 * Return true if:
 * - at start/end of a margin.
 * - next to a margin. Move selection to the page break.<br>
 * Adapted from withDeleteTable.ts in `@udecode/plate-table`.
 */
const preventDeleteMargin = <V extends Value = Value>(
    editor: PlateEditor<V>, {unit, reverse}: { unit?: 'character' | 'word' | 'line' | 'block'; reverse?: boolean }
): boolean => {
    const {selection} = editor,

        getPoint = reverse ? getEndPoint : getStartPoint,
        getNextPoint = reverse ? getPointAfter : getPointBefore;

    if (isCollapsed(selection)) {
        const marginEntry = getBlockAbove(editor, {match: matchHeaderOrFooter});
        if (marginEntry) {
            // Prevent deleting margin at the start or end of a margin
            const [, marginPath] = marginEntry,
                start = getPoint(editor, marginPath);
            if (selection && Point.equals(selection.anchor, start)) return true;
        } else {
            // Prevent deleting margin when selection is before or after a page break
            const nextPoint = getNextPoint(editor, selection!, {unit}),
                nextMarginEntry = getBlockAbove(editor, {
                    match: matchHeaderOrFooter, at: nextPoint
                });
            if (nextMarginEntry) {
                const nextBreakEntry = getBlockAbove(editor, {
                    match: {type: ELEMENT_PAGE_BREAK}, at: nextPoint
                });
                if (nextBreakEntry && !(nextBreakEntry[0].attributes as { auto? })?.auto) {
                    removeNodes(editor, {at: nextBreakEntry[1]});
                    return true;
                }
                moveSelection(editor, {reverse: !reverse});
                return true;
            }
        }
    }

    return false;
};

/**
 * Update headers and footers based on the section nodes' header and footer attributes.
 * @param editor - the editor to update headers and footers for.
 */
export const updateHeadersAndFooters = <V extends Value = Value>(
    editor: PlateEditor<V>
): void => {
    let pageNumber = 0;
    (editor.children[0].children as TElement[]).forEach((section, sectionIdx) => {
        if (section.children[0].type !== ELEMENT_PAGE_BREAK) // the first header
            section.children.unshift({type: ELEMENT_PAGE_BREAK, attributes: {auto: true}, children: []});
        if (section.children[section.children.length - 1].type !== ELEMENT_PAGE_BREAK) // the last footer
            section.children.push({type: ELEMENT_PAGE_BREAK, attributes: {auto: true}, children: []});
        const {defaultHeader, evenHeader, firstHeader, defaultFooter, evenFooter, firstFooter} = section,
            pageBreaks = Array.from(editor.nodes({
                at: [0, sectionIdx], match: matchPageBreak as NodeMatch<TElement>
            }));
        for (let i = 0, j = 0; i < pageBreaks.length; pageNumber++, i++) {
            const [pageBreak, pageBreakPath] = pageBreaks[i];
            pageBreak.children = [];
            if (i > 0) {
                let footer = defaultFooter;
                if (j % 2 && evenFooter) footer = evenFooter;
                if (j++ === 0 && firstFooter) footer = firstFooter;
                pageBreak.children.push(JSON.parse(JSON.stringify(footer)) as TElement);
                // update page numbers in the footer
                Array.from(editor.nodes({
                    at: [...pageBreakPath, 0],
                    match: matchPageNumber as NodeMatch<TNode>
                })).forEach(([node]) => node.children = [{text: `${pageNumber}`}]);
            }
            if (i < pageBreaks.length - 1) {
                if (i > 0) pageBreak.children.push({type: ELEMENT_HR, children: [{text: ''}]});
                let header = defaultHeader;
                if (j % 2 && evenHeader) header = evenHeader;
                if (j === 0 && firstHeader) header = firstHeader;
                pageBreak.children.push(JSON.parse(JSON.stringify(header)) as TElement);
                // update page numbers in the header
                Array.from(editor.nodes({
                    at: [...pageBreakPath, pageBreak.children.length - 1],
                    match: matchPageNumber as NodeMatch<TNode>
                })).forEach(([node]) => node.children = [{text: `${pageNumber + 1}`}]);
            }
        }
        pageNumber--;
    });
};

/**
 * - Prevent auto-inserted page breaks from being deleted, adapted from withDeleteTable.ts in `@udecode/plate-table`.
 * - Move selection to the content before/after the page break when navigating with arrow keys. TODO: up/down keys.
 * - Prevent moving out of page margin using arrow keys, adapted from move.ts in `@udecode/plate`. TODO: up/down keys.
 */
const withPagination = <
    V extends Value = Value,
    E extends PlateEditor<V> = PlateEditor<V>,
>(editor: E): E => {
    // Prevent auto-inserted page breaks from being deleted
    const {deleteBackward, deleteForward} = editor;
    editor.deleteBackward = (unit) => {
        if (preventDeleteMargin(editor, {unit})) return;
        deleteBackward(unit);
    };
    editor.deleteForward = (unit) => {
        if (preventDeleteMargin(editor, {unit, reverse: true})) return;
        deleteForward(unit);
    };

    // Move selection to the content before/after the page break when navigating with arrow keys
    const {isElementReadOnly, isSelectable} = editor;
    editor.isElementReadOnly = (element) =>
        Element.isElementType(element, ELEMENT_PAGE_NUMBER) || isElementReadOnly(element);
    const getPageMargin = (at: Location): NodeEntry<Ancestor> | undefined => editor.above({
            at, match: matchHeaderOrFooter as NodeMatch<TNode>
        }),
        isSelectableOutsideMargin = (element: Element): boolean => !matchHeaderOrFooter(element) && !matchPageBreak(element)
            && isSelectable(element),
        {apply} = editor;
    editor.apply = (operation) => {
        const {type} = operation;
        if (type === 'set_selection') {
            const anchor = operation.properties?.anchor,
                newAnchor = operation.newProperties?.anchor;
            if ((!anchor || getPageMargin(anchor)) && newAnchor && !getPageMargin(newAnchor)) {
                editor.isSelectable = isSelectableOutsideMargin;
            }
        }
        apply(operation);
    };

    // Prevent moving out of page margin when navigating with arrow keys
    const {setSelection, move} = editor;
    editor.setSelection = (props) => {
        if (props.anchor && getPageMargin(props.anchor)) {
            editor.isSelectable = isSelectable;
        }
        setSelection(props);
    };
    editor.move = (options = {}) => {
        const {selection} = editor,
            {distance = 1, unit = 'character', reverse = false} = options;
        let {edge = undefined} = options;
        if (!selection) return;
        if (edge === 'start') {
            edge = Range.isBackward(selection) ? 'focus' : 'anchor';
        }
        if (edge === 'end') {
            edge = Range.isBackward(selection) ? 'anchor' : 'focus';
        }

        const {anchor, focus} = selection,
            opts = {distance, unit, ignoreNonSelectable: true},
            props: Partial<Range> = {};

        if (edge == null || edge === 'anchor') {
            const point = reverse ? editor.before(anchor, opts) : editor.after(anchor, opts);
            if (point) {
                props.anchor = point;
            }
        }
        if (edge == null || edge === 'focus') {
            const point = reverse ? editor.before(focus, opts) : editor.after(focus, opts);
            if (point) {
                props.focus = point;
            }
        }

        // prevent moving out of the margin using arrow keys
        if (props.anchor && getPageMargin(selection.anchor) && !getPageMargin(props.anchor)) return;

        move(options);
    };

    return editor;
};

/**
 * Enables support for pagination.
 */
export const createPaginationPlugin = createPluginFactory({
    key: ELEMENT_SECTION,
    isElement: true,
    withOverrides: withPagination,
    plugins: [
        {
            key: ELEMENT_ROOT,
            isElement: true,
        },
        {
            key: ELEMENT_PAGE_BREAK,
            isElement: true,
        },
        {
            key: ELEMENT_HEADER,
            isElement: true,
        },
        {
            key: ELEMENT_FOOTER,
            isElement: true,
        },
        {
            key: ELEMENT_PAGE_NUMBER,
            isElement: true,
            isInline: true,
        }
    ]
});
