import * as React from 'react';

import {cn, withProps, withRef} from '@udecode/cn';
import {PlateElement} from '@udecode/plate-common';
import {
    useTableCellElement,
    useTableCellElementResizable,
    useTableCellElementResizableState,
    useTableCellElementState,
} from '@udecode/plate-table';

import {ResizeHandle} from './resizable';

export const TableCellElement = withRef<
    typeof PlateElement,
    {
        hideBorder?: boolean;
        isHeader?: boolean;
    }
>(({children, className, hideBorder, isHeader, style, ...props}, ref) => {
    const {element} = props,

        {
            borders,
            colIndex,
            colSpan,
            hovered,
            hoveredLeft,
            isSelectingCell,
            readOnly,
            rowIndex,
            rowSize,
            selected,
        } = useTableCellElementState(),
        {props: cellProps} = useTableCellElement({element: props.element}),
        resizableState = useTableCellElementResizableState({
            colIndex,
            colSpan,
            rowIndex,
        }),

        {bottomProps, hiddenLeft, leftProps, rightProps} =
            useTableCellElementResizable(resizableState),

        Cell = isHeader ? 'th' : 'td',

        ghost = React.useMemo(() => <span className='ghost' contentEditable={false}/>, []);

    return (
        <PlateElement
            asChild
            className={cn(
                !hideBorder &&
                cn(
                    isHeader && 'text-left [&_>_*]:m-0',
                    selected && 'before:z-10 before:bg-muted',
                ),
                className
            )}
            // @ts-expect-error ignore
            ref={ref}
            {...cellProps}
            {...props}
            style={
                {
                    ...style,
                    /* relative h-full overflow-visible border-none bg-background p-0 */
                    overflow: 'visible',
                    position: 'relative',
                    padding: 0,
                    height: '100%',
                    borderStyle: hideBorder ? 'none' : undefined,
                    backgroundColor: element.background,
                    borderTopColor: hideBorder ? undefined : borders?.top?.color,
                    borderTopStyle: hideBorder ? undefined : borders?.top?.style,
                    borderTopWidth: hideBorder ? undefined : borders?.top?.size,
                    borderLeftColor: hideBorder ? undefined : borders?.left?.color,
                    borderLeftStyle: hideBorder ? undefined : borders?.left?.style,
                    borderLeftWidth: hideBorder ? undefined : borders?.left?.size,
                    borderRightColor: hideBorder ? undefined : borders?.right?.color,
                    borderRightStyle: hideBorder ? undefined : borders?.right?.style,
                    borderRightWidth: hideBorder ? undefined : borders?.right?.size,
                    borderBottomColor: hideBorder ? undefined : borders?.bottom?.color,
                    borderBottomStyle: hideBorder ? undefined : borders?.bottom?.style,
                    borderBottomWidth: hideBorder ? undefined : borders?.bottom?.size,
                } as React.CSSProperties
            }
        >
            <Cell>
                {ghost}
                <div
                    style={{
                        /* relative z-20 box-border h-full px-3 py-2 */
                        minHeight: rowSize,
                        boxSizing: 'border-box',
                        position: 'relative',
                        zIndex: 20,
                        padding: '0.5rem 0.75rem',
                        height: '100%',
                    }}
                >
                    {children}
                </div>

                {!isSelectingCell && (
                    <div
                        /* group absolute top-0 size-full select-none */
                        style={{position: 'absolute', top: 0, userSelect: 'none', width: '100%', height: '100%'}}
                        contentEditable={false}
                        suppressContentEditableWarning={true}
                    >
                        {!readOnly && (
                            <>
                                <ResizeHandle
                                    {...rightProps}
                                    /* -top-3 right-[-5px] w-[10px] */
                                    style={{top: '-0.75rem', right: '-5px', width: '10px'}}
                                />
                                <ResizeHandle
                                    {...bottomProps}
                                    /* bottom-[-5px] h-[10px] */
                                    style={{bottom: '-5px', height: '10px'}}
                                />
                                {!hiddenLeft && (
                                    <ResizeHandle
                                        {...leftProps}
                                        /* -top-3 left-[-5px] w-[10px] */
                                        style={{top: '-0.75rem', left: '-5px', width: '10px'}}
                                    />
                                )}

                                {hovered && (
                                    <div
                                        style={{
                                            /* absolute -top-3 z-30 h-[calc(100%_+_12px)] w-1 bg-ring right-[-1.5px] */
                                            position: 'absolute', top: '-0.75rem', zIndex: 30,
                                            height: 'calc(100% + 12px)', width: '0.25rem', right: '-1.5px'
                                        }}

                                    />
                                )}
                                {hoveredLeft && (
                                    <div
                                        style={{
                                            /* absolute -top-3 z-30 h-[calc(100%_+_12px)] w-1 bg-ring left-[-1.5px] */
                                            position: 'absolute', top: '-0.75rem', zIndex: 30,
                                            height: 'calc(100% + 12px)', width: '0.25rem', left: '-1.5px'
                                        }}
                                    />
                                )}
                            </>
                        )}
                    </div>
                )}
            </Cell>
        </PlateElement>
    );
});

TableCellElement.displayName = 'TableCellElement';

export const TableCellHeaderElement = withProps(TableCellElement, {
    isHeader: true,
});
