'use client';

import {createBasicElementsPlugin} from '@udecode/plate-basic-elements';
import {createBasicMarksPlugin} from '@udecode/plate-basic-marks';
import {
    AnyObject,
    createPlateEditor,
    createPlugins,
    getBlockAbove,
    Plate,
    PlateEditor,
    PlatePlugin,
    TOperation,
    Value,
    WithPlatePlugin,
    withTReact
} from '@udecode/plate-common';

// Elements generated with npx @udecode/plate-ui@latest
import {createPlateUI} from '@plate/create-plate-ui';
import {Editor} from '@plate/components/editor';

import * as React from 'react';
import {createRoot, Root} from 'react-dom/client';
import {TypedEmitter} from 'tiny-typed-emitter';
import {createAlignPlugin} from '@udecode/plate-alignment';
import {
    createTablePlugin,
    withDeleteTable,
    withGetFragmentTable,
    withInsertFragmentTable,
    withInsertTextTable,
    withSelectionTable,
    withSetFragmentDataTable
} from '@udecode/plate-table';
import {ELEMENT_PARAGRAPH} from '@udecode/plate-paragraph';
import {ELEMENT_H1, ELEMENT_H2, ELEMENT_H3, ELEMENT_H4, ELEMENT_H5, ELEMENT_H6} from '@udecode/plate-heading';
import {
    createFontBackgroundColorPlugin,
    createFontColorPlugin,
    createFontFamilyPlugin,
    createFontSizePlugin,
    createFontWeightPlugin
} from '@udecode/plate-font';
import {createHighlightPlugin} from '@udecode/plate-highlight';
import {createKbdPlugin} from '@udecode/plate-kbd';
import {createIndentPlugin} from '@udecode/plate-indent';
import {createLineHeightPlugin} from '@udecode/plate-line-height';
import {createDeserializeDocxPlugin} from '@udecode/plate-serializer-docx';
import {createDeserializeCsvPlugin} from '@udecode/plate-serializer-csv';
import {createDeserializeMdPlugin} from '@udecode/plate-serializer-md';
import {
    createPaginationPlugin,
    ELEMENT_FOOTER,
    ELEMENT_HEADER,
    ELEMENT_SECTION,
    matchHeaderOrFooter,
    matchPageBreak,
    updateHeadersAndFooters,
} from '@plate/plugins/plate-pagination';
import {createHorizontalRulePlugin} from '@udecode/plate-horizontal-rule';
import {createMentionPlugin} from '@udecode/plate-mention';
import {ReactEditor} from 'slate-react';
import {Element, Path} from 'slate';
import {createComponentPlugin} from '@plate/plugins/plate-component';

function EditorDefault({zclass, emitter, initialValue}: {
    zclass: string;
    emitter: TypedEmitter;
    initialValue: TElement[];
}): React.ReactElement {
    const textBlockTypes = [ELEMENT_PARAGRAPH, ELEMENT_H1, ELEMENT_H2, ELEMENT_H3, ELEMENT_H4, ELEMENT_H5, ELEMENT_H6],
        plugins = createPlugins(
            [
                createBasicElementsPlugin(),
                createHorizontalRulePlugin(),
                createTablePlugin({options: {enableMerging: true},
                    withOverrides: (editor: PlateEditor, plugin: WithPlatePlugin) => {
                        // remove withNormalizeTable to support nested tables
                        editor = withDeleteTable(editor);
                        editor = withGetFragmentTable(editor, plugin);
                        editor = withInsertFragmentTable(editor, plugin);
                        editor = withInsertTextTable(editor, plugin);
                        editor = withSelectionTable(editor);
                        editor = withSetFragmentDataTable(editor);
                        return editor;
                    }
                }),

                createAlignPlugin({inject: {props: {validTypes: textBlockTypes}}}),
                createIndentPlugin({inject: {props: {validTypes: textBlockTypes}}}),
                createLineHeightPlugin({inject: {props: {defaultNodeValue: undefined, validTypes: textBlockTypes}}}),

                createBasicMarksPlugin(),
                createFontBackgroundColorPlugin(),
                createFontColorPlugin(),
                createFontFamilyPlugin(),
                createFontSizePlugin(),
                createFontWeightPlugin(),
                createHighlightPlugin(),
                createKbdPlugin(),

                createDeserializeDocxPlugin(),
                createDeserializeCsvPlugin(),
                createDeserializeMdPlugin(),

                createPaginationPlugin(),
                createMentionPlugin(),
                createComponentPlugin(),
            ] as PlatePlugin<AnyObject, Value, PlateEditor<Value>>[],
            {components: createPlateUI(zclass)}
        ),
        [key, setKey] = React.useState(0),
        [value, setValue] = React.useState<TElement[]>(initialValue),
        editor = React.useMemo(() => {
            const editor = withTReact(createPlateEditor({plugins, disableCorePlugins: {history: true}})),
                {apply, onChange} = editor,
                renderUpdatedHeadersAndFooters = (): void => {
                    updateHeadersAndFooters(editor);
                    setKey(Math.random()); // to force re-render
                    setValue([...editor.children]);
                };
            editor.apply = (operation: TOperation) => {
                apply(operation);
                const {type, path} = operation;
                if (type === 'set_selection') return;
                const margin = getHeaderOrFooter(editor, type === 'set_node' ? path : path.slice(0, -1));
                if (margin) { // update headers and footers if operates on a header or footer
                    const capitalizedType = margin.type.charAt(0).toUpperCase() + margin.type.slice(1);
                    editor.node(path.slice(0, 2))[0][`${margin.pageType as string}${capitalizedType}`] = margin;
                    renderUpdatedHeadersAndFooters();
                }
                if ((type === 'insert_node' || type === 'remove_node') && matchPageBreak(operation.node as TElement))
                    renderUpdatedHeadersAndFooters(); // if inserts or removes a page break
            };
            editor.onChange = (options) => {
                const {operations} = editor;
                if (operations.length && !operations[0]['fromServer']) {
                    const kOperations: KOperation[] = [];
                    operations.forEach(operation => {
                        if (operation.type === 'set_selection') return;
                        kOperations.push(toKOperation(editor, operation));
                    });
                    if (editor.selection) {
                        const start = editor.start(editor.selection),
                            end = editor.end(editor.selection);
                        kOperations.push({
                            type: 'set_selection', properties: undefined, newProperties: {
                                anchor: {path: toKPath(editor, start.path), offset: start.offset},
                                focus: {path: toKPath(editor, end.path), offset: end.offset},
                            }
                        });
                    }
                    emitter.emit('onChange', kOperations);
                }
                onChange(options);
            };
            emitter.on('setValue', (value: TElement[]) => {
                setKey(Math.random()); // to force re-render
                setValue([...value]);
            });
            emitter.on('apply', (operations: KOperation[]) => {
                const normalizing = editor.isNormalizing();
                editor.setNormalizing(false);
                try {
                    let shouldUpdateHeadersAndFooters = false;
                    operations.forEach((kOperation: KOperation) => {
                        const pOperation = toPOperation(editor, kOperation),
                            {type, path} = pOperation;
                        if (type === 'merge_node' && path.length === 2) { // merge two sections
                            (editor.node(Path.previous(path))[0] as TElement).children.pop(); // remove the last footer
                            (editor.node(path)[0] as TElement).children.shift(); // remove the first header
                        }
                        pOperation['fromServer'] = true;
                        editor.apply(pOperation);
                        // if updates a section or inserts a section break
                        if ((type === 'set_node' || type === 'split_node')
                            && Element.isElementType(editor.node(path)[0], ELEMENT_SECTION))
                            renderUpdatedHeadersAndFooters();
                        shouldUpdateHeadersAndFooters ||= ((type === 'insert_node' || type === 'remove_node')
                            && (Element.isElementType(kOperation.node, 'root')
                                || matchPageBreak(kOperation.node as TElement)));
                    });
                    if (shouldUpdateHeadersAndFooters) renderUpdatedHeadersAndFooters();
                } finally {
                    editor.setNormalizing(normalizing);
                }
            });
            emitter.on('focus', () => ReactEditor.focus(editor as ReactEditor));
            return editor;
        }, []),
        onMouseUpHandler = React.useCallback(() => {
            // wait for the onChange (cause by onMouseUp) finishes,
            // to make sure the selection state is up-to-date,
            // then fire the onMouseUp
            setTimeout(() => emitter.emit('onMouseUp'), 10);
        }, []);
    return (
        <Plate key={key} editor={editor} plugins={plugins} initialValue={value}>
            <Editor onMouseUp={onMouseUpHandler}/>
        </Plate>
    );
}

/**
 * Converts a path in the KKDoc model to a path in the Plate editor.
 * @param editor - the Plate editor
 * @param kPath - the path in the KKDoc model
 */
const toPPath = (editor: PlateEditor, kPath: KPath): Path => {
        const {sectionIndex, inHeader, pageType, relativePath} = kPath;
        if (pageType) {
            const [_, headerPath] = Array.from(editor.nodes({
                at: [sectionIndex],
                match: node => (node as TElement).type === (inHeader ? ELEMENT_HEADER : ELEMENT_FOOTER)
                    && (node as TElement).pageType === pageType
            }))[0];
            return [...headerPath, ...relativePath];
        }
        if (relativePath.length) relativePath[0]++; // for the first header not in the model
        return sectionIndex === -1 ? [0] : [0, sectionIndex, ...relativePath];
    },
    /**
     * Converts a KKDoc operation to a Plate operation.
     * @param editor - the Plate editor
     * @param kOperation - the KKDoc operation
     */
    toPOperation = (editor: PlateEditor, kOperation: KOperation): TOperation => {
        const {type} = kOperation, {path, newPath, ...pOperation} = kOperation;
        if (type === 'set_selection') {
            const {newProperties} = kOperation;
            if (newProperties) {
                const {anchor, focus} = newProperties as KRange;
                pOperation.newProperties = {
                    anchor: {path: toPPath(editor, anchor.path), offset: anchor.offset},
                    focus: {path: toPPath(editor, focus.path), offset: focus.offset},
                };
            }
        }
        if (path) pOperation.path = toPPath(editor, path as KPath);
        if (newPath) pOperation.newPath = toPPath(editor, newPath as KPath);
        if (type === 'split_node' && (pOperation.path as Path).length === 2)
            (pOperation.position as number)++; // for the first header not in the model
        pOperation['fromServer'] = true;
        return pOperation as TOperation;
    },
    /**
     * Converts a path in the Plate editor to a path in the KKDoc model.
     * @param editor - the Plate editor
     * @param pPath - the path in the Plate editor
     */
    toKPath = (editor: PlateEditor, pPath: Path): KPath => {
        const [_, sectionIndex] = pPath,
            parentPath = pPath.slice(0, -1),
            margin = getHeaderOrFooter(editor, parentPath),
            relativePath = pPath.slice(2); // skip [root, section]
        if (margin) {
            return {
                sectionIndex,
                inHeader: margin.type === ELEMENT_HEADER,
                inFooter: margin.type === ELEMENT_FOOTER,
                pageType: margin.pageType as string | undefined,
                relativePath: relativePath.slice(2) // skip [page-break, header/footer]
            };
        }
        relativePath[0]--; // for the first header not in the model
        return {sectionIndex, inHeader: false, inFooter: false, relativePath};
    },
    /**
     * Converts a Plate operation to a KKDoc operation.
     * @param editor - the Plate editor
     * @param pOperation - the Plate operation
     */
    toKOperation = (editor: PlateEditor, pOperation: TOperation): KOperation => {
        if (pOperation.type === 'set_selection') {
            const pAnchor = pOperation.newProperties?.anchor!,
                pFocus = pOperation.newProperties?.focus!;
            return {
                type: 'set_selection', properties: undefined, newProperties: {
                    anchor: {path: toKPath(editor, pAnchor.path), offset: pAnchor.offset},
                    focus: {path: toKPath(editor, pFocus.path), offset: pFocus.offset},
                }
            };
        }
        return {
            ...pOperation,
            path: toKPath(editor, pOperation.path),
            newPath: pOperation.newPath ? toKPath(editor, pOperation.newPath as Path) : undefined,
        };
    },
    getHeaderOrFooter = (editor: PlateEditor, path: Path): TElement | undefined => {
        if (editor.hasPath(path)) {
            const node = editor.node(path)[0] as TElement;
            return matchHeaderOrFooter(node) ? node
                : getBlockAbove(editor, {at: path, match: matchHeaderOrFooter})?.[0] as TElement;
        }
        return undefined;
    };

@zk.WrapClass('kkdoc.PlateObject')
export class PlateObject extends zk.Object {
    /** @internal */
    declare _zclass: string;
    /** @internal */
    declare _emitter: TypedEmitter;
    /** @internal */
    declare _root: Root;

    constructor(uuid: string, zclass: string, initialValue: TElement, onChange: (operations: KOperation[]) => void, onMouseUp: () => void) {
        super();
        this._zclass = zclass;
        this._emitter = new TypedEmitter();
        this._emitter.on('onChange', onChange);
        this._emitter.on('onMouseUp', onMouseUp);
        this._root = createRoot(document.getElementById(uuid)!);
        this._root.render(<EditorDefault zclass={this._zclass} emitter={this._emitter}
                                         initialValue={this._beforeSetValue(initialValue)}/>);
    }

    setValue(value: TElement): this {
        this._emitter.emit('setValue', this._beforeSetValue(value));
        return this;
    }

    /** @internal */
    _beforeSetValue(value: TElement): TElement[] {
        const dummyEditor = createPlateEditor();
        dummyEditor.children = [value];
        updateHeadersAndFooters(dummyEditor);
        return dummyEditor.children;
    }

    applyOperation(operations: KOperation[]): void {
        this._emitter.emit('apply', operations);
    }

    focus(): void {
        this._emitter.emit('focus');
    }
}
