import {cn, withProps, withRef} from '@udecode/cn';
import {useBlockSelected} from '@udecode/plate-selection/react';
import {TTableCellElement} from '@udecode/plate-table';
import {
    TablePlugin,
    TableRowPlugin,
    useTableCellElement,
    useTableCellElementResizable,
} from '@udecode/plate-table/react';
import {PlateElement, useEditorPlugin, useElementSelector, useReadOnly} from '@udecode/plate/react';
import * as React from 'react';
import {blockSelectionVariants} from './block-selection';
import {ResizeHandle} from './resizable';

export const TableCellElement = withRef<
    typeof PlateElement,
    {
        isHeader?: boolean;
    }
>(({children, className, isHeader, style, ...props}, ref) => {
    const {api} = useEditorPlugin(TablePlugin),
        readOnly = useReadOnly(),
        element = props.element as TTableCellElement,

        rowId = useElementSelector(([node]) => node.id as string, [], {
            key: TableRowPlugin.key,
        }),
        isSelectingRow = useBlockSelected(rowId) as boolean,
        // isSelectionAreaVisible = usePluginOption(
        //     BlockSelectionPlugin,
        //     'isSelectionAreaVisible'
        // ),

        {borders, colIndex, colSpan, minHeight, rowIndex, selected, width} =
            useTableCellElement(),

        {bottomProps, hiddenLeft, leftProps, rightProps} =
            useTableCellElementResizable({
                colIndex,
                colSpan,
                rowIndex,
            }),

        ghost = React.useMemo(() => <span className='ghost' contentEditable={false}/>, []);

    return (
        <PlateElement
            ref={ref}
            as={isHeader ? 'th' : 'td'}
            className={cn(
                className,
                'h-full overflow-visible border-none bg-background p-0',
                element.background ? 'bg-(--cellBackground)' : 'bg-background',
                isHeader && 'text-left *:m-0',
                'before:size-full',
                selected && 'before:z-10 before:bg-brand/5',
                "before:absolute before:box-border before:content-[''] before:select-none",
                borders.bottom?.size && 'before:border-b before:border-b-border',
                borders.right?.size && 'before:border-r before:border-r-border',
                borders.left?.size && 'before:border-l before:border-l-border',
                borders.top?.size && 'before:border-t before:border-t-border'
            )}
            style={{
                ...style,
                '--cellBackground': element.background,
                maxWidth: width || 240,
                minWidth: width || 120,
            } as React.CSSProperties}
            {...{
                colSpan: api.table.getColSpan(element),
                rowSpan: api.table.getRowSpan(element),
            }}
            {...props}
        >
            <div
                className="relative z-20 box-border h-full px-3 py-2"
                style={{
                    minHeight
                }}
            >
                {ghost}
                {children}
            </div>

            {/*{!isSelectionAreaVisible && (*/}
                <div
                    className="group absolute top-0 size-full select-none"
                    contentEditable={false}
                    suppressContentEditableWarning={true}
                >
                    {!readOnly && (
                        <>
                            <ResizeHandle
                                {...rightProps}
                                className="-top-2 -right-1 h-[calc(100%_+_8px)] w-2"
                                data-col={colIndex}
                            />
                            <ResizeHandle
                                {...bottomProps}
                                className="-bottom-1 h-2"/>
                            {!hiddenLeft && (
                                <ResizeHandle
                                    {...leftProps}
                                    className="top-0 -left-1 w-2"
                                    data-resizer-left={colIndex === 0 ? 'true' : undefined}
                                />
                            )}

                            <div
                                className={cn(
                                    'absolute top-0 z-30 hidden h-full w-1 bg-ring',
                                    'right-[-1.5px]',
                                    // columnResizeVariants({ colIndex: colIndex as any })
                                )}
                            />
                        </>
                    )}
                </div>
            {/*)}*/}

            {isSelectingRow && (
                <div className={blockSelectionVariants()} contentEditable={false}/>
            )}
        </PlateElement>
    );
});

export const TableCellHeaderElement = withProps(TableCellElement, {
    isHeader: true,
});
