import {Editor, EditorContainer} from '@plate/components/editor';
import {PlateStatic} from '@plate/components/PlateStatic';
import {createPlatePlugins} from '@plate/create-plate-plugins';
import {createPlateUI} from '@plate/create-plate-ui';
import {createStaticComponents} from '@plate/create-static-components';
import {KKDocToolbar} from '@plate/toolbar/KKDocToolbar';
import {createSlateEditor, Descendant, NodeApi, PathApi, serializeHtml} from '@udecode/plate';
import {computeDiff} from '@udecode/plate-diff';
import {createPlateEditor, Plate, PlateEditor} from '@udecode/plate/react';
import * as React from 'react';
import {DndProvider} from 'react-dnd';
import {HTML5Backend} from 'react-dnd-html5-backend';
import {createRoot, Root} from 'react-dom/client';
import {TypedEmitter} from 'tiny-typed-emitter';
import {type Document} from "@kkdoc/Document";

/**
 * Normalizes the value without triggering Plate's onChange.
 * @param value - the value to normalize
 */
const normalize = (value: string): TElement[] =>
    createSlateEditor({plugins: createPlatePlugins(), value, shouldNormalizeEditor: true}).children;

interface GetInitialValue {
    (type: 'value'): string;
    (type: 'readonly'): boolean;
    (type: 'documentConfig'): kkdoc.DocumentConfig;
}
function EditorDefault({uuid, zclass, emitter, getInitialValue, onRender}: {
    uuid: string;
    zclass: string;
    emitter: TypedEmitter;
    getInitialValue: GetInitialValue;
    onRender: () => void;
}): React.ReactElement {
    const [documentConfig, setDocumentConfig] = React.useState<kkdoc.DocumentConfig>(getInitialValue('documentConfig')),
        [readonly, setReadonly] = React.useState<boolean>(getInitialValue('readonly')),
        editor = React.useMemo(() => {
            const editor = createPlateEditor({
                override: {components: createPlateUI(zclass, new TypedEmitter())},
                plugins: createPlatePlugins(),
                value: normalize(getInitialValue('value')),
            }) as PlateEditor;
            editor.id = uuid;
            editor.zclass = zclass;
            const staticComponents = createStaticComponents(zclass);
            emitter.on('getValue', (cb: (value: string) => void) => {
                serializeHtml(editor, {
                    components: staticComponents,
                    editorComponent: PlateStatic,
                    preserveClassNames: [],
                    stripClassNames: true,
                    stripDataAttributes: true,
                }).then(cb).catch(error => {
                    throw error;
                });
            });
            emitter.on('setValue', (value: string) => {
                // editor.tf.setValue(value) without firing back to the server
                editor.tf.withoutNormalizing(() => {
                    const children = normalize(value);
                    for (const [node, path] of Array.from(NodeApi.children(editor, [], {reverse: true}))) {
                        editor.tf.apply({fromServer: true, type: 'remove_node', path, node});
                    }
                    let path = [0];
                    for (const node of children) {
                        editor.tf.apply({fromServer: true, type: 'insert_node', path, node});
                        path = PathApi.next(path);
                    }
                });
                setOnChangeValue(editor.children);
            });
            emitter.on('setDocumentConfig', (documentConfig: kkdoc.DocumentConfig) => {
                setDocumentConfig(documentConfig);
            });
            emitter.on('setReadonly', (readonly: boolean) => {
                setReadonly(readonly);
            });

            // call onRender after the editor is created
            onRender();
            return editor;
        }, []),
        [onChangeValue, setOnChangeValue] = React.useState(editor.children);

    React.useEffect(() => {
        const document = zk.$('#' + uuid) as Document,
            $document = document.$n_(),
            $toolbar = document.$n_('toolbar'),
            $editor = document.$n_('editor'),
            resizeObserver = new ResizeObserver(() => {
                if (document.getHeight() || document.getVflex()) {
                    // update $editor's height
                    $editor.style.height = ($document.clientHeight - $toolbar.clientHeight) + 'px';
                } else { // otherwise, remove height-related styles
                    $editor.style.removeProperty('height');
                }
            });
        resizeObserver.observe($document as Element);
        resizeObserver.observe($toolbar as Element);

        return () => {
            resizeObserver.disconnect();
        };
    }, []);

    // eslint-disable-next-line zk/noMixedHtml
    return (
        <DndProvider backend={HTML5Backend}>
            <Plate
                editor={editor}
                onChange={() => {
                    const {operations} = editor;
                    if (operations.every(op => op.type === 'set_selection' || op.fromServer)) return;
                    emitter.emit('onChanging');
                }}
            >
                <KKDocToolbar
                    id={uuid + '-toolbar'}
                    documentConfig={documentConfig}
                />
                <EditorContainer
                    id={uuid + '-editor'}
                    onBlur={(event) => {
                        if (event.relatedTarget?.getAttribute('data-plate-focus-id') === uuid) return;
                        let changed = false;
                        const setChanged = (_: Descendant) => changed = true;
                        computeDiff(onChangeValue, editor.children,
                            {getDeleteProps: setChanged, getInsertProps: setChanged, getUpdateProps: setChanged}
                        );
                        if (changed) {
                            setOnChangeValue(editor.children);
                            emitter.emit('onChange');
                        }
                    }}
                >
                    <Editor variant='demo' readOnly={readonly}/>
                </EditorContainer>
            </Plate>
        </DndProvider>
    );
}

@zk.WrapClass('kkdoc.PlateObject')
export class PlateObject extends zk.Object {
    /** @internal */
    declare _emitter: TypedEmitter;
    /** @internal */
    declare _root: Root;
    /** @internal */
    declare _initialValue: string;
    /** @internal */
    declare _initialDocumentConfig: kkdoc.DocumentConfig;
    /** @internal */
    declare _initialReadonly: boolean;
    /** @internal */
    declare _isRendered: boolean;

    constructor(uuid: string, zclass: string, initialReadonly: boolean, initialValue: string, initialDocumentConfig: kkdoc.DocumentConfig,
                onChange: () => void, onChanging: () => void) {
        super();
        this._initialValue = initialValue;
        this._initialDocumentConfig = initialDocumentConfig;
        this._initialReadonly = initialReadonly;

        const getInitialValue = (type => {
            switch (type) {
                case 'value':
                    return this._initialValue;
                case 'readonly':
                    return this._initialReadonly;
                case 'documentConfig':
                    return this._initialDocumentConfig;
            }
        }) as GetInitialValue;

        this._emitter = new TypedEmitter();
        this._emitter.on('onChange', onChange);
        this._emitter.on('onChanging', onChanging);
        this._root = createRoot(document.getElementById(uuid)!);
        this._root.render(<EditorDefault uuid={uuid} zclass={zclass} emitter={this._emitter}
                     getInitialValue={getInitialValue}
                    onRender={() => this._isRendered = true}/>);
    }

    destroy(): void {
        this._emitter.removeAllListeners();
        this._root.unmount();
    }

    getValue(cb: (value: string) => void): void {
        if (this._isRendered) {
            this._emitter.emit('getValue', cb);
        }
    }

    setValue(value: string): this {
        this._initialValue = value;
        if (this._isRendered) {
            this._emitter.emit('setValue', value);
        }
        return this;
    }

    setReadonly(readonly: boolean): this {
        this._initialReadonly = readonly;
        if (this._isRendered) {
            this._emitter.emit('setReadonly', readonly);
        }
        return this;
    }

    setDocumentConfig(documentConfig: kkdoc.DocumentConfig): this {
        this._initialDocumentConfig = documentConfig;
        if (this._isRendered) {
            this._emitter.emit('setDocumentConfig', documentConfig);
        }
        return this;
    }
}
