/** JSFunction.java.

	Purpose:
		
	Description:
		
	History:
		3:21:15 PM Jan 17, 2014, Created by jumperchen

Copyright (C) 2014 Potix Corporation. All Rights Reserved.
 */
package org.zkoss.chart.util;

import java.util.LinkedList;
import java.util.ListIterator;

import org.zkoss.json.JSONAware;
import org.zkoss.json.JSONValue;

/**
 * A Javascript function wrap to generate as a formal Javascript code. Used for
 * the implementation of Highcharts' dynamical attributes.
 * @author jumperchen
 * 
 */
public class JSFunction implements JSONAware {

	private LinkedList<String> js;
	private static String FUNC = "%1$s()";
	private static String FUNC_ARG = "%1$s(%2$s)";
	private static String ARRAY_ARG = "%1$s[%2$s]";

	private static String WRAP_FUNC = "function () {this.%1$s;}";
	private static String EXEC_FUNC = "function () {%1$s;}";
	private static String RETURN_FUNC = "function () {return this.%1$s;}";
	private static String EMPTY_FUNC = "zk.$void";
	public JSFunction() {
		js = new LinkedList<String>();
	}
	
	/**
	 * Evaluates an attribute of Javascript Object.
	 * @param attribute the name of the attribute
	 */
	public JSFunction evalAttribute(String attribute) {
		js.add(attribute);
		return this;
	}

	/**
	 * Evaluates a Javascript code with parentheses, if any.
	 * @param javascript the formal Javascript code
	 * @see #toExecFunction()
	 */
	public JSFunction evalJavascript(String javascript) {
		js.add(javascript);
		return this;
	}

	/**
	 * Calls a Javascript Function without any arguments.
	 * @param funcName the name of the Javascript function without parentheses.
	 */
	public JSFunction callFunction(String funcName) {
		js.add(String.format(FUNC, funcName));
		return this;
	}

	/**
	 * Calls a Javascript Function with any arguments.
	 * @param funcName the name of the Javascript function without parentheses.
	 * @param arguments the arguments in ordering.
	 */
	public JSFunction callFunction(String funcName, Object... arguments) {
		StringBuilder args = new StringBuilder();
		for (Object arg : arguments) {
			args.append(JSONValue.toJSONString(arg)).append(',');
		}
		if (arguments.length > 0)
			args.deleteCharAt(args.length() - 1);

		js.add(String.format(FUNC_ARG, funcName, args.toString()));
		return this;
	}

	public void empty() {
		js.clear();
	}
	/**
	 * Calls a array with the given index.
	 * @param attribute the name of the array
	 * @param index the index of the array.
	 */
	public JSFunction callArray(String attribute, int index) {
		js.add(String.format(ARRAY_ARG, attribute, index));
		return this;
	}

	private boolean _reverse = false;
	
	/**
	 * Sets the Javascript calls in a reverse ordering.
	 */
	public JSFunction reverse() {
		_reverse = true;
		return this;
	}
	public String toJSONString() {
		StringBuilder sb = new StringBuilder();
		if (_reverse) {
			ListIterator<String> li = js.listIterator(js.size());

			// Iterate in reverse.
			while(li.hasPrevious()) {
				sb.append(li.previous()).append('.');
			}
			
		} else {
			for (String s : js)
				sb.append(s).append('.');
		}
		
		if (sb.length() > 0 && sb.charAt(sb.length() - 1) == '.') {
			sb.deleteCharAt(sb.length() - 1);
		}
		
		return sb.toString();
	}
	
	public boolean isEmpty() {
		return js.isEmpty();
	}
	/**
	 * Returns the string in a wrap Javascript function.
	 * <p> For example,
	 * <pre><code>function () { this.YourCodesHere;}</code></pre>
	 */
	public String toWrapFunction() {
		return isEmpty() ? EMPTY_FUNC : String.format(WRAP_FUNC, toString());
	}
	
	/**
	 * Returns the string in a executable Javascript function.
	 * <p> For example,
	 * <pre><code>function () { YourCodesHere;}</code></pre>
	 */
	public String toExecFunction() {
		return isEmpty() ? EMPTY_FUNC : String.format(EXEC_FUNC, toString());
	}
	
	/**
	 * Returns the string in a return value of Javascript function.
	 * <p> For example,
	 * <pre><code>function () { return this.YourCodesHere;}</code></pre>
	 */
	public String toReturnFunction() {
		return isEmpty() ? EMPTY_FUNC :String.format(RETURN_FUNC, toString());
	}
	
	public String toString() {
		return isEmpty() ? EMPTY_FUNC :toJSONString();
	}

}
