/* TooltipPlotOptions.java

	Purpose:
		
	Description:
		
	History:
		Tue, Jan 14, 2014  5:53:19 PM, Created by RaymondChao

Copyright (C) 2014 Potix Corporation. All Rights Reserved.

 */
package org.zkoss.chart.plotOptions;

import org.zkoss.chart.Chart;
import org.zkoss.chart.DateTimeLabelFormats;
import org.zkoss.chart.Charts;
import org.zkoss.chart.Optionable;
import org.zkoss.chart.PlotAttribute;
import org.zkoss.chart.util.DynamicalAttribute;

/**
 * An option for tooltip plot
 * <p>
 * All the options in this class support {@link DynamicalAttribute}.
 * 
 * @author RaymondChao
 */
public class TooltipPlotOptions extends Optionable {
	private enum Attrs implements PlotAttribute, DynamicalAttribute {
		dateTimeLabelFormats,
		followPointer,
		followTouchMove,
		footerFormat,
		headerFormat,
		hideDelay,
		pointFormat,
		valueDecimals,
		valuePrefix,
		valueSuffix,
		xDateFormat,
		format
	}

	/**
	 * Returns the date format in the tooltip's header
	 * <p>
	 * For series on a datetime axes, the date format in the tooltip's header
	 * will by default be guessed based on the closest data points. This member
	 * gives the default string representations used for each unit. For an
	 * overview of the replacement codes, see
	 * {@link Charts#setDateFormat(String, Number, boolean)}.
	 * </p>
	 * <p>
	 * Defaults:
	 * 
	 * <pre>
	 * {
	 *     millisecond:"%A, %b %e, %H:%M:%S.%L",
	 *     second:"%A, %b %e, %H:%M:%S",
	 *     minute:"%A, %b %e, %H:%M",
	 *     hour:"%A, %b %e, %H:%M",
	 *     day:"%A, %b %e, %Y",
	 *     week:"Week from %A, %b %e, %Y",
	 *     month:"%B %Y",
	 *     year:"%Y"
	 * }
	 * </pre>
	 * 
	 * </p>
	 * 
	 * @return dateTimeLabelFormats the date format in the tooltip's header
	 */
	public DateTimeLabelFormats getDateTimeLabelFormats() {
		DateTimeLabelFormats dateTimeLabelFormats = (DateTimeLabelFormats) getAttr(Attrs.dateTimeLabelFormats);
		if (dateTimeLabelFormats == null) {
			dateTimeLabelFormats = new DateTimeLabelFormats();
			setDateTimeLabelFormats(dateTimeLabelFormats);
		}
		return dateTimeLabelFormats;
	}

	/**
	 * Sets the date format in the tooltip's header
	 * <p>
	 * For series on a datetime axes, the date format in the tooltip's header
	 * will by default be guessed based on the closest data points. This member
	 * gives the default string representations used for each unit. For an
	 * overview of the replacement codes, see
	 * {@link Charts#setDateFormat(String, Number, boolean)}.
	 * </p>
	 * 
	 * @param dateTimeLabelFormats
	 *            the date format in the tooltip's header
	 */
	public void setDateTimeLabelFormats(
			DateTimeLabelFormats dateTimeLabelFormats) {
		setAttr(Attrs.dateTimeLabelFormats, dateTimeLabelFormats);
	}

	/**
	 * Returns whether the tooltip should follow the mouse as it moves across
	 * columns, pie slices and other point types with an extent. By default it
	 * behaves this way for scatter, bubble and pie series by override in the
	 * <code>plotOptions</code> for those series types. </p>
	 * <p>
	 * For touch moves to behave the same way,
	 * {@link #setFollowTouchMove(boolean)} must be <code>true</code> also.
	 * </p>
	 * <p>
	 * Default: false.
	 */
	public boolean isFollowPointer() {
		return getAttr(Attrs.followPointer, false).asBoolean();
	}

	/**
	 * Sets whether the tooltip should follow the mouse as it moves across
	 * columns, pie slices and other point types with an extent. By default it
	 * behaves this way for scatter, bubble and pie series by override in the
	 * <code>plotOptions</code> for those series types. </p>
	 * <p>
	 * For touch moves to behave the same way,
	 * {@link #setFollowTouchMove(boolean)} must be <code>true</code> also.
	 * </p>
	 */
	public void setFollowPointer(boolean followPointer) {
		setAttr(Attrs.followPointer, followPointer);
	}

	/**
	 * Returns whether the tooltip should follow the finger as it moves on a
	 * touch device. The default value of <code>false</code> causes a touch move
	 * to scroll the web page, as is default behaviour on touch devices. Setting
	 * it to <code>true</code> may cause the user to be trapped inside the chart
	 * and unable to scroll away, so it should be used with care. If
	 * {@link Chart#setZoomType(String)} is set, it will override
	 * {@link #setFollowTouchMove(boolean)}.
	 * <p>
	 * Default: false.
	 */
	public boolean isFollowTouchMove() {
		return getAttr(Attrs.followTouchMove, false).asBoolean();
	}

	/**
	 * Sets whether the tooltip should follow the finger as it moves on a touch
	 * device. The default value of <code>false</code> causes a touch move to
	 * scroll the web page, as is default behaviour on touch devices. Setting it
	 * to <code>true</code> may cause the user to be trapped inside the chart
	 * and unable to scroll away, so it should be used with care. If
	 * {@link Chart#setZoomType(String)} is set, it will override
	 * {@link #setFollowTouchMove(boolean)}.
	 */
	public void setFollowTouchMove(boolean followTouchMove) {
		setAttr(Attrs.followTouchMove, followTouchMove);
	}

	/**
	 * Returns a string to append to the tooltip format.
	 * <p>
	 * Default: false.
	 */
	public String getFooterFormat() {
		return getAttr(Attrs.footerFormat, false).asString();
	}

	/**
	 * Sets a string to append to the tooltip format.
	 */
	public void setFooterFormat(String footerFormat) {
		setAttr(Attrs.footerFormat, footerFormat);
	}

	/**
	 * Returns a format text of the tooltip.
	 * <p>
	 * Default: null.
	 */
	public String getFormat() {
		return getAttr(Attrs.format, null).asString();
	}

	/**
	 * Sets a format text of the tooltip.
	 * <p>
	 * Default: null.
	 */
	public void setFormat(String format) {
		setAttr(Attrs.format, format);
	}

	/**
	 * Returns the HTML of the tooltip header line. Variables are enclosed by
	 * curly brackets. Available variables are <code>point.key</code>,
	 * <code>series.name</code>, <code>series.color</code> and other members
	 * from the <code>point</code> and <code>series</code> objects. The
	 * <code>point.key</code> variable contains the category name, x value or
	 * datetime string depending on the type of axis. For datetime axes, the
	 * <code>point.key</code> date format can be set using tooltip.xDateFormat.
	 * </p>
	 * <p>
	 * Default:
	 * <code>&lt;span style="font-size: 10px"&gt;{point.key}&lt;/span&gt;&lt;br/&gt;</code>
	 * </p>
	 */
	public String getHeaderFormat() {
		return getAttr(Attrs.headerFormat,
				"<span style=\"font-size: 10px\">{point.key}</span><br/>")
				.asString();
	}

	/**
	 * Sets the HTML of the tooltip header line. Variables are enclosed by curly
	 * brackets. Available variables are <code>point.key</code>,
	 * <code>series.name</code>, <code>series.color</code> and other members
	 * from the <code>point</code> and <code>series</code> objects. The
	 * <code>point.key</code> variable contains the category name, x value or
	 * datetime string depending on the type of axis. For datetime axes, the
	 * <code>point.key</code> date format can be set using tooltip.xDateFormat.
	 * </p>
	 */
	public void setHeaderFormat(String headerFormat) {
		setAttr(Attrs.headerFormat, headerFormat, "<span style=\"font-size: 10px\">{point.key}</span><br/>");
	}

	/**
	 * Returns the number of milliseconds to wait until the tooltip is hidden
	 * when mouse out from a point or chart.
	 * <p>
	 * Default: 500.
	 */
	public Number getHideDelay() {
		return getAttr(Attrs.hideDelay, 500).asNumber();
	}

	/**
	 * Sets the number of milliseconds to wait until the tooltip is hidden when
	 * mouse out from a point or chart.
	 */
	public void setHideDelay(Number hideDelay) {
		setAttr(Attrs.hideDelay, hideDelay, 500);
	}

	/**
	 * Returns the HTML of the point's line in the tooltip. Variables are
	 * enclosed by curly brackets. Available variables are point.x, point.y,
	 * series.name and series.color and other properties on the same form.
	 * Furthermore, point.y can be extended by the <code>tooltip.yPrefix</code>
	 * and <code>tooltip.ySuffix</code> variables. This can also be overridden
	 * for each series, which makes it a good hook for displaying units. </p>
	 * <p>
	 * Default:
	 * <code>&lt;span style="color:{series.color}"&gt;{series.name}&lt;/span&gt;: &lt;b&gt;{point.y}&lt;/b&gt;&lt;br/&gt;</code>
	 * </p>
	 */
	public String getPointFormat() {
		return getAttr(
				Attrs.pointFormat,
				"<span style=\"color:{series.color}\">{series.name}</span>: <b>{point.y}</b><br/>")
				.asString();
	}

	/**
	 * Sets the HTML of the point's line in the tooltip. Variables are enclosed
	 * by curly brackets. Available variables are point.x, point.y, series.name
	 * and series.color and other properties on the same form. Furthermore,
	 * point.y can be extended by the <code>tooltip.yPrefix</code> and
	 * <code>tooltip.ySuffix</code> variables. This can also be overridden for
	 * each series, which makes it a good hook for displaying units. </p>
	 */
	public void setPointFormat(String pointFormat) {
		setAttr(Attrs.pointFormat, pointFormat, "<span style=\"color:{series.color}\">{series.name}</span>: <b>{point.y}</b><br/>");
	}

	/**
	 * Returns how many decimals to show in each series' y value. This is
	 * overridable in each series' tooltip options object. The default is to
	 * preserve all decimals.
	 * <p>
	 * Default: null.
	 */
	public Number getValueDecimals() {
		return getAttr(Attrs.valueDecimals, null).asNumber();
	}

	/**
	 * Sets how many decimals to show in each series' y value. This is
	 * overridable in each series' tooltip options object. The default is to
	 * preserve all decimals.
	 */
	public void setValueDecimals(Number valueDecimals) {
		setAttr(Attrs.valueDecimals, valueDecimals);
	}

	/**
	 * Returns a string to prepend to each series' y value. Overridable in each
	 * series' tooltip options object.
	 * <p>
	 * Default: null.
	 */
	public String getValuePrefix() {
		return getAttr(Attrs.valuePrefix, null).asString();
	}

	/**
	 * Sets a string to prepend to each series' y value. Overridable in each
	 * series' tooltip options object.
	 */
	public void setValuePrefix(String valuePrefix) {
		setAttr(Attrs.valuePrefix, valuePrefix);
	}

	/**
	 * Returns a string to append to each series' y value. Overridable in each
	 * series' tooltip options object.
	 * <p>
	 * Default: null.
	 */
	public String getValueSuffix() {
		return getAttr(Attrs.valueSuffix, null).asString();
	}

	/**
	 * Sets the string to append to each series' y value. Overridable in each
	 * series' tooltip options object.
	 */
	public void setValueSuffix(String valueSuffix) {
		setAttr(Attrs.valueSuffix, valueSuffix);
	}

	/**
	 * Returns the format for the date in the tooltip header if the X axis is a
	 * datetime axis. The default is a best guess based on the smallest distance
	 * between points in the chart.
	 * <p>
	 * Default: null.
	 */
	public String getXDateFormat() {
		return getAttr(Attrs.xDateFormat, null).asString();
	}

	/**
	 * Sets the format for the date in the tooltip header if the X axis is a
	 * datetime axis. The default is a best guess based on the smallest distance
	 * between points in the chart.
	 */
	public void setXDateFormat(String xDateFormat) {
		setAttr(Attrs.xDateFormat, xDateFormat);
	}

}