/* SeriesPlotOptions.java

	Purpose:
		
	Description:
		
	History:
		Tue, Jan 14, 2014  5:08:03 PM, Created by RaymondChao

Copyright (C) 2014 Potix Corporation. All Rights Reserved.

 */
package org.zkoss.chart.plotOptions;

import org.zkoss.chart.Animation;
import org.zkoss.chart.Chart;
import org.zkoss.chart.Charts;
import org.zkoss.chart.Color;
import org.zkoss.chart.LinearGradient;
import org.zkoss.chart.Marker;
import org.zkoss.chart.OptionDataListener;
import org.zkoss.chart.Optionable;
import org.zkoss.chart.PlotAttribute;
import org.zkoss.chart.RadialGradient;
import org.zkoss.chart.Series;
import org.zkoss.chart.States;
import org.zkoss.chart.Tooltip;
import org.zkoss.chart.util.DynamicalAttribute;

/**
 * The plotOptions is a wrapper object for config series. The config objects for
 * each series can also be overridden for each series item as given in the
 * {@link Series}.
 * <p>
 * All the options in this class support {@link DynamicalAttribute}.
 * 
 * @author jumperchen
 * @author RaymondChao
 */
public class SeriesPlotOptions extends Optionable implements OptionDataListener {
	private enum Attrs implements PlotAttribute, DynamicalAttribute {
		allowPointSelect,
		borderWidth,
		animation,
		color,
		connectEnds,
		connectNulls,
		cropThreshold,
		cursor,
		dashStyle,
		dataLabels,
		enableMouseTracking,
		id,
		lineWidth,
		linkedTo,
		marker,
		neckHeight,
		neckWidth,
		negativeColor,
		point,
		pointInterval,
		pointPlacement,
		pointStart,
		selected,
		shadow,
		showCheckbox,
		showInLegend,
		stacking,
		states,
		stickyTracking,
		threshold,
		tooltip,
		turboThreshold,
		visible
	}

	/**
	 * Returns to allow this series' points to be selected by clicking on the
	 * markers, bars or pie slices.
	 * <p>
	 * Default: false
	 */
	public boolean isAllowPointSelect() {
		return getAttr(Attrs.allowPointSelect, false).asBoolean();
	}

	/**
	 * Sets to allow this series' points to be selected by clicking on the
	 * markers, bars or pie slices.
	 * <p>
	 * Default: false
	 */
	public void setAllowPointSelect(boolean allowPointSelect) {
		setAttr(Attrs.allowPointSelect, allowPointSelect);
	}

	/**
	 * Returns whether to enable or disable the initial animation when a series
	 * is displayed. The animation can also be set as a {@link Animation}
	 * object. Please note that this option only applies to the initial
	 * animation of the series itself. For other animations, see
	 * {@link Chart#setAnimation(Animation)} and the animation parameter under
	 * the API methods. The following properties are supported: </p>
	 * <dl>
	 * <dt>duration</dt>
	 * <dd>The duration of the animation in milliseconds.</dd>
	 * <dt>easing</dt>
	 * <dd>When using jQuery as the general framework, the easing can be set to
	 * <code>linear</code> or <code>swing</code>. More easing functions are
	 * available with the use of jQuery plug-ins, most notably the jQuery UI
	 * suite. See <a href="http://api.jquery.com/animate/">the jQuery docs</a>.
	 * When using MooTools as the general framework, use the property name
	 * <code>transition</code> instead of <code>easing</code>.</dd>
	 * </dl>
	 * <p>
	 * due to poor performance, animation is disabled in old IE browsers for
	 * column charts and polar charts.
	 * </p>
	 */
	public boolean isAnimation() {
		return getAnimation() != Animation.NONE;
	}

	/**
	 * Returns whether to enable or disable the initial animation when a series
	 * is displayed. The animation can also be set as a {@link Animation}
	 * object.
	 * <p>
	 * Default: an instance of {@link Animation}
	 */
	public Animation getAnimation() {
		if (containsKey(Attrs.animation)) {
			return (Animation) getAttr(Attrs.animation);
		} else {
			return (Animation) getAttr(Attrs.animation,
					org.zkoss.chart.Animation.class).asValue();
		}
	}

	/**
	 * Sets whether to enable or disable the initial animation when a series is
	 * displayed. The animation can also be set as a {@link Animation} object.
	 * Please note that this option only applies to the initial animation of the
	 * series itself. For other animations, see
	 * {@link Chart#setAnimation(Animation)} and the animation parameter under
	 * the API methods. The following properties are supported: </p>
	 * <dl>
	 * <dt>duration</dt>
	 * <dd>The duration of the animation in milliseconds.</dd>
	 * <dt>easing</dt>
	 * <dd>When using jQuery as the general framework, the easing can be set to
	 * <code>linear</code> or <code>swing</code>. More easing functions are
	 * available with the use of jQuery plug-ins, most notably the jQuery UI
	 * suite. See <a href="http://api.jquery.com/animate/">the jQuery docs</a>.
	 * When using MooTools as the general framework, use the property name
	 * <code>transition</code> instead of <code>easing</code>.</dd>
	 * </dl>
	 * <p>
	 * due to poor performance, animation is disabled in old IE browsers for
	 * column charts and polar charts.
	 * </p>
	 * 
	 * @see #setAnimation(Animation)
	 * @see Animation
	 */
	public void setAnimation(boolean animation) {
		setAnimation(animation ? new Animation() : Animation.NONE);
	}

	/**
	 * Sets whether to enable or disable the initial animation when a series is
	 * displayed. The animation can also be set as a {@link Animation} object.
	 * Please note that this option only applies to the initial animation of the
	 * series itself. For other animations, see
	 * {@link Chart#setAnimation(Animation)} and the animation parameter under
	 * the API methods. The following properties are supported: </p>
	 * <dl>
	 * <dt>duration</dt>
	 * <dd>The duration of the animation in milliseconds.</dd>
	 * <dt>easing</dt>
	 * <dd>When using jQuery as the general framework, the easing can be set to
	 * <code>linear</code> or <code>swing</code>. More easing functions are
	 * available with the use of jQuery plug-ins, most notably the jQuery UI
	 * suite. See <a href="http://api.jquery.com/animate/">the jQuery docs</a>.
	 * When using MooTools as the general framework, use the property name
	 * <code>transition</code> instead of <code>easing</code>.</dd>
	 * </dl>
	 * <p>
	 * due to poor performance, animation is disabled in old IE browsers for
	 * column charts and polar charts.
	 * </p>
	 * 
	 * @see #setAnimation(boolean)
	 * @see Animation
	 */
	public void setAnimation(Animation animation) {
		setAttr(Attrs.animation, animation == null ? Animation.NONE : animation);
	}

	/**
	 * Returns the border width of the series.
	 * <p>
	 * Default: null
	 */
	public Number getBorderWidth() {
		return getAttr(Attrs.borderWidth, null).asNumber();
	}

	/**
	 * Sets the border width of the series.
	 */
	public void setBorderWidth(Number borderWidth) {
		setAttr(Attrs.borderWidth, borderWidth);
	}

	/**
	 * Returns the main color of the series. In line type series it applies to
	 * the line and the point markers unless otherwise specified. In bar type
	 * series it applies to the bars unless a color is specified per point. The
	 * default value is pulled from the {@link Charts#getColors()} array.
	 */
	public Color getColor() {
		return (Color) getAttr(Attrs.color, null).asValue();
	}

	/**
	 * Sets the main color of the series. In line type series it applies to the
	 * line and the point markers unless otherwise specified. In bar type series
	 * it applies to the bars unless a color is specified per point. The default
	 * value is pulled from the {@link Charts#getColors()} array.
	 */
	public void setColor(Color color) {
		setAttr(Attrs.color, color);
	}

	/**
	 * Sets the main color of the series. In line type series it applies to the
	 * line and the point markers unless otherwise specified. In bar type series
	 * it applies to the bars unless a color is specified per point. The default
	 * value is pulled from the {@link Charts#getColors()} array.
	 */
	public void setColor(String color) {
		setColor(new Color(color));
	}

	/**
	 * Sets the main color of the series. In line type series it applies to the
	 * line and the point markers unless otherwise specified. In bar type series
	 * it applies to the bars unless a color is specified per point. The default
	 * value is pulled from the {@link Charts#getColors()} array.
	 */
	public void setColor(LinearGradient color) {
		setColor(new Color(color));
	}

	/**
	 * Sets the main color of the series. In line type series it applies to the
	 * line and the point markers unless otherwise specified. In bar type series
	 * it applies to the bars unless a color is specified per point. The default
	 * value is pulled from the {@link Charts#getColors()} array.
	 */
	public void setColor(RadialGradient color) {
		setColor(new Color(color));
	}

	/**
	 * Returns polar charts only. Whether to connect the ends of a line series
	 * plot across the extremes.
	 */
	public boolean isConnectEnds() {
		return getAttr(Attrs.connectEnds, true).asBoolean();
	}

	/**
	 * Sets polar charts only. Whether to connect the ends of a line series plot
	 * across the extremes.
	 */
	public void setConnectEnds(boolean connectEnds) {
		setAttr(Attrs.connectEnds, connectEnds);
	}

	/**
	 * Returns whether to connect a graph line across null points.
	 */
	public boolean isConnectNulls() {
		return getAttr(Attrs.connectNulls, false).asBoolean();
	}

	/**
	 * Sets whether to connect a graph line across null points.
	 */
	public void setConnectNulls(boolean connectNulls) {
		setAttr(Attrs.connectNulls, connectNulls);
	}

	/**
	 * Returns when the series contains less points than the crop threshold, all
	 * points are drawn, even if the points fall outside the visible plot area
	 * at the current zoom. The advantage of drawing all points (including
	 * markers and columns), is that animation is performed on updates. On the
	 * other hand, when the series contains more points than the crop threshold,
	 * the series data is cropped to only contain points that fall within the
	 * plot area. The advantage of cropping away invisible points is to increase
	 * performance on large series.
	 */
	public Number getCropThreshold() {
		return getAttr(Attrs.cropThreshold, 300).asNumber();
	}

	/**
	 * Sets when the series contains less points than the crop threshold, all
	 * points are drawn, even if the points fall outside the visible plot area
	 * at the current zoom. The advantage of drawing all points (including
	 * markers and columns), is that animation is performed on updates. On the
	 * other hand, when the series contains more points than the crop threshold,
	 * the series data is cropped to only contain points that fall within the
	 * plot area. The advantage of cropping away invisible points is to increase
	 * performance on large series.
	 */
	public void setCropThreshold(Number cropThreshold) {
		setAttr(Attrs.cropThreshold, cropThreshold, 300);
	}

	/**
	 * Returns you can set the cursor to "pointer" if you have click events
	 * attached to the series, to signal to the user that the points and lines
	 * can be clicked.
	 */
	public String getCursor() {
		return getAttr(Attrs.cursor, null).asString();
	}

	/**
	 * Sets you can set the cursor to "pointer" if you have click events
	 * attached to the series, to signal to the user that the points and lines
	 * can be clicked.
	 */
	public void setCursor(String cursor) {
		setAttr(Attrs.cursor, cursor);
	}

	/**
	 * Returns a name for the dash style to use for the graph. Applies only to
	 * series type having a graph, like <code>line</code>, <code>spline</code>,
	 * <code>area</code> and <code>scatter</code> in case it has a
	 * <code>lineWidth</code>. The value for the <code>dashStyle</code> include:
	 * <ul>
	 * <li>Solid</li>
	 * <li>ShortDash</li>
	 * <li>ShortDot</li>
	 * <li>ShortDashDot</li>
	 * <li>ShortDashDotDot</li>
	 * <li>Dot</li>
	 * <li>Dash</li>
	 * <li>LongDash</li>
	 * <li>DashDot</li>
	 * <li>LongDashDot</li>
	 * <li>LongDashDotDot</li>
	 * </ul>
	 */
	public String getDashStyle() {
		return getAttr(Attrs.dashStyle, "Solid").asString();
	}

	/**
	 * Sets a name for the dash style to use for the graph. Applies only to
	 * series type having a graph, like <code>line</code>, <code>spline</code>,
	 * <code>area</code> and <code>scatter</code> in case it has a
	 * <code>lineWidth</code>. The value for the <code>dashStyle</code> include:
	 * <ul>
	 * <li>Solid</li>
	 * <li>ShortDash</li>
	 * <li>ShortDot</li>
	 * <li>ShortDashDot</li>
	 * <li>ShortDashDotDot</li>
	 * <li>Dot</li>
	 * <li>Dash</li>
	 * <li>LongDash</li>
	 * <li>DashDot</li>
	 * <li>LongDashDot</li>
	 * <li>LongDashDotDot</li>
	 * </ul>
	 */
	public void setDashStyle(String dashStyle) {
		if (!("Solid".equalsIgnoreCase(dashStyle)
				|| "ShortDash".equalsIgnoreCase(dashStyle)
				|| "ShortDot".equalsIgnoreCase(dashStyle)
				|| "ShortDashDot".equalsIgnoreCase(dashStyle)
				|| "ShortDashDotDot".equalsIgnoreCase(dashStyle)
				|| "Dot".equalsIgnoreCase(dashStyle)
				|| "Dash".equalsIgnoreCase(dashStyle)
				|| "LongDash".equalsIgnoreCase(dashStyle)
				|| "DashDot".equalsIgnoreCase(dashStyle)
				|| "LongDashDot".equalsIgnoreCase(dashStyle) || "LongDashDotDot"
					.equalsIgnoreCase(dashStyle)))
			throw new IllegalArgumentException("Unsupported style: ["
					+ dashStyle + "]");
		setAttr(Attrs.dashStyle, dashStyle, "Solid");
	}

	/**
	 * Returns the option of data labels.
	 */
	public DataLabels getDataLabels() {
		DataLabels dataLabels = (DataLabels) this.getAttr(Attrs.dataLabels);

		if (dataLabels == null) {
			dataLabels = new DataLabels();
			setDataLabels(dataLabels);
		}
		return dataLabels;
	}

	/**
	 * Sets the option of data labels.
	 * 
	 * @see DataLabels
	 */
	public void setDataLabels(DataLabels dataLabels) {
		setAttr(Attrs.dataLabels, dataLabels);
	}

	/**
	 * Returns enable or disable the mouse tracking for a specific series. This
	 * includes point tooltips and click events on graphs and points. For large
	 * datasets it improves performance.
	 */
	public boolean isEnableMouseTracking() {
		return getAttr(Attrs.enableMouseTracking, true).asBoolean();
	}

	/**
	 * Sets enable or disable the mouse tracking for a specific series. This
	 * includes point tooltips and click events on graphs and points. For large
	 * datasets it improves performance.
	 */
	public void setEnableMouseTracking(boolean enableMouseTracking) {
		setAttr(Attrs.enableMouseTracking, enableMouseTracking);
	}

	/**
	 * Returns an id for the series.
	 */
	public String getId() {
		return getAttr(Attrs.id, null).asString();
	}

	/**
	 * Sets an id for the series.
	 */
	public void setId(String id) {
		setAttr(Attrs.id, id);
	}

	/**
	 * Returns pixel with of the graph line.
	 */
	public Number getLineWidth() {
		return getAttr(Attrs.lineWidth, 2).asNumber();
	}

	/**
	 * Sets pixel with of the graph line.
	 */
	public void setLineWidth(Number lineWidth) {
		setAttr(Attrs.lineWidth, lineWidth, 2);
	}

	/**
	 * Returns the {@link #setId(String)} of another series to link to.
	 * Additionally, the value can be ":previous" to link to the previous
	 * series. When two series are linked, only the first one appears in the
	 * legend. Toggling the visibility of this also toggles the linked series.
	 */
	public String getLinkedTo() {
		return getAttr(Attrs.linkedTo, null).asString();
	}

	/**
	 * Sets the {@link #setId(String)} of another series to link to.
	 * Additionally, the value can be ":previous" to link to the previous
	 * series. When two series are linked, only the first one appears in the
	 * legend. Toggling the visibility of this also toggles the linked series.
	 */
	public void setLinkedTo(String linkedTo) {
		setAttr(Attrs.linkedTo, linkedTo);
	}

	/**
	 * Returns the series marker
	 */
	public Marker getMarker() {
		Marker marker = (Marker) getAttr(Attrs.marker);
		if (marker == null) {
			marker = new Marker();
			setMarker(marker);
		}
		return marker;
	}

	/**
	 * Sets the series marker
	 */
	public void setMarker(Marker marker) {
		setAttr(Attrs.marker, marker);
	}

	/**
	 * Returns the neck height for funnel chart only.
	 */
	public String getNeckHeight() {
		return getAttr(Attrs.neckHeight, null).asString();
	}

	/**
	 * Sets the neck height for funnel chart only.
	 */
	public void setNeckHeight(String neckHeight) {
		setAttr(Attrs.neckHeight, neckHeight);
	}

	/**
	 * Returns the neck width for funnel chart only.
	 */
	public String getNeckWidth() {
		return getAttr(Attrs.neckWidth, null).asString();
	}

	/**
	 * Sets the neck width for funnel chart only.
	 */
	public void setNeckWidth(String neckWidth) {
		setAttr(Attrs.neckWidth, neckWidth);
	}

	/**
	 * Returns the color for the parts of the graph or points that are
	 * below the {@link #setThreshold(Number)}.
	 * <p> Defaults to null.
	 */
	public Color getNegativeColor() {
		return (Color) getAttr(Attrs.negativeColor, null).asValue();
	}

	/**
	 * Sets the color for the parts of the graph or points that are
	 * below the {@link #setThreshold(Number)}.
	 * <p> Defaults to null.
	 */
	public void setNegativeColor(Color color) {
		setAttr(Attrs.negativeColor, color);
	}

	/**
	 * Sets the color for the parts of the graph or points that are
	 * below the {@link #setThreshold(Number)}.
	 * <p> Defaults to null.
	 */
	public void setNegativeColor(String color) {
		setNegativeColor(new Color(color));
	}

	/**
	 * Sets the color for the parts of the graph or points that are
	 * below the {@link #setThreshold(Number)}.
	 * <p> Defaults to null.
	 */
	public void setNegativeColor(LinearGradient color) {
		setNegativeColor(new Color(color));
	}

	/**
	 * Sets the color for the parts of the graph or points that are
	 * below the {@link #setThreshold(Number)}.
	 * <p> Defaults to null.
	 */
	public void setNegativeColor(RadialGradient color) {
		setNegativeColor(new Color(color));
	}

	/**
	 * Returns if no x values are given for the points in a series,
	 * pointInterval defines the interval of the x values. For example, if a
	 * series contains one value every decade starting from year 0, set
	 * pointInterval to 10.
	 * <p>
	 * Default: 1
	 */
	public Number getPointInterval() {
		return getAttr(Attrs.pointInterval, 1).asNumber();
	}

	/**
	 * Sets if no x values are given for the points in a series, pointInterval
	 * defines the interval of the x values. For example, if a series contains
	 * one value every decade starting from year 0, set pointInterval to 10.
	 * <p>
	 * Default: 1
	 */
	public void setPointInterval(Number pointInterval) {
		setAttr(Attrs.pointInterval, pointInterval, 1);
	}

	/**
	 * Returns the point placement.
	 * <p>
	 * Possible values: null, "on", "between" or numeric.
	 * </p>
	 * <p>
	 * In a column chart, when pointPlacement is "on", the point will not create
	 * any padding of the X axis. In a polar column chart this means that the
	 * first column points directly north. If the pointPlacement is "between",
	 * the columns will be laid out between ticks. This is useful for example
	 * for visualising an amount between two points in time or in a certain
	 * sector of a polar chart.
	 * </p>
	 * <p>
	 * The point placement can also be numeric, where 0 is on the axis value,
	 * -0.5 is between this value and the previous, and 0.5 is between this
	 * value and the next. Unlike the textual options, numeric point placement
	 * options won't affect axis padding.
	 * </p>
	 * <p>
	 * Defaults to <code>null</code> in cartesian charts, <code>"between"</code>
	 * in polar charts.
	 */
	public Object getPointPlacement() {
		return getAttr(Attrs.pointPlacement, null).asValue();
	}

	/**
	 * Sets the point placement.
	 * <p>
	 * Possible values: null, "on", "between".
	 * </p>
	 * <p>
	 * In a column chart, when pointPlacement is "on", the point will not create
	 * any padding of the X axis. In a polar column chart this means that the
	 * first column points directly north. If the pointPlacement is "between",
	 * the columns will be laid out between ticks. This is useful for example
	 * for visualising an amount between two points in time or in a certain
	 * sector of a polar chart.
	 * <p>
	 * Defaults to <code>null</code> in cartesian charts, <code>"between"</code>
	 * in polar charts.
	 */
	public void setPointPlacement(String pointPlacement) {
		if (!("on".equals(pointPlacement) || "between".equals(pointPlacement)))
			throw new IllegalArgumentException("Unsupported pointPlacement: ["
					+ pointPlacement + "]");
		setAttr(Attrs.pointPlacement, pointPlacement);
	}

	/**
	 * Sets the point placement with a number
	 * <p>
	 * The point placement can also be numeric, where 0 is on the axis value,
	 * -0.5 is between this value and the previous, and 0.5 is between this
	 * value and the next. Unlike the textual options, numeric point placement
	 * options won't affect axis padding.
	 * </p>
	 */
	public void setPointPlacement(Number pointPlacement) {
		setAttr(Attrs.pointPlacement, pointPlacement);
	}

	/**
	 * Returns if no x values are given for the points in a series, pointStart
	 * defines on what value to start. For example, if a series contains one
	 * yearly value starting from 1945, set pointStart to 1945.
	 */
	public Number getPointStart() {
		return getAttr(Attrs.pointStart, 0).asNumber();
	}

	/**
	 * Sets if no x values are given for the points in a series, pointStart
	 * defines on what value to start. For example, if a series contains one
	 * yearly value starting from 1945, set pointStart to 1945.
	 */
	public void setPointStart(Number pointStart) {
		setAttr(Attrs.pointStart, pointStart, 0);
	}

	/**
	 * Returns whether to select the series initially. If
	 * <code>showCheckbox</code> is true, the checkbox next to the series name
	 * will be checked for a selected series.
	 */
	public boolean isSelected() {
		return getAttr(Attrs.selected, false).asBoolean();
	}

	/**
	 * Sets whether to select the series initially. If <code>showCheckbox</code>
	 * is true, the checkbox next to the series name will be checked for a
	 * selected series.
	 */
	public void setSelected(boolean selected) {
		setAttr(Attrs.selected, selected);
	}

	/**
	 * Returns whether to apply a drop shadow to the graph line. Since 2.3 the
	 * shadow can be an object configuration containing <code>color</code>,
	 * <code>offsetX</code>, <code>offsetY</code>, <code>opacity</code> and
	 * <code>width</code>.
	 */
	public boolean isShadow() {
		return getAttr(Attrs.shadow, false).asBoolean();
	}

	/**
	 * Sets whether to apply a drop shadow to the graph line. Since 2.3 the
	 * shadow can be an object configuration containing <code>color</code>,
	 * <code>offsetX</code>, <code>offsetY</code>, <code>opacity</code> and
	 * <code>width</code>.
	 */
	public void setShadow(boolean shadow) {
		setAttr(Attrs.shadow, shadow);
	}

	/**
	 * Returns if true, a checkbox is displayed next to the legend item to allow
	 * selecting the series. The state of the checkbox is determined by the
	 * <code>selected</code> option.
	 */
	public boolean isShowCheckbox() {
		return getAttr(Attrs.showCheckbox, false).asBoolean();
	}

	/**
	 * Sets if true, a checkbox is displayed next to the legend item to allow
	 * selecting the series. The state of the checkbox is determined by the
	 * <code>selected</code> option.
	 */
	public void setShowCheckbox(boolean showCheckbox) {
		setAttr(Attrs.showCheckbox, showCheckbox);
	}

	/**
	 * Returns whether to display this particular series or series type in the
	 * legend. The default value is <code>true</code> for standalone series,
	 * <code>false</code> for linked series.
	 */
	public boolean isShowInLegend() {
		return getAttr(Attrs.showInLegend, true).asBoolean();
	}

	/**
	 * Sets whether to display this particular series or series type in the
	 * legend. The default value is <code>true</code> for standalone series,
	 * <code>false</code> for linked series.
	 */
	public void setShowInLegend(boolean showInLegend) {
		setAttr(Attrs.showInLegend, showInLegend);
	}

	/**
	 * Returns whether to stack the values of each series on top of each other.
	 * Possible values are null to disable, "normal" to stack by value or
	 * "percent".
	 */
	public String getStacking() {
		return getAttr(Attrs.stacking, null).asString();
	}

	/**
	 * Sets whether to stack the values of each series on top of each other.
	 * Possible values are null to disable, "normal" to stack by value or
	 * "percent".
	 */
	public void setStacking(String stacking) {
		if (stacking != null
				&& !("normal".equals(stacking) || "percent".equals(stacking)))
			throw new IllegalArgumentException("Unsupported value: ["
					+ stacking + "]");
		setAttr(Attrs.stacking, stacking);
	}

	/**
	 * Returns a wrapper object for all the series options in specific states.
	 */
	public States getStates() {
		States states = (States) getAttr(Attrs.states);
		if (states == null) {
			states = new States();
			setStates(states);
		}
		return states;
	}

	/**
	 * Sets a wrapper object for all the series options in specific states.
	 */
	public void setStates(States states) {
		setAttr(Attrs.states, states);
	}

	/**
	 * Returns sticky tracking of mouse events. When true, the
	 * <code>mouseOut</code> event on a series isn't triggered until the mouse
	 * moves over another series, or out of the plot area. When false, the
	 * <code>mouseOut</code> event on a series is triggered when the mouse
	 * leaves the area around the series' graph or markers. This also implies
	 * the tooltip. When <code>stickyTracking</code> is false and
	 * {@link Tooltip#isShared()} is false, the tooltip will be hidden when
	 * moving the mouse between series. Defaults to true for line and area type
	 * series, but to false for columns, pies etc.
	 */
	public boolean isStickyTracking() {
		return getAttr(Attrs.stickyTracking, true).asBoolean();
	}

	/**
	 * Sets sticky tracking of mouse events. When true, the
	 * <code>mouseOut</code> event on a series isn't triggered until the mouse
	 * moves over another series, or out of the plot area. When false, the
	 * <code>mouseOut</code> event on a series is triggered when the mouse
	 * leaves the area around the series' graph or markers. This also implies
	 * the tooltip. When <code>stickyTracking</code> is false and
	 * {@link Tooltip#isShared()} is false, the tooltip will be hidden when
	 * moving the mouse between series. Defaults to true for line and area type
	 * series, but to false for columns, pies etc.
	 */
	public void setStickyTracking(boolean stickyTracking) {
		setAttr(Attrs.stickyTracking, stickyTracking);
	}

	/**
	 * Returns the threshold, also called zero level or base level. For line
	 * type series this is only used in conjunction with
	 * {@link #setNegativeColor(String)}
	 */
	public Number getThreshold() {
		return getAttr(Attrs.threshold, 0).asNumber();
	}

	/**
	 * Sets the threshold, also called zero level or base level. For line type
	 * series this is only used in conjunction with
	 * {@link #setNegativeColor(String)}
	 */
	public void setThreshold(Number threshold) {
		setAttr(Attrs.threshold, threshold, 0);
	}

	/**
	 * Returns a configuration object for the tooltip rendering of each single
	 * series. Properties are inherited from {@link Tooltip}, but only the
	 * following properties can be defined on a series level.
	 */
	public TooltipPlotOptions getTooltip() {
		TooltipPlotOptions tooltip = (TooltipPlotOptions) getAttr(Attrs.tooltip);
		if (tooltip == null) {
			tooltip = new TooltipPlotOptions();
			setTooltip(tooltip);
		}
		return tooltip;
	}

	/**
	 * Sets a configuration object for the tooltip rendering of each single
	 * series. Properties are inherited from {@link Tooltip}, but only the
	 * following properties can be defined on a series level.
	 */
	public void setTooltip(TooltipPlotOptions tooltip) {
		setAttr(Attrs.tooltip, tooltip);
	}

	/**
	 * Returns when a series contains a data array that is longer than this,
	 * only one dimensional arrays of numbers, or two dimensional arrays with x
	 * and y values are allowed. Also, only the first point is tested, and the
	 * rest are assumed to be the same format. This saves expensive data
	 * checking and indexing in long series. Set it to <code>0</code> disable.
	 */
	public Number getTurboThreshold() {
		return getAttr(Attrs.turboThreshold, 1000).asNumber();
	}

	/**
	 * Sets when a series contains a data array that is longer than this, only
	 * one dimensional arrays of numbers, or two dimensional arrays with x and y
	 * values are allowed. Also, only the first point is tested, and the rest
	 * are assumed to be the same format. This saves expensive data checking and
	 * indexing in long series. Set it to <code>0</code> disable.
	 */
	public void setTurboThreshold(Number turboThreshold) {
		setAttr(Attrs.turboThreshold, turboThreshold, 1000);
	}

	/**
	 * Returns set the initial visibility of the series.
	 */
	public boolean isVisible() {
		return getAttr(Attrs.visible, true).asBoolean();
	}

	/**
	 * Sets set the initial visibility of the series.
	 */
	public void setVisible(boolean visible) {
		setAttr(Attrs.visible, visible);
	}
}