/* PiePlotOptions.java

	Purpose:
		
	Description:
		
	History:
		Thu, Jan 16, 2014  4:25:01 PM, Created by RaymondChao

Copyright (C) 2014 Potix Corporation. All Rights Reserved.

 */
package org.zkoss.chart.plotOptions;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.zkoss.chart.Chart;
import org.zkoss.chart.Charts;
import org.zkoss.chart.Color;
import org.zkoss.chart.LinearGradient;
import org.zkoss.chart.PlotAttribute;
import org.zkoss.chart.RadialGradient;
import org.zkoss.chart.util.DynamicalAttribute;
import org.zkoss.lang.Generics;

/**
 * An option for pie plot
 * <p>
 * All the options in this class support {@link DynamicalAttribute}.
 * 
 * @author jumperchen
 * @author RaymondChao
 */
public class PiePlotOptions extends SeriesPlotOptions {
	private enum Attrs implements PlotAttribute, DynamicalAttribute {
		borderColor,
		borderWidth,
		center,
		colors,
		dataLabels,
		endAngle,
		events,
		ignoreHiddenPoint,
		innerSize,
		minSize,
		showInLegend,
		size,
		slicedOffset,
		startAngle,
		stickyTracking
	}

	/**
	 * Returns the option of data labels.
	 */
	public DataLabels getDataLabels() {
		DataLabels dataLabels = (DataLabels) this.getAttr(Attrs.dataLabels);

		if (dataLabels == null) {
			dataLabels = new PieDataLabels();
			setDataLabels(dataLabels);
		}
		return dataLabels;
	}

	/**
	 * Returns the color of the border surrounding each slice.
	 * <p>
	 * Default: #FFFFFF
	 */
	public Color getBorderColor() {
		if (!containsKey(Attrs.borderColor)) {
			setBorderColor("#FFFFFF");
		}
		return (Color) getAttr(Attrs.borderColor);
	}

	/**
	 * Sets the color of the border surrounding each slice.
	 */
	public void setBorderColor(Color color) {
		setAttr(Attrs.borderColor, color, NOT_NULL_VALUE);
	}

	/**
	 * Sets the color of the border surrounding each slice.
	 */
	public void setBorderColor(String color) {
		setBorderColor(new Color(color));
	}

	/**
	 * Sets the color of the border surrounding each slice.
	 */
	public void setBorderColor(LinearGradient color) {
		setBorderColor(new Color(color));
	}

	/**
	 * Sets the color of the border surrounding each slice.
	 */
	public void setBorderColor(RadialGradient color) {
		setBorderColor(new Color(color));
	}

	/**
	 * Returns the width of the border surrounding each slice.
	 * <p>
	 * Default: 1
	 */
	public Number getBorderWidth() {
		return getAttr(Attrs.borderWidth, 1).asNumber();
	}

	/**
	 * Sets the width of the border surrounding each slice.
	 */
	public void setBorderWidth(Number borderWidth) {
		setAttr(Attrs.borderWidth, borderWidth, 1);
	}

	/**
	 * Returns the center of the pie chart relative to the plot area. Can be
	 * percentages or pixel values. The default behavior is to center the pie so
	 * that all slices and data labels are within the plot area. As a
	 * consequence, the pie may actually jump around in a chart with dynamic
	 * values, as the data labels move. In that case, the center should be
	 * explicitly set, for example to <code>["50%", "50%"]</code>.
	 */
	public List<String> getCenter() {
		if (!containsKey(Attrs.center)) {
			ArrayList<String> list = new ArrayList<String>();
			list.add("50%");
			list.add("50%");
			setCenter(list);
		}
		return Generics.cast(getAttr(Attrs.center));
	}

	/**
	 * Sets the center of the pie chart relative to the plot area. Can be
	 * percentages or pixel values. The default behavior is to center the pie so
	 * that all slices and data labels are within the plot area. As a
	 * consequence, the pie may actually jump around in a chart with dynamic
	 * values, as the data labels move. In that case, the center should be
	 * explicitly set, for example to <code>["50%", "50%"]</code>.
	 */
	public <T> void setCenter(List<T> center) {
		setAttr(Attrs.center, center, NOT_NULL_VALUE);
	}
	
	public void setCenter(Object... center) {
		setCenter(Arrays.asList(center));
	}

	/**
	 * Returns a series specific or series type specific color set to use
	 * instead of the global {@link Charts#getColors()}.
	 */
	public List<Color> getColors() {
		return Generics.cast(getAttr(Attrs.colors, null).asValue());
	}

	/**
	 * Sets a series specific or series type specific color set to use instead
	 * of the global {@link Charts#getColors()}.
	 */
	public <T> void setColors(List<Color> colors) {
		setAttr(Attrs.colors, colors);
	}

	/**
	 * Sets a series specific or series type specific color set to use instead
	 * of the global {@link Charts#getColors()}.
	 */
	public void setColors(String... source) {
		Color[] colors = new Color[source.length];
		int i = 0;
		for (String s : source)
			colors[i++] = new Color(s);
		setColors(Arrays.asList(colors));
	}

	/**
	 * Returns the end angle of the pie in degrees where 0 is top and 90 is
	 * right.
	 * <p>
	 * Defaults to <code>startAngle</code> plus 360.
	 */
	public Number getEndAngle() {
		if (containsKey(Attrs.endAngle)) {
			return (Number) getAttr(Attrs.endAngle);
		} else {
			return (Integer) getStartAngle() + 360;
		}
	}

	/**
	 * Sets the end angle of the pie in degrees where 0 is top and 90 is right.
	 * <p>
	 * Defaults to <code>startAngle</code> plus 360.
	 */
	public void setEndAngle(Number endAngle) {
		setAttr(Attrs.endAngle, endAngle, NOT_NULL_VALUE);
	}

	/**
	 * Returns whether ignore the hidden point.
	 * <p>
	 * Equivalent to {@link Chart#isIgnoreHiddenSeries()}, this option tells
	 * whether the series shall be redrawn as if the hidden point were
	 * <code>null</code>.
	 * </p>
	 * <p>
	 * The default value true.
	 * </p>
	 */
	public boolean isIgnoreHiddenPoint() {
		return getAttr(Attrs.ignoreHiddenPoint, true).asBoolean();
	}

	/**
	 * Sets whether ignore the hidden point.
	 * <p>
	 * Equivalent to {@link Chart#isIgnoreHiddenSeries()}, this option tells
	 * whether the series shall be redrawn as if the hidden point were
	 * <code>null</code>.
	 * </p>
	 * <p>
	 * The default value true.
	 * </p>
	 */
	public void setIgnoreHiddenPoint(boolean ignoreHiddenPoint) {
		setAttr(Attrs.ignoreHiddenPoint, ignoreHiddenPoint);
	}

	/**
	 * Returns the size of the inner diameter for the pie. A size greater than 0
	 * renders a donut chart. Can be a percentage or pixel value. Percentages
	 * are relative to the size of the plot area. Pixel values are given as
	 * integers.
	 */
	public String getInnerSize() {
		return getAttr(Attrs.innerSize, "0").asString();
	}

	/**
	 * Sets the size of the inner diameter for the pie. A size greater than 0
	 * renders a donut chart. Can be a percentage or pixel value. Percentages
	 * are relative to the size of the plot area. Pixel values are given as
	 * integers.
	 */
	public void setInnerSize(String innerSize) {
		setAttr(Attrs.innerSize, innerSize, "0");
	}

	/**
	 * Sets the size of the inner diameter for the pie. A size greater than 0
	 * renders a donut chart. Can be a percentage or pixel value. Percentages
	 * are relative to the size of the plot area. Pixel values are given as
	 * integers.
	 */
	public void setInnerSize(Number innerSize) {
		setAttr(Attrs.innerSize, innerSize, 0);
	}

	/**
	 * Returns the minimum size for a pie in response to auto margins. The pie
	 * will try to shrink to make room for data labels in side the plot area,
	 * but only to this size.
	 */
	public Number getMinSize() {
		return getAttr(Attrs.minSize, 80).asNumber();
	}

	/**
	 * Sets the minimum size for a pie in response to auto margins. The pie will
	 * try to shrink to make room for data labels in side the plot area, but
	 * only to this size.
	 */
	public void setMinSize(Number minSize) {
		setAttr(Attrs.minSize, minSize, 80);
	}

	/**
	 * Returns whether to display this particular series or series type in the
	 * legend. Pies are not shown in the legend by default.
	 */
	public boolean isShowInLegend() {
		return getAttr(Attrs.showInLegend, false).asBoolean();
	}

	/**
	 * Sets whether to display this particular series or series type in the
	 * legend. Pies are not shown in the legend by default.
	 */
	public void setShowInLegend(boolean showInLegend) {
		setAttr(Attrs.showInLegend, showInLegend);
	}

	/**
	 * Returns the diameter of the pie relative to the plot area. Can be a
	 * percentage or pixel value. Pixel values are given as integers. The
	 * default behavior is to scale to the plot area and give room for data
	 * labels within the plot area. As a consequence, the size of the pie may
	 * vary when points are updated and data labels more around. In that case it
	 * is best to set a fixed value, for example <code>"75%"</code>.
	 * <p>
	 * Default: null
	 */
	public Object getSize() {
		return getAttr(Attrs.size, null).asValue();
	}

	/**
	 * Sets the diameter of the pie relative to the plot area. Can be a
	 * percentage or pixel value. Pixel values are given as integers. The
	 * default behavior is to scale to the plot area and give room for data
	 * labels within the plot area. As a consequence, the size of the pie may
	 * vary when points are updated and data labels more around. In that case it
	 * is best to set a fixed value, for example <code>"75%"</code>.
	 */
	public void setSize(String size) {
		setAttr(Attrs.size, size);
	}

	/**
	 * Sets the diameter of the pie relative to the plot area. Can be a
	 * percentage or pixel value. Pixel values are given as integers. The
	 * default behavior is to scale to the plot area and give room for data
	 * labels within the plot area. As a consequence, the size of the pie may
	 * vary when points are updated and data labels more around. In that case it
	 * is best to set a fixed value, for example <code>"75%"</code>.
	 */
	public void setSize(Number size) {
		setAttr(Attrs.size, size);
	}

	/**
	 * Returns if a point is sliced, moved out from the center, how many pixels
	 * should it be moved?
	 * <p> Default: 10
	 */
	public Number getSlicedOffset() {
		return getAttr(Attrs.slicedOffset, 10).asNumber();
	}

	/**
	 * Sets if a point is sliced, moved out from the center, how many pixels
	 * should it be moved?
	 */
	public void setSlicedOffset(Number slicedOffset) {
		setAttr(Attrs.slicedOffset, slicedOffset, 10);
	}

	/**
	 * Returns the start angle of the pie slices in degrees where 0 is top and
	 * 90 right.
	 * <p> Default: 0
	 */
	public Number getStartAngle() {
		return getAttr(Attrs.startAngle, 0).asNumber();
	}

	/**
	 * Sets the start angle of the pie slices in degrees where 0 is top and 90
	 * right.
	 */
	public void setStartAngle(Number startAngle) {
		setAttr(Attrs.startAngle, startAngle, 0);
	}

	/**
	 * Returns sticky tracking of mouse events. When true, the
	 * <code>mouseOut</code> event on a series isn't triggered until the mouse
	 * moves over another series, or out of the plot area. When false, the
	 * <code>mouseOut</code> event on a series is triggered when the mouse
	 * leaves the area around the series' graph or markers. This also implies
	 * the tooltip. When <code>stickyTracking</code> is false and
	 * <code>tooltip.shared</code> is false, the tooltip will be hidden when
	 * moving the mouse between series.
	 * <p> Default: false
	 */
	public boolean isStickyTracking() {
		return getAttr(Attrs.stickyTracking, false).asBoolean();
	}

	/**
	 * Sets sticky tracking of mouse events. When true, the
	 * <code>mouseOut</code> event on a series isn't triggered until the mouse
	 * moves over another series, or out of the plot area. When false, the
	 * <code>mouseOut</code> event on a series is triggered when the mouse
	 * leaves the area around the series' graph or markers. This also implies
	 * the tooltip. When <code>stickyTracking</code> is false and
	 * <code>tooltip.shared</code> is false, the tooltip will be hidden when
	 * moving the mouse between series.
	 */
	public void setStickyTracking(boolean stickyTracking) {
		setAttr(Attrs.stickyTracking, stickyTracking);
	}

}