/* DataLabels.java

	Purpose:
		
	Description:
		
	History:
		Tue, Jan 14, 2014 11:36:32 PM, Created by RaymondChao

Copyright (C) 2014 Potix Corporation. All Rights Reserved.

 */
package org.zkoss.chart.plotOptions;

import java.util.LinkedHashMap;
import java.util.Map;

import org.zkoss.chart.Color;
import org.zkoss.chart.LinearGradient;
import org.zkoss.chart.Optionable;
import org.zkoss.chart.PlotAttribute;
import org.zkoss.chart.RadialGradient;
import org.zkoss.chart.Series;
import org.zkoss.chart.Shadow;
import org.zkoss.chart.util.DynamicalAttribute;
import org.zkoss.chart.util.MapsHelper;
import org.zkoss.lang.Generics;
import org.zkoss.util.Maps;

/**
 * An option of data labels used for {@link Series}
 * <p>
 * All the options in this class support {@link DynamicalAttribute}.
 * 
 * @author RaymondChao
 * @author jumperchen
 */
public class DataLabels extends Optionable {
	private enum Attrs implements PlotAttribute, DynamicalAttribute {
		align,
		backgroundColor,
		borderColor,
		borderRadius,
		borderWidth,
		color,
		crop,
		distance,
		enabled,
		format,
		formatter,
		inside,
		overflow,
		padding,
		rotation,
		shadow,
		softConnector,
		style,
		useHTML,
		verticalAlign,
		x,
		y,
		zIndex
	}

	/**
	 * Returns the alignment of the data label compared to the point. Can be one
	 * of "left", "center" or "right".
	 * <p>
	 * Defaults to <code>"center"</code>.
	 */
	public String getAlign() {
		return getAttr(Attrs.align, "center").asString();
	}

	/**
	 * Sets the alignment of the data label compared to the point. Can be one of
	 * "left", "center" or "right". Defaults to <code>"center"</code>. 'align
	 * the alignment of the data label compared to the point. Can be one of
	 * "left", "center" or "right".
	 * <p>
	 * Defaults to <code>"center"</code>.
	 */
	public void setAlign(String align) {
		if (!("left".equals(align) || "center".equals(align) || "right"
				.equals(align)))
			throw new IllegalArgumentException("Unsupported align: [" + align
					+ "]");
		setAttr(Attrs.align, align, "center");
	}
	/**
	 * Returns the background color for the data label.
	 * <p>
	 * Defaults to null.
	 */
	public Color getBackgroundColor() {
		return (Color) getAttr(Attrs.backgroundColor, null).asValue();
	}

	/**
	 * Sets the background color for the data label.
	 * <p>
	 * Defaults to null.
	 */
	public void setBackgroundColor(Color color) {
		setAttr(Attrs.backgroundColor, color);
	}

	/**
	 * Sets the background color for the data label.
	 * <p>
	 * Defaults to null.
	 */
	public void setBackgroundColor(String color) {
		setBackgroundColor(new Color(color));
	}

	/**
	 * Sets the background color for the data label.
	 * <p>
	 * Defaults to null.
	 */
	public void setBackgroundColor(LinearGradient color) {
		setBackgroundColor(new Color(color));
	}

	/**
	 * Sets the background color for the data label.
	 * <p>
	 * Defaults to null.
	 */
	public void setBackgroundColor(RadialGradient color) {
		setBackgroundColor(new Color(color));
	}

	/**
	 * Returns the border color for the data label.
	 * <p>
	 * Default: null
	 */
	public Color getBorderColor() {
		return (Color) getAttr(Attrs.borderColor, null).asValue();
	}

	/**
	 * Sets the border color for the data label.
	 * <p>
	 * Default: null
	 */
	public void setBorderColor(Color color) {
		setAttr(Attrs.borderColor, color);
	}

	/**
	 * Sets the border color for the data label.
	 * <p>
	 * Default: null
	 */
	public void setBorderColor(String color) {
		setBorderColor(new Color(color));
	}

	/**
	 * Sets the border color for the data label.
	 * <p>
	 * Default: null
	 */
	public void setBorderColor(LinearGradient color) {
		setBorderColor(new Color(color));
	}

	/**
	 * Sets the border color for the data label.
	 * <p>
	 * Default: null
	 */
	public void setBorderColor(RadialGradient color) {
		setBorderColor(new Color(color));
	}

	/**
	 * Returns the border radius in pixels for the data label.
	 * <p>
	 * Default: 0
	 */
	public Number getBorderRadius() {
		return getAttr(Attrs.borderRadius, 0).asNumber();
	}

	/**
	 * Sets the border radius in pixels for the data label.
	 */
	public void setBorderRadius(Number borderRadius) {
		setAttr(Attrs.borderRadius, borderRadius, 0);
	}

	/**
	 * Returns the border width in pixels for the data label.
	 * <p>
	 * Default: 0
	 */
	public Number getBorderWidth() {
		return getAttr(Attrs.borderWidth, 0).asNumber();
	}

	/**
	 * Sets the border width in pixels for the data label.
	 * <p>
	 * Default: 0
	 */
	public void setBorderWidth(Number borderWidth) {
		setAttr(Attrs.borderWidth, borderWidth, 0);
	}

	/**
	 * Returns the text color for the data labels.
	 * <p>
	 * Default: null
	 */
	public Color getColor() {
		return (Color) getAttr(Attrs.color, null).asValue();
	}

	/**
	 * Sets the text color for the data labels.
	 */
	public void setColor(Color color) {
		setAttr(Attrs.color, color);
	}

	/**
	 * Sets the text color for the data labels.
	 */
	public void setColor(String color) {
		setColor(new Color(color));
	}

	/**
	 * Sets the text color for the data labels.
	 */
	public void setColor(LinearGradient color) {
		setColor(new Color(color));
	}

	/**
	 * Sets the text color for the data labels.
	 */
	public void setColor(RadialGradient color) {
		setColor(new Color(color));
	}

	/**
	 * Returns whether to hide data labels that are outside the plot area. By
	 * default, the data label is moved inside the plot area according to the
	 * {@link #setOverflow(String)} option.
	 * <p>
	 * Default: true
	 */
	public boolean isCrop() {
		return getAttr(Attrs.crop, true).asBoolean();
	}

	/**
	 * Sets whether to hide data labels that are outside the plot area. By
	 * default, the data label is moved inside the plot area according to the
	 * {@link #setOverflow(String)} option.
	 * <p>
	 * Default: true
	 */
	public void setCrop(boolean crop) {
		setAttr(Attrs.crop, crop);
	}

	/**
	 * Returns the distance of the data labels which is used for Pie chart only.
	 * <p>
	 * Default: 0
	 */
	public Number getDistance() {
		return getAttr(Attrs.distance, 0).asNumber();
	}

	/**
	 * Sets the distance of the data labels which is used for Pie chart only.
	 */
	public void setDistance(Number distance) {
		setAttr(Attrs.distance, distance, 0);
	}

	/**
	 * Returns enable or disable the data labels.
	 * <p>
	 * Default: false
	 */
	public boolean isEnabled() {
		return getAttr(Attrs.enabled, false).asBoolean();
	}

	/**
	 * Sets enable or disable the data labels.
	 * <p>
	 * Default: false
	 */
	public void setEnabled(boolean enabled) {
		setAttr(Attrs.enabled, enabled);
	}

	/**
	 * Returns a <a href="http://docs.highcharts.com/#formatting">format
	 * string</a> for the data label.
	 */
	public String getFormat() {
		return getAttr(Attrs.format, "{y}").asString();
	}

	/**
	 * Sets a <a href="http://docs.highcharts.com/#formatting">format string</a>
	 * for the data label.
	 */
	public void setFormat(String format) {
		setAttr(Attrs.format, format, "{y}");
	}

	/**
	 * Returns for points with an extent, like columns, whether to align the
	 * data label inside the box or to the actual value point.
	 * <p>
	 * Defaults to <code>false</code> in most cases, <code>true</code> in
	 * stacked columns.
	 */
	public boolean isInside() {
		return getAttr(Attrs.inside, false).asBoolean();
	}

	/**
	 * Sets for points with an extent, like columns, whether to align the data
	 * label inside the box or to the actual value point. Defaults to
	 * <code>false</code> in most cases, <code>true</code> in stacked columns.
	 * 'inside for points with an extent, like columns, whether to align the
	 * data label inside the box or to the actual value point.
	 * <p>
	 * Defaults to <code>false</code> in most cases, <code>true</code> in
	 * stacked columns.
	 */
	public void setInside(boolean inside) {
		setAttr(Attrs.inside, inside);
	}

	/**
	 * Returns how to handle data labels that flow outside the plot area.
	 * <p>
	 * The default is <code>justify</code>, which aligns them inside the plot
	 * area. For columns and bars, this means it will be moved inside the bar.
	 * To display data labels outside the plot area, set <code>crop</code> to
	 * <code>false</code> and <code>overflow</code> to <code>"none"</code>.
	 */
	public String getOverflow() {
		return getAttr(Attrs.overflow, "justify").asString();
	}

	/**
	 * Returns how to handle data labels that flow outside the plot area.
	 * <p>
	 * The default is <code>justify</code>, which aligns them inside the plot
	 * area. For columns and bars, this means it will be moved inside the bar.
	 * To display data labels outside the plot area, set
	 * {@link #setCrop(boolean)} to <code>false</code> and
	 * {@link #setOverflow(String)} to <code>"none"</code>.
	 */
	public void setOverflow(String overflow) {
		setAttr(Attrs.overflow, overflow, "justify");
	}

	/**
	 * Returns when either the {@link #getBorderWidth()} or the
	 * {@link #getBackgroundColor()} is set, this is the padding within the box.
	 * <p>
	 * Default: 2
	 */
	public Number getPadding() {
		return getAttr(Attrs.padding, 2).asNumber();
	}

	/**
	 * Sets when either the {@link #getBorderWidth()} or the
	 * {@link #getBackgroundColor()} is set, this is the padding within the box.
	 */
	public void setPadding(int padding) {
		setAttr(Attrs.padding, padding, 2);
	}

	/**
	 * Returns text rotation in degrees. Note that due to a more complex
	 * structure, backgrounds and borders will be lost on a rotated data label.
	 * <p>
	 * Default: 0
	 */
	public Object getRotation() {
		return getAttr(Attrs.rotation, 0).asValue();
	}

	/**
	 * Sets text rotation in degrees. Note that due to a more complex structure,
	 * backgrounds and borders will be lost on a rotated data label.
	 */
	public void setRotation(Number rotation) {
		setAttr(Attrs.rotation, rotation, 0);
	}

	/**
	 * Sets text rotation in degrees. Note that due to a more complex structure,
	 * backgrounds and borders will be lost on a rotated data label.
	 */
	public void setRotation(String rotation) {
		setAttr(Attrs.rotation, rotation);
	}

	/**
	 * Returns whether to apply a drop shadow to the data label area. Works best
	 * with borderWidth or backgroundColor.
	 * <p>
	 * Default: false.
	 * 
	 * @see Shadow
	 */
	public boolean isShadow() {
		return getShadow() != Shadow.NONE;
	}

	/**
	 * Returns whether to apply a drop shadow to the outer chart area. Works
	 * best with borderWidth or backgroundColor.
	 * <p>
	 * Default: {@link Shadow#NONE}.
	 * 
	 * @see Shadow
	 */
	public Shadow getShadow() {
		return (Shadow) getAttr(Attrs.shadow, Shadow.NONE).asValue();
	}

	/**
	 * Sets whether to apply a drop shadow to the outer chart area. Works best
	 * with borderWidth or backgroundColor.
	 */
	public void setShadow(boolean shadow) {
		setShadow(shadow ? new Shadow() : Shadow.NONE);
	}

	/**
	 * Sets whether to apply a drop shadow to the outer chart area. Works best
	 * with borderWidth or backgroundColor.
	 * 
	 * @see Shadow
	 */
	public void setShadow(Shadow shadow) {
		setAttr(Attrs.shadow, shadow);
	}

	/**
	 * Returns whether to enable soft connector. Funnel chart only.
	 */
	public boolean isSoftConnector() {
		return getAttr(Attrs.softConnector, false).asBoolean();
	}

	/**
	 * Sets to enable soft connector. Funnel chart only.
	 */
	public void setSoftConnector(boolean softConnector) {
		setAttr(Attrs.softConnector, softConnector);
	}

	/**
	 * Returns styles for the label.
	 */
	public <K, V> Map<K, V> getStyle() {
		return Generics.cast(getAttr(Attrs.style, null).asValue());
	}

	/**
	 * Sets styles for the label.
	 */
	public void setStyle(String style) {
		setStyle(MapsHelper.parse(new LinkedHashMap<String, String>(), style, ':', ';', '\''));
	}

	/**
	 * Sets styles for the label.
	 */
	public <K, V> void setStyle(Map<K, V> style) {
		setAttr(Attrs.style, style);
	}

	/**
	 * Return whether to use HTML to render the labels.
	 * <p>
	 * Defaults to false.
	 */
	public boolean isUseHTML() {
		return getAttr(Attrs.useHTML, false).asBoolean();
	}

	/**
	 * Sets whether to use HTML to render the labels.
	 * <p>
	 * Defaults to false.
	 */
	public void setUseHTML(boolean useHTML) {
		setAttr(Attrs.useHTML, useHTML);
	}

	/**
	 * Returns the vertical alignment of a data label. Can be one of
	 * <code>top</code>, <code>middle</code> or <code>bottom</code>.
	 * <p>
	 * The default value depends on the data, for instance in a column chart,
	 * the label is above positive values and below negative values.
	 */
	public String getVerticalAlign() {
		return getAttr(Attrs.verticalAlign, null).asString();
	}

	/**
	 * Sets the vertical alignment of a data label. Can be one of
	 * <code>top</code>, <code>middle</code> or <code>bottom</code>.
	 * <p>
	 * The default value depends on the data, for instance in a column chart,
	 * the label is above positive values and below negative values.
	 */
	public void setVerticalAlign(String verticalAlign) {
		if (!("top".equals(verticalAlign) || "middle".equals(verticalAlign) || "bottom"
				.equals(verticalAlign)))
			throw new IllegalArgumentException("Unsupported vertical align: ["
					+ verticalAlign + "]");
		setAttr(Attrs.verticalAlign, verticalAlign);
	}

	/**
	 * Returns the x position offset of the label relative to the point.
	 * <p>
	 * Default: 0
	 */
	public Number getX() {
		return getAttr(Attrs.x, 0).asNumber();
	}

	/**
	 * Sets the x position offset of the label relative to the point.
	 * <p>
	 * Default: 0
	 */
	public void setX(Number x) {
		setAttr(Attrs.x, x, 0);
	}

	/**
	 * Returns the y position offset of the label relative to the point.
	 * <p>
	 * Default: -6
	 */
	public Number getY() {
		return getAttr(Attrs.y, -6).asNumber();
	}

	/**
	 * Sets the y position offset of the label relative to the point.
	 * <p>
	 * Default: -6
	 */
	public void setY(Number y) {
		setAttr(Attrs.y, y, NOT_NULL_VALUE);
	}

	/**
	 * Returns the Z index of the data labels.
	 * <p>
	 * The default Z index puts it above the series. Use a Z index of 2 to
	 * display it behind the series.
	 * <p> Default: 6
	 */
	public Number getZIndex() {
		return getAttr(Attrs.zIndex, 6).asNumber();
	}

	/**
	 * Sets the Z index of the data labels.
	 * <p>
	 * The default Z index puts it above the series. Use a Z index of 2 to
	 * display it behind the series.
	 * <p> Default: 6
	 */
	public void setZIndex(Number zIndex) {
		setAttr(Attrs.zIndex, zIndex, 6);
	}

}