/* ColumnPlotOptions.java

	Purpose:
		
	Description:
		
	History:
		Thu, Jan 16, 2014  3:58:04 PM, Created by RaymondChao

Copyright (C) 2014 Potix Corporation. All Rights Reserved.

 */
package org.zkoss.chart.plotOptions;

import java.util.Arrays;
import java.util.List;

import org.zkoss.chart.Charts;
import org.zkoss.chart.Color;
import org.zkoss.chart.LinearGradient;
import org.zkoss.chart.PlotAttribute;
import org.zkoss.chart.RadialGradient;
import org.zkoss.chart.util.DynamicalAttribute;
import org.zkoss.lang.Generics;

/**
 * An option for column plot
 * <p>
 * All the options in this class support {@link DynamicalAttribute}.
 * 
 * @author RaymondChao
 * @author jumperchen
 */
public class ColumnPlotOptions extends SeriesPlotOptions {

	private enum Attrs implements PlotAttribute, DynamicalAttribute {
		borderColor,
		borderRadius,
		borderWidth,
		colorByPoint,
		colors,
		groupPadding,
		grouping,
		minPointLength,
		pointPadding,
		pointRange,
		pointWidth,
		threshold,
		cropThreshold
	}

	/**
	 * Returns when the series contains less points than the crop threshold, all
	 * points are drawn, even if the points fall outside the visible plot area
	 * at the current zoom. The advantage of drawing all points (including
	 * markers and columns), is that animation is performed on updates. On the
	 * other hand, when the series contains more points than the crop threshold,
	 * the series data is cropped to only contain points that fall within the
	 * plot area. The advantage of cropping away invisible points is to increase
	 * performance on large series.
	 * <p>
	 * Default: 50
	 */
	public Number getCropThreshold() {
		return getAttr(Attrs.cropThreshold, 50).asNumber();
	}

	/**
	 * Sets when the series contains less points than the crop threshold, all
	 * points are drawn, even if the points fall outside the visible plot area
	 * at the current zoom. The advantage of drawing all points (including
	 * markers and columns), is that animation is performed on updates. On the
	 * other hand, when the series contains more points than the crop threshold,
	 * the series data is cropped to only contain points that fall within the
	 * plot area. The advantage of cropping away invisible points is to increase
	 * performance on large series.
	 */
	public void setCropThreshold(Number cropThreshold) {
		setAttr(Attrs.cropThreshold, cropThreshold, 50);
	}

	/**
	 * Returns the color of the border surrounding each column or bar.
	 * <p>
	 * Default: #FFFFFF
	 */
	public Color getBorderColor() {
		if (!containsKey(Attrs.borderColor)) {
			setBorderColor("#FFFFFF");
		}
		return (Color) getAttr(Attrs.borderColor);
	}

	/**
	 * Sets the color of the border surrounding each column or bar. 'borderColor
	 * the color of the border surrounding each column or bar.
	 */
	public void setBorderColor(Color color) {
		setAttr(Attrs.borderColor, color, NOT_NULL_VALUE);
	}

	/**
	 * Sets the color of the border surrounding each column or bar. 'borderColor
	 * the color of the border surrounding each column or bar.
	 */
	public void setBorderColor(String color) {
		setBorderColor(new Color(color));
	}

	/**
	 * Sets the color of the border surrounding each column or bar. 'borderColor
	 * the color of the border surrounding each column or bar.
	 */
	public void setBorderColor(LinearGradient color) {
		setBorderColor(new Color(color));
	}

	/**
	 * Sets the color of the border surrounding each column or bar. 'borderColor
	 * the color of the border surrounding each column or bar.
	 */
	public void setBorderColor(RadialGradient color) {
		setBorderColor(new Color(color));
	}

	/**
	 * Returns the corner radius of the border surrounding each column or bar.
	 * <p>
	 * Default: 0
	 */
	public Number getBorderRadius() {
		return getAttr(Attrs.borderRadius, 0).asNumber();
	}

	/**
	 * Sets the corner radius of the border surronding each column or bar.
	 */
	public void setBorderRadius(Number borderRadius) {
		setAttr(Attrs.borderRadius, borderRadius, 0);
	}

	/**
	 * Returns the width of the border surrounding each column or bar.
	 * <p>
	 * Default: 1
	 */
	public Number getBorderWidth() {
		return getAttr(Attrs.borderWidth, 1).asNumber();
	}

	/**
	 * Sets the width of the border surrounding each column or bar.
	 */
	public void setBorderWidth(Number borderWidth) {
		setAttr(Attrs.borderWidth, borderWidth, 1);
	}

	/**
	 * Returns when using automatic point colors pulled from the
	 * {@link Charts#getColors()} collection, this option determines whether the
	 * chart should receive one color per series or one color per point.
	 */
	public boolean isColorByPoint() {
		return getAttr(Attrs.colorByPoint, false).asBoolean();
	}

	/**
	 * Sets when using automatic point colors pulled from the
	 * {@link Charts#getColors()} collection, this option determines whether the
	 * chart should receive one color per series or one color per point.
	 */
	public void setColorByPoint(boolean colorByPoint) {
		setAttr(Attrs.colorByPoint, colorByPoint);
	}

	/**
	 * Returns a series specific or series type specific color set to apply
	 * instead of the global {@link Charts#getColors()} when
	 * {@link #setColorByPoint(boolean)} is true.
	 */
	public List<Color> getColors() {
		return Generics.cast(getAttr(Attrs.colors, null).asValue());
	}

	/**
	 * Sets a series specific or series type specific color set to apply
	 * instead of the global {@link Charts#getColors()} when
	 * {@link #setColorByPoint(boolean)} is true.
	 */
	public void setColors(List<Color> colors) {
		setAttr(Attrs.colors, colors);
	}

	/**
	 * Sets a series specific or series type specific color set to apply
	 * instead of the global {@link Charts#getColors()} when
	 * {@link #setColorByPoint(boolean)} is true.
	 */
	public void setColors(String... source) {
		Color[] colors = new Color[source.length];
		int i = 0;
		for (String s : source)
			colors[i++] = new Color(s);
		setColors(Arrays.asList(colors));
	}

	/**
	 * Returns padding between each value groups, in x axis units.
	 * <p>Default: 0.2
	 */
	public Number getGroupPadding() {
		return getAttr(Attrs.groupPadding, 0.2).asNumber();
	}

	/**
	 * Sets padding between each value groups, in x axis units.
	 */
	public void setGroupPadding(Number groupPadding) {
		setAttr(Attrs.groupPadding, groupPadding, 0.2);
	}

	/**
	 * Returns whether to group non-stacked columns or to let them render
	 * independent of each other. Non-grouped columns will be laid out
	 * individually and overlap each other.
	 */
	public boolean isGrouping() {
		return getAttr(Attrs.grouping, true).asBoolean();
	}

	/**
	 * Sets whether to group non-stacked columns or to let them render
	 * independent of each other. Non-grouped columns will be laid out
	 * individually and overlap each other.
	 */
	public void setGrouping(boolean grouping) {
		setAttr(Attrs.grouping, grouping);
	}

	/**
	 * Returns the minimal height for a column or width for a bar. By default, 0
	 * values are not shown. To visualize a 0 (or close to zero) point, set the
	 * minimal point length to a pixel value like 3. In stacked column charts,
	 * minPointLength might not be respected for tightly packed values.
	 * <p>Default: 0
	 */
	public Number getMinPointLength() {
		return getAttr(Attrs.minPointLength, 0).asNumber();
	}

	/**
	 * Sets the minimal height for a column or width for a bar. By default, 0
	 * values are not shown. To visualize a 0 (or close to zero) point, set the
	 * minimal point length to a pixel value like 3. In stacked column charts,
	 * minPointLength might not be respected for tightly packed values.
	 */
	public void setMinPointLength(Number minPointLength) {
		setAttr(Attrs.minPointLength, minPointLength, 0);
	}

	/**
	 * Returns padding between each column or bar, in x axis units.
	 * <p> Default: 0.1
	 */
	public Number getPointPadding() {
		return getAttr(Attrs.pointPadding, 0.1).asNumber();
	}

	/**
	 * Sets padding between each column or bar, in x axis units. 'pointPadding
	 * padding between each column or bar, in x axis units.
	 */
	public void setPointPadding(Number pointPadding) {
		setAttr(Attrs.pointPadding, pointPadding, 0.1);
	}

	/**
	 * Returns the X axis range that each point is valid for. This determines
	 * the width of the column. On a categorized axis, the range will be 1 by
	 * default (one category unit). On linear and datetime axes, the range will
	 * be computed as the distance between the two closest data points.
	 * <p> Default: null
	 */
	public Number getPointRange() {
		return getAttr(Attrs.pointRange, null).asNumber();
	}

	/**
	 * Sets the X axis range that each point is valid for. This determines the
	 * width of the column. On a categorized axis, the range will be 1 by
	 * default (one category unit). On linear and datetime axes, the range will
	 * be computed as the distance between the two closest data points.
	 */
	public void setPointRange(Number pointRange) {
		setAttr(Attrs.pointRange, pointRange);
	}

	/**
	 * Returns a pixel value specifying a fixed width for each column or bar.
	 * When <code>null</code>, the width is calculated from the
	 * <code>pointPadding</code> and <code>groupPadding</code>.
	 * <p>Default: null
	 */
	public Number getPointWidth() {
		return getAttr(Attrs.pointWidth, null).asNumber();
	}

	/**
	 * Sets a pixel value specifying a fixed width for each column or bar. When
	 * <code>null</code>, the width is calculated from the
	 * <code>pointPadding</code> and <code>groupPadding</code>.
	 */
	public void setPointWidth(int pointWidth) {
		setAttr(Attrs.pointWidth, pointWidth);
	}
}