/* BubblePlotOptions.java

	Purpose:
		
	Description:
		
	History:
		Thu, Jan 16, 2014  3:50:30 PM, Created by RaymondChao

Copyright (C) 2014 Potix Corporation. All Rights Reserved.

 */
package org.zkoss.chart.plotOptions;

import org.zkoss.chart.Color;
import org.zkoss.chart.LinearGradient;
import org.zkoss.chart.PlotAttribute;
import org.zkoss.chart.RadialGradient;
import org.zkoss.chart.util.DynamicalAttribute;

/**
 * An option for bubble plot.
 * <p>
 * All the options in this class support {@link DynamicalAttribute}.
 * 
 * @author RaymondChao
 * @author jumperchen
 */
public class BubblePlotOptions extends ColumnPlotOptions {
	private enum Attrs implements PlotAttribute, DynamicalAttribute {
		displayNegative,
		maxSize,
		minSize,
		negativeColor,
		sizeBy,
		zThreshold,
		cropThreshold
	}

	/**
	 * Returns when the series contains less points than the crop threshold, all
	 * points are drawn, even if the points fall outside the visible plot area
	 * at the current zoom. The advantage of drawing all points (including
	 * markers and columns), is that animation is performed on updates. On the
	 * other hand, when the series contains more points than the crop threshold,
	 * the series data is cropped to only contain points that fall within the
	 * plot area. The advantage of cropping away invisible points is to increase
	 * performance on large series.
	 * <p>
	 * Default: 300
	 */
	public Number getCropThreshold() {
		return getAttr(Attrs.cropThreshold, 300).asNumber();
	}

	/**
	 * Sets when the series contains less points than the crop threshold, all
	 * points are drawn, even if the points fall outside the visible plot area
	 * at the current zoom. The advantage of drawing all points (including
	 * markers and columns), is that animation is performed on updates. On the
	 * other hand, when the series contains more points than the crop threshold,
	 * the series data is cropped to only contain points that fall within the
	 * plot area. The advantage of cropping away invisible points is to increase
	 * performance on large series.
	 */
	public void setCropThreshold(Number cropThreshold) {
		setAttr(Attrs.cropThreshold, cropThreshold, 300);
	}

	/**
	 * Returns whether to display negative sized bubbles. The threshold is given
	 * by the {@link #setZThreshold(Number)} option, and negative bubbles can be
	 * visualized by setting {@link #setNegativeColor(String)}
	 * <p>
	 * Default: true
	 */
	public boolean isDisplayNegative() {
		return getAttr(Attrs.displayNegative, true).asBoolean();
	}

	/**
	 * Sets whether to display negative sized bubbles. The threshold is given by
	 * the {@link #setZThreshold(Number)} option, and negative bubbles can be
	 * visualized by setting {@link #setNegativeColor(String)}
	 * <p>
	 * Default: true
	 */
	public void setDisplayNegative(boolean displayNegative) {
		setAttr(Attrs.displayNegative, displayNegative);
	}

	/**
	 * Returns maximum bubble size. Bubbles will automatically size between the
	 * <code>minSize</code> and <code>maxSize</code> to reflect the
	 * <code>z</code> value of each bubble. Can be either pixels (when no unit
	 * is given), or a percentage of the smallest one of the plot width and
	 * height.
	 * <p>
	 * Default: 20%
	 */
	public String getMaxSize() {
		return getAttr(Attrs.maxSize, "20%").asString();
	}

	/**
	 * Sets maximum bubble size. Bubbles will automatically size between the
	 * <code>minSize</code> and <code>maxSize</code> to reflect the
	 * <code>z</code> value of each bubble. Can be either pixels (when no unit
	 * is given), or a percentage of the smallest one of the plot width and
	 * height.
	 */
	public void setMaxSize(String maxSize) {
		setAttr(Attrs.maxSize, maxSize, "20%");
	}

	/**
	 * Returns minimum bubble size. Bubbles will automatically size between the
	 * <code>minSize</code> and <code>maxSize</code> to reflect the
	 * <code>z</code> value of each bubble. Can be either pixels (when no unit
	 * is given), or a percentage of the smallest one of the plot width and
	 * height.
	 * <p>
	 * Default: 8
	 */
	public String getMinSize() {
		return getAttr(Attrs.minSize, 8).asString();
	}

	/**
	 * Sets minimum bubble size. Bubbles will automatically size between the
	 * <code>minSize</code> and <code>maxSize</code> to reflect the
	 * <code>z</code> value of each bubble. Can be either pixels (when no unit
	 * is given), or a percentage of the smallest one of the plot width and
	 * height.
	 */
	public void setMinSize(String minSize) {
		setAttr(Attrs.minSize, minSize, 8);
	}

	/**
	 * Returns when a point's Z value is below the
	 * {@link #setZThreshold(Number)} setting, this color is used.
	 * <p>
	 * Default: null
	 */
	public Color getNegativeColor() {
		return (Color) getAttr(Attrs.negativeColor, null).asValue();
	}

	/**
	 * Sets when a point's Z value is below the {@link #setZThreshold(Number)}
	 * setting, this color is used.
	 */
	public void setNegativeColor(Color color) {
		setAttr(Attrs.negativeColor, color);
	}

	/**
	 * Sets when a point's Z value is below the {@link #setZThreshold(Number)}
	 * setting, this color is used.
	 */
	public void setNegativeColor(String color) {
		setNegativeColor(new Color(color));
	}

	/**
	 * Sets when a point's Z value is below the {@link #setZThreshold(Number)}
	 * setting, this color is used.
	 */
	public void setNegativeColor(LinearGradient color) {
		setNegativeColor(new Color(color));
	}

	/**
	 * Sets when a point's Z value is below the {@link #setZThreshold(Number)}
	 * setting, this color is used.
	 */
	public void setNegativeColor(RadialGradient color) {
		setNegativeColor(new Color(color));
	}

	/**
	 * Returns whether the bubble's value should be represented by the area or
	 * the width of the bubble.
	 * <p>
	 * The default, <code>area</code>, corresponds best to the human perception
	 * of the size of each bubble.
	 */
	public String getSizeBy() {
		return getAttr(Attrs.sizeBy, "area").asString();
	}

	/**
	 * Sets whether the bubble's value should be represented by the area or the
	 * width of the bubble. The default, <code>area</code>, corresponds best to
	 * the human perception of the size of each bubble.
	 */
	public void setSizeBy(String sizeBy) {
		setAttr(Attrs.sizeBy, sizeBy, "area");
	}

	/**
	 * Returns when {@link #setDisplayNegative(boolean)} is <code>false</code>,
	 * bubbles with lower Z values are skipped. When
	 * <code>displayNegative</code> is <code>true</code> and a
	 * {@link #setNegativeColor(String)} is given, points with lower Z is
	 * colored.
	 * <p>
	 * Default: 0
	 */
	public Number getZThreshold() {
		return getAttr(Attrs.zThreshold, 0).asNumber();
	}

	/**
	 * Sets when {@link #setDisplayNegative(boolean)} is <code>false</code>,
	 * bubbles with lower Z values are skipped. When
	 * <code>displayNegative</code> is <code>true</code> and a
	 * {@link #setNegativeColor(String)} is given, points with lower Z is
	 * colored.
	 */
	public void setZThreshold(Number zThreshold) {
		setAttr(Attrs.zThreshold, zThreshold, 0);
	}

}