/* BoxPlotOptions.java

	Purpose:
		
	Description:
		
	History:
		Thu, Jan 16, 2014  3:55:39 PM, Created by RaymondChao

Copyright (C) 2014 Potix Corporation. All Rights Reserved.

 */
package org.zkoss.chart.plotOptions;

import org.zkoss.chart.Color;
import org.zkoss.chart.LinearGradient;
import org.zkoss.chart.PlotAttribute;
import org.zkoss.chart.RadialGradient;
import org.zkoss.chart.util.DynamicalAttribute;

/**
 * An option for box plot
 * <p>
 * All the options in this class support {@link DynamicalAttribute}.
 * 
 * @author RaymondChao
 */
public class BoxPlotOptions extends ColumnPlotOptions {
	private enum Attrs implements PlotAttribute, DynamicalAttribute {
		fillColor,
		lineWidth,
		medianColor,
		medianWidth,
		stemColor,
		stemDashStyle,
		stemWidth,
		whiskerColor,
		whiskerLength,
		whiskerWidth
	}

	/**
	 * Returns the fill color of the box.
	 * <p>
	 * Default: #FFFFFF
	 */
	public Color getFillColor() {
		if (!containsKey(Attrs.fillColor)) {
			setFillColor("#FFFFFF");
		}
		return (Color) getAttr(Attrs.fillColor);
	}

	/**
	 * Sets the fill color of the box.
	 */
	public void setFillColor(Color color) {
		setAttr(Attrs.fillColor, color, NOT_NULL_VALUE);
	}

	/**
	 * Sets the fill color of the box.
	 */
	public void setFillColor(String color) {
		setFillColor(new Color(color));
	}

	/**
	 * Sets the fill color of the box.
	 */
	public void setFillColor(LinearGradient color) {
		setFillColor(new Color(color));
	}

	/**
	 * Sets the fill color of the box.
	 */
	public void setFillColor(RadialGradient color) {
		setFillColor(new Color(color));
	}

	/**
	 * Returns the width of the line surrounding the box. If any of
	 * {@link #setStemWidth(Number)} , {@link #setMedianWidth(Number)}, or
	 * {@link #setWhiskerWidth(Number)} are <code>null</code>, the lineWidth
	 * also applies to these lines.
	 */
	public Number getLineWidth() {
		return getAttr(Attrs.lineWidth, 1).asNumber();
	}

	/**
	 * Sets the width of the line surrounding the box. If any of
	 * {@link #setStemWidth(Number)} , {@link #setMedianWidth(Number)}, or
	 * {@link #setWhiskerWidth(Number)} are <code>null</code>, the lineWidth
	 * also applies to these lines.
	 */
	public void setLineWidth(Number lineWidth) {
		setAttr(Attrs.lineWidth, lineWidth, 1);
	}

	/**
	 * Returns the color of the median line. If <code>null</code>, the general
	 * series color applies.
	 */
	public String getMedianColor() {
		return getAttr(Attrs.medianColor, null).asString();
	}

	/**
	 * Sets the color of the median line. If <code>null</code>, the general
	 * series color applies. 'medianColor the color of the median line. If
	 * <code>null</code>, the general series color applies.
	 */
	public void setMedianColor(String medianColor) {
		setAttr(Attrs.medianColor, medianColor);
	}

	/**
	 * Returns the pixel width of the median line. If <code>null</code>, the
	 * {@link #setLineWidth(Number)} is used.
	 * <p>
	 * Default: 2
	 */
	public Number getMedianWidth() {
		return getAttr(Attrs.medianWidth, 2).asNumber();
	}

	/**
	 * Sets the pixel width of the median line. If <code>null</code>, the
	 * {@link #setLineWidth(Number)} is used.
	 * <p>
	 * Default: 2
	 */
	public void setMedianWidth(Number medianWidth) {
		setAttr(Attrs.medianWidth, medianWidth, 2);
	}

	/**
	 * Returns the color of the stem, the vertical line extending from the box
	 * to the whiskers. If <code>null</code>, the series color is used.
	 */
	public Color getStemColor() {
		return (Color) getAttr(Attrs.stemColor, null).asValue();
	}

	/**
	 * Sets the color of the stem, the vertical line extending from the box to
	 * the whiskers. If <code>null</code>, the series color is used.
	 */
	public void setStemColor(Color color) {
		setAttr(Attrs.stemColor, color);
	}

	/**
	 * Sets the color of the stem, the vertical line extending from the box to
	 * the whiskers. If <code>null</code>, the series color is used.
	 */
	public void setStemColor(String color) {
		setStemColor(new Color(color));
	}

	/**
	 * Sets the color of the stem, the vertical line extending from the box to
	 * the whiskers. If <code>null</code>, the series color is used.
	 */
	public void setStemColor(LinearGradient color) {
		setStemColor(new Color(color));
	}

	/**
	 * Sets the color of the stem, the vertical line extending from the box to
	 * the whiskers. If <code>null</code>, the series color is used.
	 */
	public void setStemColor(RadialGradient color) {
		setStemColor(new Color(color));
	}

	/**
	 * Returns the dash style of the stem, the vertical line extending from the
	 * box to the whiskers.
	 */
	public String getStemDashStyle() {
		return getAttr(Attrs.stemDashStyle, "Solid").asString();
	}

	/**
	 * Sets the dash style of the stem, the vertical line extending from the box
	 * to the whiskers.
	 */
	public void setStemDashStyle(String stemDashStyle) {
		if (!("Solid".equalsIgnoreCase(stemDashStyle)
				|| "ShortDash".equalsIgnoreCase(stemDashStyle)
				|| "ShortDot".equalsIgnoreCase(stemDashStyle)
				|| "ShortDashDot".equalsIgnoreCase(stemDashStyle)
				|| "ShortDashDotDot".equalsIgnoreCase(stemDashStyle)
				|| "Dot".equalsIgnoreCase(stemDashStyle)
				|| "Dash".equalsIgnoreCase(stemDashStyle)
				|| "LongDash".equalsIgnoreCase(stemDashStyle)
				|| "DashDot".equalsIgnoreCase(stemDashStyle)
				|| "LongDashDot".equalsIgnoreCase(stemDashStyle) || "LongDashDotDot"
					.equalsIgnoreCase(stemDashStyle)))
			throw new IllegalArgumentException("Unsupported style: ["
					+ stemDashStyle + "]");
		setAttr(Attrs.stemDashStyle, stemDashStyle, "Solid");
	}

	/**
	 * Returns the width of the stem, the vertical line extending from the box
	 * to the whiskers. If <code>null</code>, the width is inherited from the
	 * {@link #setLineWidth(Number)} option.
	 */
	public Number getStemWidth() {
		return getAttr(Attrs.stemWidth, null).asNumber();
	}

	/**
	 * Sets the width of the stem, the vertical line extending from the box to
	 * the whiskers. If <code>null</code>, the width is inherited from the
	 * {@link #setLineWidth(Number)} option.
	 */
	public void setStemWidth(Number stemWidth) {
		setAttr(Attrs.stemWidth, stemWidth);
	}

	/**
	 * Returns the color of the whiskers, the horizontal lines marking low and
	 * high values. When <code>null</code>, the general series color is used.
	 * <p>
	 * Default: null
	 */
	public Color getWhiskerColor() {
		return (Color) getAttr(Attrs.whiskerColor, null).asValue();
	}

	/**
	 * Sets the color of the whiskers, the horizontal lines marking low and high
	 * values. When <code>null</code>, the general series color is used.
	 */
	public void setWhiskerColor(Color color) {
		setAttr(Attrs.whiskerColor, color);
	}

	/**
	 * Sets the color of the whiskers, the horizontal lines marking low and high
	 * values. When <code>null</code>, the general series color is used.
	 */
	public void setWhiskerColor(String color) {
		setWhiskerColor(new Color(color));
	}

	/**
	 * Sets the color of the whiskers, the horizontal lines marking low and high
	 * values. When <code>null</code>, the general series color is used.
	 */
	public void setWhiskerColor(LinearGradient color) {
		setWhiskerColor(new Color(color));
	}

	/**
	 * Sets the color of the whiskers, the horizontal lines marking low and high
	 * values. When <code>null</code>, the general series color is used.
	 */
	public void setWhiskerColor(RadialGradient color) {
		setWhiskerColor(new Color(color));
	}

	/**
	 * Returns the length of the whiskers, the horizontal lines marking low and
	 * high values. It can be a numerical pixel value, or a percentage value of
	 * the box width. Set <code>0</code> to disable whiskers.
	 * <p>
	 * Default: 50%
	 */
	public String getWhiskerLength() {
		return getAttr(Attrs.whiskerLength, "50%").asString();
	}

	/**
	 * Sets the length of the whiskers, the horizontal lines marking low and
	 * high values. It can be a numerical pixel value, or a percentage value of
	 * the box width. Set <code>0</code> to disable whiskers.
	 */
	public void setWhiskerLength(String whiskerLength) {
		setAttr(Attrs.whiskerLength, whiskerLength, "50%");
	}

	/**
	 * Returns the line width of the whiskers, the horizontal lines marking low
	 * and high values. When <code>null</code>, the general
	 * {@link #setLineWidth(Number)} applies.
	 */
	public Number getWhiskerWidth() {
		return getAttr(Attrs.whiskerWidth, 2).asNumber();
	}

	/**
	 * Sets the line width of the whiskers, the horizontal lines marking low and
	 * high values. When <code>null</code>, the general
	 * {@link #setLineWidth(Number)} applies.
	 */
	public void setWhiskerWidth(Number whiskerWidth) {
		setAttr(Attrs.whiskerWidth, whiskerWidth, 2);
	}

}