/** PlotEngineImpl.java.

	Purpose:
		
	Description:
		
	History:
		10:10:29 AM Jan 9, 2014, Created by jumperchen

Copyright (C) 2014 Potix Corporation. All Rights Reserved.
*/
package org.zkoss.chart.impl;

import java.awt.Color;
import java.awt.Font;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;

import org.zkoss.chart.AxisLabels;
import org.zkoss.chart.Charts;
import org.zkoss.chart.LinearGradient;
import org.zkoss.chart.Pane;
import org.zkoss.chart.PaneBackground;
import org.zkoss.chart.PlotBand;
import org.zkoss.chart.PlotData;
import org.zkoss.chart.PlotEngine;
import org.zkoss.chart.Series;
import org.zkoss.chart.YAxis;
import org.zkoss.chart.YAxisTitle;
import org.zkoss.chart.model.CategoryModel;
import org.zkoss.chart.model.DialModel;
import org.zkoss.chart.model.DialModelRange;
import org.zkoss.chart.model.DialModelScale;
import org.zkoss.chart.model.SingleValueCategoryModel;
import org.zkoss.chart.model.XYModel;
import org.zkoss.chart.model.XYZModel;
import org.zkoss.chart.plotOptions.DataLabels;
import org.zkoss.chart.plotOptions.GaugePivotPlotOptions;
import org.zkoss.chart.plotOptions.GaugePlotOptions;

/**
 * A skeleton implementation of chart model.
 * @author jumperchen
 *
 */
abstract public class PlotEngineImpl implements PlotEngine {
	protected Charts _chart;
	public PlotEngineImpl(Charts chart) {
		_chart = chart;
	}
	/**
	 * Returns the highcharts component that belongs to this chart model
	 */
	public Charts getChart() {
		return _chart;
	}
	
	/**
	 * Draws the chart with category model
	 */
	@SuppressWarnings({ "unchecked", "rawtypes" })
	protected PlotData drawCategoryModel(CategoryModel model) {
		PlotData plotData = getChart().getPlotData();
		int i = 0;
		for (Comparable<?> ser : model.getSeries()) {
			Series series = plotData.getSeries(i++);
			//JSONArrayHelper data = new JSONArrayHelper();
			for (Comparable<?> cat : model.getCategories()) {
				//data.add(model.getValue(ser, cat));
				series.addPoint(model.getValue(ser, cat));
			}
			series.setName(ser.toString());
			//series.setData(data);
		}
		if (!model.getCategories().isEmpty())
			plotData.getXAxis().setCategories(new LinkedList(model.getCategories()));
		return plotData;
	}
	
	/**
	 * Draws the chart with single value category model
	 */
	protected PlotData drawSingleValueCategoryModel(SingleValueCategoryModel model) {
		PlotData plotData = getChart().getPlotData();
		Series series = plotData.getSeries();
	//	JSONArrayHelper data = new JSONArrayHelper();
		for (Comparable<?> cate : model.getCategories()) {
			//data.addTuple(cate , model.getValue(cate));
			series.addPoint(cate.toString(), model.getValue(cate));
		}
		//series.setData(data);
		return plotData;
	}
	
	/**
	 * Draws the chart with dial model
	 */
	protected PlotData drawDialModel(DialModel model) {
		PlotData plotData = getChart().getPlotData();
		
		//prepare DialCap
		final double capRadius = model.getCapRadius();
		GaugePlotOptions gauge = plotData.getPlotOptions().getGauge();
		GaugePivotPlotOptions pivot = new GaugePivotPlotOptions();
		pivot.setRadius(capRadius * 100);
		gauge.setPivot(pivot);
		
		//handle the dial data scale.
		for (int j=0, len=model.size(); j < len; ++j) {
			final DialModelScale scale = model.getScale(j);
			
			Series series = plotData.getSeries(j);
			YAxis yAxis = plotData.getYAxis(j);
			
			series.setData(scale.getValue());

			//prepare the text annotation
			final String text = scale.getText();
			if (text != null && text.length() > 0) {
				YAxisTitle title = new YAxisTitle();
				title.setText(text);
				Font font = scale.getTextFont();
				if (font != null) {
					final String fontWeight = font.isBold() ? "bold" : font.isItalic() ? "italic" : "normal"; 
					title.setStyle(String.format("fontWeight: %1$s; fontFamily: %2$s; fontSize: %3$spx",
							fontWeight, font.getFamily(), font.getSize()));
				}
				yAxis.setTitle(title);
			}
			
			//prepare the value indicator
			final double valueRadius = scale.getValueRadius();
			if (valueRadius >= 0) {
				DataLabels dataLabels = new DataLabels();
				Font font = scale.getValueFont();
				if (font != null) {
					final String fontWeight = font.isBold() ? "bold" : font.isItalic() ? "italic" : "normal"; 
					dataLabels.setStyle(String.format("fontWeight: %1$s; fontFamily: %2$s; fontSize: %3$spx",
							fontWeight, font.getFamily(), font.getSize()));
				}
				dataLabels.setColor(String.format("RGB(%1$s,  %2$s, %3$s)",Color.darkGray.getRed(),
						Color.darkGray.getGreen(), Color.darkGray.getBlue()));
				
				dataLabels.setY((int)(valueRadius * 100));
				
				// fix jFreechart to highcharts offset
				dataLabels.setRotation(scale.getValueAngle() + 90);
				
				gauge.setDataLabels(dataLabels);
			}
			
			//prepare the scale
			double low = scale.getScaleLowerBound();
			double up = scale.getScaleUpperBound();
			
			
			double sa = scale.getScaleStartAngle();
			// we don't need to fix Jfreechart to Highcharts offset
			//sa -= 30;
			
			double ext = scale.getScaleExtent();
			ext = sa - ext;
			
			
			double ti = scale.getMajorTickInterval();
			int mtc = scale.getMinorTickCount();
			
			yAxis.setMin(low);
			yAxis.setMax(up);
			
			Pane pane = plotData.getPane();
			pane.setStartAngle(sa);
			pane.setEndAngle(ext);
			
			List<PaneBackground> backgrounds = new LinkedList<PaneBackground>();
			
			//prepare pane background


			String fgColor = model.getFrameFgColor();
			String bgColor = model.getFrameBgColor();
			String bgColor1 = model.getFrameBgColor1();
			String bgColor2 = model.getFrameBgColor2();
			
			if (fgColor != null) {
				PaneBackground background1 = new PaneBackground();
				background1.setBackgroundColor(fgColor);
				background1.setOuterRadius("111%");
				background1.setBorderWidth(0);
				backgrounds.add(background1);
			}
			
			if (bgColor != null) {
				PaneBackground background2 = new PaneBackground();
				background2.setBackgroundColor(bgColor);
				background2.setOuterRadius("109%");
				background2.setBorderWidth(0);
				backgrounds.add(background2);
			}
			
			// Default background
			PaneBackground background3 = new PaneBackground();
			
			if (bgColor2 != null) {
				final String direction = model.getGradientDirection();

				LinearGradient gradient;
				if ("vertical".equalsIgnoreCase(direction) || "center_vertical".equalsIgnoreCase(direction)) {
					gradient = new LinearGradient(0, 0, 0, 1);				
				} else if ("horizontal".equalsIgnoreCase(direction) || "center_horizontal".equalsIgnoreCase(direction)) {
					gradient = new LinearGradient(0, 0, 1, 0);
				} else {
					throw new IllegalArgumentException("Unsupported direction [" + direction + "]");
				}
				
				if (direction.toLowerCase().startsWith("center")) {
					gradient.addStop(0, bgColor1);
					gradient.addStop(0.5, bgColor2);
					gradient.addStop(1, bgColor1);
				} else {
					gradient.setStops(bgColor1, bgColor2);
				}
				background3.setBackgroundColor(gradient);
				backgrounds.add(background3);
			} else if (bgColor1 != null) {
				background3.setBackgroundColor(bgColor1);
				backgrounds.add(background3);
			}

			if (fgColor != null) {
				PaneBackground background4 = new PaneBackground();
				background4.setBackgroundColor(fgColor);
				background4.setInnerRadius("105%");
				background4.setOuterRadius("107%");
				background4.setBorderWidth(0);
				backgrounds.add(background4);
			}
			
			if (!backgrounds.isEmpty()) {
				pane.setBackground(backgrounds);
			}

			if (ti > 0) {
				yAxis.setTickInterval(ti);
				yAxis.setMinorTickInterval(Math.round(ti/(mtc+1)));
			}	
			 
			//handle the tick
			Font font = scale.getTickFont();
			AxisLabels labels = yAxis.getLabels();
			if (font != null) {
				final String fontWeight = font.isBold() ? "bold" : font.isItalic() ? "italic" : "normal"; 
				labels.setStyle(String.format("fontWeight: %1$s; fontFamily: %2$s; fontSize: %3$spx",
						fontWeight, font.getFamily(), font.getSize()));
			}

			double tickLabelOffset = scale.getTickLabelOffset();
			if (tickLabelOffset > 0) {
				labels.setY((int)tickLabelOffset);
				labels.setX((int)tickLabelOffset);
			}
			if (labels.getStep() == null)
				labels.setStep(1);
			yAxis.setLabels(labels);
			
			int[] tickRGB = scale.getTickRGB();
			if (tickRGB != null) {
				String tickColor = String.format("RGB(%1$s,  %2$s, %3$s)", tickRGB[0], tickRGB[1], tickRGB[2]);
				yAxis.setMinorTickColor(tickColor);
				yAxis.setTickColor(tickColor);
			}

			//prepare color ranges
			ArrayList<PlotBand> plotBands = new ArrayList<PlotBand>();
			for(int k=0, klen=scale.rangeSize(); k < klen; ++k) {
				final DialModelRange rng = scale.getRange(k);
				int[] rngRGB = rng.getRangeRGB();
				if (rngRGB == null) {
					rngRGB = new int[] {0x00, 0x00, 0xFF};
				}


				final String plotBandColor = String.format("RGB(%1$s,  %2$s, %3$s)", rngRGB[0], rngRGB[1], rngRGB[2]);

				PlotBand plotBand = 
						new PlotBand(rng.getLowerBound(), rng.getUpperBound(), plotBandColor);
				
				plotBand.setInnerRadius(Double.toString(rng.getInnerRadius() * 100) + "%");
				plotBand.setOuterRadius(Double.toString(rng.getOuterRadius() * 100) + "%");
				plotBands.add(plotBand);
			}
			
			yAxis.setPlotBands(plotBands);
			
		}
		
		return plotData;
	}
	
	/**
	 * Draws the chart with xy model
	 */
	protected PlotData drawXYModel(XYModel model) {
		PlotData plotData = getChart().getPlotData();

		Map<Comparable<?>, Map<Number, Integer>> seriesMap = new HashMap<Comparable<?>, Map<Number, Integer>>();
		Set<Number> xSet = null;

		if (model.isAutoSort())
			xSet = new TreeSet<Number>();
		else
			xSet = new LinkedHashSet<Number>(13);

		// collect all x
		for (Comparable<?> ser : model.getSeries()) {
			final int size = model.getDataCount(ser);
			Map<Number, Integer> indexMap = new TreeMap<Number, Integer>();
			for (int j = 0; j < size; ++j) {
				Number x = model.getX(ser, j);
				xSet.add(x);
				indexMap.put(x, new Integer(j));
			}
			seriesMap.put(ser, indexMap);
		}
		
		// set after sorting
		int i = 0;
		for (Comparable<?> ser : model.getSeries()) {
			Map<Number, Integer> indexMap = seriesMap.get(ser);

			Series series = plotData.getSeries(i++);
			for (final Iterator<?> it2 = xSet.iterator(); it2.hasNext();) {
				Number x = (Number) it2.next();
				Integer o = indexMap.get(x);
				if (o == null)
					continue;
				series.addPoint(x, model.getY(ser, o));
			}
			series.setName(ser.toString());
		}

		return plotData;
	}
	
	/**
	 * Draws the chart with xyz model
	 */
	protected PlotData drawXYZModel(XYZModel model) {
		PlotData plotData = getChart().getPlotData();

		Map<Comparable<?>, Map<Number, Integer>> seriesMap = new HashMap<Comparable<?>, Map<Number, Integer>>();
		Set<Number> xSet = null;

		if (model.isAutoSort())
			xSet = new TreeSet<Number>();
		else
			xSet = new LinkedHashSet<Number>(13);

		// collect all x
		for (Comparable<?> ser : model.getSeries()) {
			final int size = model.getDataCount(ser);
			Map<Number, Integer> indexMap = new TreeMap<Number, Integer>();
			for (int j = 0; j < size; ++j) {
				Number x = model.getX(ser, j);
				xSet.add(x);
				indexMap.put(x, new Integer(j));
			}
			seriesMap.put(ser, indexMap);
		}
		
		// set after sorting
		int i = 0;
		for (Comparable<?> ser : model.getSeries()) {
			Map<Number, Integer> indexMap = seriesMap.get(ser);

			Series series = plotData.getSeries(i++);
			for (final Iterator<?> it2 = xSet.iterator(); it2.hasNext();) {
				Number x = (Number) it2.next();
				Integer o = indexMap.get(x);
				if (o == null)
					continue;
				series.addPoint(x, model.getY(ser, o), model.getZ(ser, o));
			}
			series.setName(ser.toString());
		}

		return plotData;
	}
}
