/** YAxis.java.

	Purpose:
		
	Description:
		
	History:
		5:05:49 PM Jan 9, 2014, Created by jumperchen

Copyright (C) 2014 Potix Corporation. All Rights Reserved.
 */
package org.zkoss.chart;

import org.zkoss.chart.util.DeferredCall;
import org.zkoss.chart.util.DynamicalAttribute;
import org.zkoss.chart.util.JSFunction;
import org.zkoss.json.JSONObject;

/**
 * The Y axis or value axis. Normally this is the vertical axis, though if the
 * chart is inverted this is the horizontal axis. In case of multiple axes, the
 * yAxis node is an array of configuration objects.
 * 
 * @author jumperchen
 */
public class YAxis extends Axis {
	private enum Attrs implements PlotAttribute, DynamicalAttribute {
		stackLabels,
		endOnTick,
		title,
		gridLineWidth,
		tickPixelInterval,
		showLastLabel,
		labels,
		lineWidth,
		maxPadding,
		minPadding,
		startOnTick,
		tickWidth
	}

	/**
	 * Returns the stack labels show the total value for each bar in a stacked
	 * column or bar chart. The label will be placed on top of positive columns
	 * and below negative columns. In case of an inverted column chart or a bar
	 * chart the label is placed to the right of positive bars and to the left
	 * of negative bars.
	 */
	public StackLabels getStackLabels() {
		StackLabels stackLabels = (StackLabels) getAttr(Attrs.stackLabels);
		if (stackLabels == null) {
			stackLabels = new StackLabels();
			setStackLabels(stackLabels);
		}
		return stackLabels;
	}

	/**
	 * Sets the stack labels show the total value for each bar in a stacked
	 * column or bar chart. The label will be placed on top of positive columns
	 * and below negative columns. In case of an inverted column chart or a bar
	 * chart the label is placed to the right of positive bars and to the left
	 * of negative bars.
	 */
	public void setStackLabels(StackLabels stackLabels) {
		setAttr(Attrs.stackLabels, stackLabels);
	}

	/**
	 * Returns whether to force the axis to end on a tick. Use this option with
	 * the {@link #setMaxPadding(Number)} option to control the axis end.
	 * <p>
	 * Default: true.
	 */
	public boolean isEndOnTick() {
		return getAttr(Attrs.endOnTick, true).asBoolean();
	}

	/**
	 * Sets whether to force the axis to end on a tick. Use this option with the
	 * {@link #setMaxPadding(Number)} option to control the axis end.
	 * <p>
	 * Default: true.
	 */
	public void setEndOnTick(boolean endOnTick) {
		setAttr(Attrs.endOnTick, endOnTick);
	}

	/**
	 * Returns whether to show the last tick label.
	 * <p>
	 * Default: true.
	 */
	public boolean isShowLastLabel() {
		return getAttr(Attrs.showLastLabel, true).asBoolean();
	}

	/**
	 * Returns whether to force the axis to start on a tick. Use this option
	 * with the <code>maxPadding</code> option to control the axis start.
	 * <p>
	 * Default: true.
	 */
	public boolean isStartOnTick() {
		return getAttr(Attrs.startOnTick, false).asBoolean();
	}

	/**
	 * Returns the pixel width of the major tick marks.
	 * <p>
	 * Default: 0.
	 */
	public Number getTickWidth() {
		return getAttr(Attrs.tickWidth, 0).asNumber();
	}

	/**
	 * Sets the pixel width of the major tick marks.
	 */
	public void setTickWidth(Number tickWidth) {
		setAttr(Attrs.tickWidth, tickWidth, 0);
	}

	/**
	 * Returns padding of the max value relative to the length of the axis. A
	 * padding of 0.05 will make a 100px axis 5px longer. This is useful when
	 * you don't want the highest data value to appear on the edge of the plot
	 * area. When the {@link #setMax(Number)} option is set or a max extreme is
	 * set using {@link #setExtremes(Number, Number)}, the maxPadding will be
	 * ignored.
	 * <p>
	 * Default: 0.05.
	 */
	public Number getMaxPadding() {
		return getAttr(Attrs.maxPadding, 0.05).asNumber();
	}

	/**
	 * Sets padding of the max value relative to the length of the axis. A
	 * padding of 0.05 will make a 100px axis 5px longer. This is useful when
	 * you don't want the highest data value to appear on the edge of the plot
	 * area. When the {@link #setMax(Number)} option is set or a max extreme is
	 * set using {@link #setExtremes(Number, Number)}, the maxPadding will be
	 * ignored.
	 */
	public void setMaxPadding(Number maxPadding) {
		setAttr(Attrs.maxPadding, maxPadding, 0.05);
	}

	/**
	 * Returns padding of the min value relative to the length of the axis. A
	 * padding of 0.05 will make a 100px axis 5px longer. This is useful when
	 * you don't want the lowest data value to appear on the edge of the plot
	 * area. When the {@link #setMin(Number)} option is set or a min extreme is
	 * set using {@link #setExtremes(Number, Number)}, the minPadding will be
	 * ignored.
	 * <p>
	 * Default: 0.05.
	 */
	public Number getMinPadding() {
		return getAttr(Attrs.minPadding, 0.05).asNumber();
	}

	/**
	 * Sets padding of the min value relative to the length of the axis. A
	 * padding of 0.05 will make a 100px axis 5px longer. This is useful when
	 * you don't want the lowest data value to appear on the edge of the plot
	 * area. When the {@link #setMin(Number)} option is set or a min extreme is
	 * set using {@link #setExtremes(Number, Number)}, the minPadding will be
	 * ignored.
	 */
	public void setMinPadding(Number minPadding) {
		setAttr(Attrs.minPadding, minPadding, 0.05);
	}

	/**
	 * Returns the width of the grid lines extending the ticks across the plot
	 * area.
	 * <p>
	 * Default: 1.
	 */
	public Number getGridLineWidth() {
		return getAttr(Attrs.gridLineWidth, 1).asNumber();
	}

	/**
	 * Sets the width of the grid lines extending the ticks across the plot
	 * area.
	 */
	public void setGridLineWidth(Number gridLineWidth) {
		setAttr(Attrs.gridLineWidth, gridLineWidth, 1);
	}

	/**
	 * Returns if tickInterval is <code>null</code> this option sets the
	 * approximate pixel interval of the tick marks. Not applicable to
	 * categorized axis. Defaults to <code>72</code> for the Y axis and
	 * <code>100</code> for the X axis.
	 * <p>
	 * Default: 72.
	 */
	public Number getTickPixelInterval() {
		return getAttr(Attrs.tickPixelInterval, 72).asNumber();
	}

	/**
	 * Sets if tickInterval is <code>null</code> this option sets the
	 * approximate pixel interval of the tick marks. Not applicable to
	 * categorized axis. Defaults to <code>72</code> for the Y axis and
	 * <code>100</code> for the X axis.
	 * <p>
	 * Default: 72.
	 */
	public void setTickPixelInterval(Number tickPixelInterval) {
		setAttr(Attrs.tickPixelInterval, tickPixelInterval, 72);
	}

	/**
	 * Returns the axis labels show the number or category for each tick.
	 * <p>
	 * Default:
	 * 
	 * <pre>
	 * labels: {
	 *	  x: -8,
	 *	  y: 3
	 * }
	 * </pre>
	 */
	public AxisLabels getLabels() {
		AxisLabels labels = (AxisLabels) getAttr(Attrs.labels);
		if (labels == null) {
			labels = new YAxisLabels();
			setLabels(labels);
		}
		return labels;
	}

	/**
	 * Returns the width of the line marking the axis itself.
	 * <p>
	 * Default: 0.
	 */
	public Number getLineWidth() {
		return getAttr(Attrs.lineWidth, 0).asNumber();
	}

	/**
	 * Sets the width of the line marking the axis itself.
	 */
	public void setLineWidth(Number lineWidth) {
		setAttr(Attrs.lineWidth, lineWidth, 0);
	}

	/**
	 * Returns the axis title, showing next to the axis line.
	 * <p>
	 * Default: an instance of {@link AxisTitle}.
	 */
	public AxisTitle getTitle() {
		AxisTitle title = (AxisTitle) getAttr(Attrs.title);
		if (title == null) {
			title = new YAxisTitle();
			setAttr(Attrs.title, title);
		}
		return title;
	}

	// Override
	public void onChange(OptionDataEvent event) {
		final Optionable target = event.getTarget();

		if (target instanceof StackLabels) {
			event.addJSFunctionCall(new DeferredCall() {
				public void execute(JSFunction func) {
					JSONObject json = new JSONObject();
					json.put("stackLabels", target);
					func.callFunction("update", json);
				}
			});

			event.setCurrentTarget(this);

			fireEvent(event);
		} else {
			super.onChange(event);
		}
	}
}
