/** Tooltip.java.

	Purpose:
		
	Description:
		
	History:
		9:15:41 AM Jan 13, 2014, Created by RaymondChao

Copyright (C) 2014 Potix Corporation. All Rights Reserved.
 */
package org.zkoss.chart;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.zkoss.chart.util.DynamicalAttribute;
import org.zkoss.chart.util.MapsHelper;
import org.zkoss.lang.Generics;
import org.zkoss.util.Maps;

/**
 * Options for the tooltip that appears when the user hovers over a series or
 * point.
 * <p>
 * All the options in this class support {@link DynamicalAttribute}.
 * 
 * @author jumperchen
 */
public class Tooltip extends Optionable {
	private enum Attrs implements PlotAttribute, DynamicalAttribute {
		animation,
		backgroundColor,
		borderColor,
		borderRadius,
		borderWidth,
		crosshairs,
		dateTimeLabelFormats,
		enabled,
		followPointer,
		followTouchMove,
		footerFormat,
		format,
		headerFormat,
		hideDelay,
		pointFormat,
		positioner,
		shadow,
		shared,
		snap,
		style,
		useHTML,
		valueDecimals,
		valuePrefix,
		valueSuffix,
		xDateFormat
	}

	/**
	 * Returns whether to enable or disable animation of the tooltip. In slow
	 * legacy IE browsers the animation is disabled by default.
	 * <p>
	 * Default: true.
	 */
	public boolean isAnimation() {
		return getAttr(Attrs.animation, true).asBoolean();
	}

	/**
	 * Sets whether to enable or disable animation of the tooltip. In slow
	 * legacy IE browsers the animation is disabled by default.
	 */
	public void setAnimation(boolean animation) {
		setAttr(Attrs.animation, animation);
	}

	/**
	 * Returns the background color or gradient for the tooltip.
	 * <p>
	 * Default: "rgba(255, 255, 255, 0.85)".
	 */
	public Color getBackgroundColor() {
		if (!containsKey(Attrs.backgroundColor)) {
			setBackgroundColor("rgba(255, 255, 255, 0.85)");
		}
		return (Color) getAttr(Attrs.backgroundColor);
	}

	/**
	 * Sets the background color or gradient for the tooltip.
	 */
	public void setBackgroundColor(Color color) {
		setAttr(Attrs.backgroundColor, color, NOT_NULL_VALUE);
	}

	/**
	 * Sets the background color or gradient for the tooltip.
	 */
	public void setBackgroundColor(String color) {
		setBackgroundColor(new Color(color));
	}

	/**
	 * Sets the background color or gradient for the tooltip.
	 */
	public void setBackgroundColor(LinearGradient color) {
		setBackgroundColor(new Color(color));
	}

	/**
	 * Sets the background color or gradient for the tooltip.
	 */
	public void setBackgroundColor(RadialGradient color) {
		setBackgroundColor(new Color(color));
	}

	/**
	 * Returns the color of the tooltip border. When <code>null</code>, the
	 * border takes the color of the corresponding series or point.
	 * <p>
	 * Default: "auto".
	 */
	public Color getBorderColor() {
		if (!containsKey(Attrs.borderColor)) {
			setBorderColor("auto");
		}
		return (Color) getAttr(Attrs.borderColor);
	}

	/**
	 * Sets the color of the tooltip border. When <code>null</code>, the border
	 * takes the color of the corresponding series or point.
	 */
	public void setBorderColor(Color color) {
		setAttr(Attrs.borderColor, color, NOT_NULL_VALUE);
	}

	/**
	 * Sets the color of the tooltip border. When <code>null</code>, the border
	 * takes the color of the corresponding series or point.
	 */
	public void setBorderColor(String color) {
		setBorderColor(new Color(color));
	}

	/**
	 * Sets the color of the tooltip border. When <code>null</code>, the border
	 * takes the color of the corresponding series or point.
	 */
	public void setBorderColor(LinearGradient color) {
		setBorderColor(new Color(color));
	}

	/**
	 * Sets the color of the tooltip border. When <code>null</code>, the border
	 * takes the color of the corresponding series or point.
	 */
	public void setBorderColor(RadialGradient color) {
		setBorderColor(new Color(color));
	}

	/**
	 * Returns the radius of the rounded border corners.
	 * <p>
	 * Default: 3.
	 * 
	 * @return borderRadius the radius of the rounded border corners.
	 */
	public Number getBorderRadius() {
		return getAttr(Attrs.borderRadius, 3).asNumber();
	}

	/**
	 * Sets the radius of the rounded border corners.
	 * 
	 * @param borderRadius
	 *            the radius of the rounded border corners
	 */
	public void setBorderRadius(Number borderRadius) {
		setAttr(Attrs.borderRadius, borderRadius, 3);
	}

	/**
	 * Returns the pixel width of the tooltip border.
	 * <p>
	 * Default: 1.
	 * 
	 * @return borderWidth the pixel width of the tooltip border.
	 */
	public Number getBorderWidth() {
		return getAttr(Attrs.borderWidth, 1).asNumber();
	}

	/**
	 * Sets the pixel width of the tooltip border.
	 * 
	 * @param borderWidth
	 *            the pixel width of the tooltip border
	 */
	public void setBorderWidth(Number borderWidth) {
		setAttr(Attrs.borderWidth, borderWidth, 1);
	}

	/**
	 * Returns display crosshairs to connect the points with their corresponding
	 * axis values. The crosshairs can be defined as a boolean, an array of
	 * booleans or an object.
	 * <dl>
	 * <dt>Boolean</dt>
	 * <dd>If the crosshairs option is true, a single crosshair relating to the
	 * x axis will be shown.</dd>
	 * <dt>Array of booleans</dt>
	 * <dd>In an array of booleans, the first value turns on the x axis
	 * crosshair and the second value to the y axis crosshair. Use
	 * <code>[true, true]</code> to show complete crosshairs.</dd>
	 * <dt>Array of objects</dt>
	 * <dd>In an array of objects, the first value applies to the x axis
	 * crosshair and the second value to the y axis crosshair. For each
	 * dimension, a <code>width</code>, <code>color</code>,
	 * <code><a href="http://jsfiddle.net/gh/get/jquery/1.7.1/highslide-software/highcharts.com/tree/master/samples/highcharts/plotoptions/series-dashstyle-all/">dashStyle</a></code>
	 * and <code>zIndex</code> can be given.</dd>
	 * </dl>
	 * <p>
	 * Defaults to <code>null</code>.
	 */
	public Object getCrosshairs() {
		return getAttr(Attrs.crosshairs, null).asValue();
	}

	/**
	 * Sets display crosshairs to connect the points with their corresponding
	 * axis values. The crosshairs can be defined as a boolean, an array of
	 * booleans or an object.
	 * <dl>
	 * <dt>Boolean</dt>
	 * <dd>If the crosshairs option is true, a single crosshair relating to the
	 * x axis will be shown.</dd>
	 * <dt>Array of booleans</dt>
	 * <dd>In an array of booleans, the first value turns on the x axis
	 * crosshair and the second value to the y axis crosshair. Use
	 * <code>[true, true]</code> to show complete crosshairs.</dd>
	 * <dt>Array of objects</dt>
	 * <dd>In an array of objects, the first value applies to the x axis
	 * crosshair and the second value to the y axis crosshair. For each
	 * dimension, a <code>width</code>, <code>color</code>,
	 * <code><a href="http://jsfiddle.net/gh/get/jquery/1.7.1/highslide-software/highcharts.com/tree/master/samples/highcharts/plotoptions/series-dashstyle-all/">dashStyle</a></code>
	 * and <code>zIndex</code> can be given.</dd>
	 * </dl>
	 * <p>
	 * Defaults to <code>null</code>.
	 */
	public void setCrosshairs(Boolean crosshairs) {
		setAttr(Attrs.crosshairs, crosshairs);
	}

	/**
	 * Sets display crosshairs to connect the points with their corresponding
	 * axis values. The crosshairs can be defined as a boolean, an array of
	 * booleans or an object.
	 * <dl>
	 * <dt>Boolean</dt>
	 * <dd>If the crosshairs option is true, a single crosshair relating to the
	 * x axis will be shown.</dd>
	 * <dt>Array of booleans</dt>
	 * <dd>In an array of booleans, the first value turns on the x axis
	 * crosshair and the second value to the y axis crosshair. Use
	 * <code>[true, true]</code> to show complete crosshairs.</dd>
	 * <dt>Array of objects</dt>
	 * <dd>In an array of objects, the first value applies to the x axis
	 * crosshair and the second value to the y axis crosshair. For each
	 * dimension, a <code>width</code>, <code>color</code>,
	 * <code><a href="http://jsfiddle.net/gh/get/jquery/1.7.1/highslide-software/highcharts.com/tree/master/samples/highcharts/plotoptions/series-dashstyle-all/">dashStyle</a></code>
	 * and <code>zIndex</code> can be given.</dd>
	 * </dl>
	 * <p>
	 * Defaults to <code>null</code>.
	 */
	public <T> void setCrosshairs(List<T> crosshairs) {
		setAttr(Attrs.crosshairs, crosshairs);
	}

	/**
	 * Returns the date format in the tooltip's header
	 * <p>
	 * For series on a datetime axes, the date format in the tooltip's header
	 * will by default be guessed based on the closest data points. This member
	 * gives the default string representations used for each unit. For an
	 * overview of the replacement codes, see
	 * {@link Charts#setDateFormat(String, Number, boolean)}.
	 * </p>
	 * <p>
	 * Defaults:
	 * 
	 * <pre>
	 * {
	 *     millisecond:"%A, %b %e, %H:%M:%S.%L",
	 *     second:"%A, %b %e, %H:%M:%S",
	 *     minute:"%A, %b %e, %H:%M",
	 *     hour:"%A, %b %e, %H:%M",
	 *     day:"%A, %b %e, %Y",
	 *     week:"Week from %A, %b %e, %Y",
	 *     month:"%B %Y",
	 *     year:"%Y"
	 * }
	 * </pre>
	 * 
	 * </p>
	 */
	public DateTimeLabelFormats getDateTimeLabelFormats() {
		DateTimeLabelFormats dateTimeLabelFormats = (DateTimeLabelFormats) getAttr(Attrs.dateTimeLabelFormats);
		if (dateTimeLabelFormats == null) {
			dateTimeLabelFormats = new DateTimeLabelFormats();
			setDateTimeLabelFormats(dateTimeLabelFormats);
		}
		return dateTimeLabelFormats;
	}

	/**
	 * Sets the date format in the tooltip's header
	 * <p>
	 * For series on a datetime axes, the date format in the tooltip's header
	 * will by default be guessed based on the closest data points. This member
	 * gives the default string representations used for each unit. For an
	 * overview of the replacement codes, see
	 * {@link Charts#setDateFormat(String, Number, boolean)}.
	 * </p>
	 */
	public void setDateTimeLabelFormats(
			DateTimeLabelFormats dateTimeLabelFormats) {
		setAttr(Attrs.dateTimeLabelFormats, dateTimeLabelFormats);
	}

	/**
	 * Returns whether to enable or disable the tooltip.
	 * <p>
	 * Default: true.
	 */
	public boolean isEnabled() {
		return getAttr(Attrs.enabled, true).asBoolean();
	}

	/**
	 * Sets whether to enable or disable the tooltip.
	 */
	public void setEnabled(boolean enabled) {
		setAttr(Attrs.enabled, enabled);
	}

	/**
	 * Returns whether the tooltip should follow the mouse as it moves across
	 * columns, pie slices and other point types with an extent. By default it
	 * behaves this way for scatter, bubble and pie series by override in the
	 * <code>plotOptions</code> for those series types. </p>
	 * <p>
	 * For touch moves to behave the same way,
	 * {@link #setFollowTouchMove(boolean)} must be <code>true</code> also.
	 * </p>
	 * <p>
	 * Default: false.
	 */
	public boolean isFollowPointer() {
		return getAttr(Attrs.followPointer, false).asBoolean();
	}

	/**
	 * Sets whether the tooltip should follow the mouse as it moves across
	 * columns, pie slices and other point types with an extent. By default it
	 * behaves this way for scatter, bubble and pie series by override in the
	 * <code>plotOptions</code> for those series types. </p>
	 * <p>
	 * For touch moves to behave the same way,
	 * {@link #setFollowTouchMove(boolean)} must be <code>true</code> also.
	 * </p>
	 */
	public void setFollowPointer(boolean followPointer) {
		setAttr(Attrs.followPointer, followPointer);
	}

	/**
	 * Returns whether the tooltip should follow the finger as it moves on a
	 * touch device. The default value of <code>false</code> causes a touch move
	 * to scroll the web page, as is default behaviour on touch devices. Setting
	 * it to <code>true</code> may cause the user to be trapped inside the chart
	 * and unable to scroll away, so it should be used with care. If
	 * {@link Chart#setZoomType(String)} is set, it will override
	 * {@link #setFollowTouchMove(boolean)}.
	 * <p>
	 * Default: false.
	 */
	public boolean isFollowTouchMove() {
		return getAttr(Attrs.followTouchMove, false).asBoolean();
	}

	/**
	 * Sets whether the tooltip should follow the finger as it moves on a touch
	 * device. The default value of <code>false</code> causes a touch move to
	 * scroll the web page, as is default behaviour on touch devices. Setting it
	 * to <code>true</code> may cause the user to be trapped inside the chart
	 * and unable to scroll away, so it should be used with care. If
	 * {@link Chart#setZoomType(String)} is set, it will override
	 * {@link #setFollowTouchMove(boolean)}.
	 */
	public void setFollowTouchMove(boolean followTouchMove) {
		setAttr(Attrs.followTouchMove, followTouchMove);
	}

	/**
	 * Returns a string to append to the tooltip format.
	 * <p>
	 * Default: false.
	 */
	public String getFooterFormat() {
		return getAttr(Attrs.footerFormat, false).asString();
	}

	/**
	 * Sets a string to append to the tooltip format.
	 */
	public void setFooterFormat(String footerFormat) {
		setAttr(Attrs.footerFormat, footerFormat);
	}

	/**
	 * Returns a format text of the tooltip.
	 * <p>
	 * Default: null.
	 */
	public String getFormat() {
		return getAttr(Attrs.format, null).asString();
	}

	/**
	 * Sets a format text of the tooltip.
	 * <p>
	 * Default: null.
	 */
	public void setFormat(String format) {
		setAttr(Attrs.format, format);
	}

	/**
	 * Returns the HTML of the tooltip header line. Variables are enclosed by
	 * curly brackets. Available variables are <code>point.key</code>,
	 * <code>series.name</code>, <code>series.color</code> and other members
	 * from the <code>point</code> and <code>series</code> objects. The
	 * <code>point.key</code> variable contains the category name, x value or
	 * datetime string depending on the type of axis. For datetime axes, the
	 * <code>point.key</code> date format can be set using tooltip.xDateFormat.
	 * </p>
	 * <p>
	 * Default:
	 * <code>&lt;span style="font-size: 10px"&gt;{point.key}&lt;/span&gt;&lt;br/&gt;</code>
	 * </p>
	 */
	public String getHeaderFormat() {
		return getAttr(Attrs.headerFormat,
				"<span style=\"font-size: 10px\">{point.key}</span><br/>")
				.asString();
	}

	/**
	 * Sets the HTML of the tooltip header line. Variables are enclosed by curly
	 * brackets. Available variables are <code>point.key</code>,
	 * <code>series.name</code>, <code>series.color</code> and other members
	 * from the <code>point</code> and <code>series</code> objects. The
	 * <code>point.key</code> variable contains the category name, x value or
	 * datetime string depending on the type of axis. For datetime axes, the
	 * <code>point.key</code> date format can be set using tooltip.xDateFormat.
	 * </p>
	 */
	public void setHeaderFormat(String headerFormat) {
		setAttr(Attrs.headerFormat, headerFormat, "<span style=\"font-size: 10px\">{point.key}</span><br/>");
	}

	/**
	 * Returns the number of milliseconds to wait until the tooltip is hidden
	 * when mouse out from a point or chart.
	 * <p>
	 * Default: 500.
	 */
	public Number getHideDelay() {
		return getAttr(Attrs.hideDelay, 500).asNumber();
	}

	/**
	 * Sets the number of milliseconds to wait until the tooltip is hidden when
	 * mouse out from a point or chart.
	 */
	public void setHideDelay(Number hideDelay) {
		setAttr(Attrs.hideDelay, hideDelay, 500);
	}

	/**
	 * Returns the HTML of the point's line in the tooltip. Variables are
	 * enclosed by curly brackets. Available variables are point.x, point.y,
	 * series.name and series.color and other properties on the same form.
	 * Furthermore, point.y can be extended by the <code>tooltip.yPrefix</code>
	 * and <code>tooltip.ySuffix</code> variables. This can also be overridden
	 * for each series, which makes it a good hook for displaying units. </p>
	 * <p>
	 * Default:
	 * <code>&lt;span style="color:{series.color}"&gt;{series.name}&lt;/span&gt;: &lt;b&gt;{point.y}&lt;/b&gt;&lt;br/&gt;</code>
	 * </p>
	 */
	public String getPointFormat() {
		return getAttr(
				Attrs.pointFormat,
				"<span style=\"color:{series.color}\">{series.name}</span>: <b>{point.y}</b><br/>")
				.asString();
	}

	/**
	 * Sets the HTML of the point's line in the tooltip. Variables are enclosed
	 * by curly brackets. Available variables are point.x, point.y, series.name
	 * and series.color and other properties on the same form. Furthermore,
	 * point.y can be extended by the <code>tooltip.yPrefix</code> and
	 * <code>tooltip.ySuffix</code> variables. This can also be overridden for
	 * each series, which makes it a good hook for displaying units. </p>
	 */
	public void setPointFormat(String pointFormat) {
		setAttr(Attrs.pointFormat, pointFormat, "<span style=\"color:{series.color}\">{series.name}</span>: <b>{point.y}</b><br/>");
	}

	/**
	 * Returns whether to apply a drop shadow to the tooltip.
	 * <p>
	 * Default: true.
	 */
	public boolean isShadow() {
		return getAttr(Attrs.shadow, true).asBoolean();
	}

	/**
	 * Sets whether to apply a drop shadow to the tooltip.
	 */
	public void setShadow(boolean shadow) {
		setAttr(Attrs.shadow, shadow);
	}

	/**
	 * Returns whether the tooltip is shared. When the tooltip is shared, the
	 * entire plot area will capture mouse movement. Tooltip texts for series
	 * types with ordered data (not pie, scatter, flags etc) will be shown in a
	 * single bubble. This is recommended for single series charts and for
	 * tablet/mobile optimized charts.
	 * <p>
	 * Default: false.
	 */
	public boolean isShared() {
		return getAttr(Attrs.shared, false).asBoolean();
	}

	/**
	 * Sets whether the tooltip is shared. When the tooltip is shared, the
	 * entire plot area will capture movement. Tooltip texts for series types
	 * with ordered data (not pie, scatter, flags etc) will be shown in a single
	 * bubble. This is recommended for single series charts and for
	 * tablet/mobile optimized charts.
	 */
	public void setShared(boolean shared) {
		setAttr(Attrs.shared, shared);
	}

	/**
	 * Returns proximity snap for graphs or single points. Does not apply to
	 * bars, columns and pie slices. It defaults to 10 for mouse-powered devices
	 * and 25 for touch devices.
	 * <p>
	 * Default: null
	 */
	public Number getSnap() {
		return getAttr(Attrs.snap, null).asNumber();
	}

	/**
	 * Sets proximity snap for graphs or single points. Does not apply to bars,
	 * columns and pie slices. It defaults to 10 for mouse-powered devices and
	 * 25 for touch devices.
	 */
	public void setSnap(Number snap) {
		setAttr(Attrs.snap, snap);
	}

	/**
	 * Returns CSS styles for the tooltip. The tooltip can also be styled
	 * through the CSS class <code>.highcharts-tooltip</code>. Default value:
	 * 
	 * <pre>
	 * color: #333333; fontSize: 12px; padding: 8px;
	 * </pre>
	 * <p>
	 */
	public <K, V> Map<K, V> getStyle() {
		if (!containsKey(Attrs.style)) {
			setStyle("color: #333333; fontSize: 12px; padding: 8px;");
		}
		return Generics.cast(getAttr(Attrs.style));
	}

	/**
	 * Sets CSS styles for the tooltip. The tooltip can also be styled through
	 * the CSS class <code>.highcharts-tooltip</code>. Default value:
	 * 
	 * <pre>
	 * color: #333333; fontSize: 12px; padding: 8px;
	 * </pre>
	 * <p>
	 */
	public void setStyle(String style) {
		setStyle(MapsHelper.parse(new LinkedHashMap<String, String>(), style, ':', ';', '\''));
	}

	/**
	 * Sets CSS styles for the tooltip. The tooltip can also be styled through
	 * the CSS class <code>.highcharts-tooltip</code>. Default value:
	 * 
	 * <pre>
	 * color: #333333; fontSize: 12px; padding: 8px;
	 * </pre>
	 * <p>
	 */
	public <K, V> void setStyle(Map<K, V> style) {
		setAttr(Attrs.style, style, NOT_NULL_VALUE);
	}

	/**
	 * Returns whether to use HTML to render the contents of the tooltip instead
	 * of SVG. Using HTML allows advanced formatting like tables and images in
	 * the tooltip. It is also recommended for rtl languages as it works around
	 * rtl bugs in early Firefox.
	 * <p>
	 * Default: false.
	 */
	public boolean isUseHTML() {
		return getAttr(Attrs.useHTML, false).asBoolean();
	}

	/**
	 * Sets whether to use HTML to render the contents of the tooltip instead of
	 * SVG. Using HTML allows advanced formatting like tables and images in the
	 * tooltip. It is also recommended for rtl languages as it works around rtl
	 * bugs in early Firefox.
	 */
	public void setUseHTML(boolean useHTML) {
		setAttr(Attrs.useHTML, useHTML);
	}

	/**
	 * Returns how many decimals to show in each series' y value. This is
	 * overridable in each series' tooltip options object. The default is to
	 * preserve all decimals.
	 * <p>
	 * Default: null.
	 */
	public Number getValueDecimals() {
		return getAttr(Attrs.valueDecimals, null).asNumber();
	}

	/**
	 * Sets how many decimals to show in each series' y value. This is
	 * overridable in each series' tooltip options object. The default is to
	 * preserve all decimals.
	 */
	public void setValueDecimals(Number valueDecimals) {
		setAttr(Attrs.valueDecimals, valueDecimals);
	}

	/**
	 * Returns a string to prepend to each series' y value. Overridable in each
	 * series' tooltip options object.
	 * <p>
	 * Default: null.
	 */
	public String getValuePrefix() {
		return getAttr(Attrs.valuePrefix, null).asString();
	}

	/**
	 * Sets a string to prepend to each series' y value. Overridable in each
	 * series' tooltip options object.
	 */
	public void setValuePrefix(String valuePrefix) {
		setAttr(Attrs.valuePrefix, valuePrefix);
	}

	/**
	 * Returns a string to append to each series' y value. Overridable in each
	 * series' tooltip options object.
	 * <p>
	 * Default: null.
	 */
	public String getValueSuffix() {
		return getAttr(Attrs.valueSuffix, null).asString();
	}

	/**
	 * Sets the string to append to each series' y value. Overridable in each
	 * series' tooltip options object.
	 */
	public void setValueSuffix(String valueSuffix) {
		setAttr(Attrs.valueSuffix, valueSuffix);
	}

	/**
	 * Returns the format for the date in the tooltip header if the X axis is a
	 * datetime axis. The default is a best guess based on the smallest distance
	 * between points in the chart.
	 * <p>
	 * Default: null.
	 */
	public String getXDateFormat() {
		return getAttr(Attrs.xDateFormat, null).asString();
	}

	/**
	 * Sets the format for the date in the tooltip header if the X axis is a
	 * datetime axis. The default is a best guess based on the smallest distance
	 * between points in the chart.
	 */
	public void setXDateFormat(String xDateFormat) {
		setAttr(Attrs.xDateFormat, xDateFormat);
	}

}