/** Title.java.

	Purpose:
		
	Description:
		
	History:
		9:25:14 AM Jan 13, 2014, Created by RaymondChao

Copyright (C) 2014 Potix Corporation. All Rights Reserved.
 */
package org.zkoss.chart;

import java.util.LinkedHashMap;
import java.util.Map;

import org.zkoss.chart.util.DynamicalAttribute;
import org.zkoss.chart.util.MapsHelper;
import org.zkoss.lang.Generics;

/**
 * The chart's main title.
 * <p>
 * All the options in this class support {@link DynamicalAttribute}.
 * 
 * @author jumperchen
 */
public class Title extends Optionable {
	private enum Attrs implements PlotAttribute, DynamicalAttribute {
		align, floating, margin, style, text, useHTML, verticalAlign, x, y
	}

	/**
	 * Returns the horizontal alignment of the title. Can be one of "left",
	 * "center" and "right".
	 * <p>
	 * Defaults to <code>center</code>.
	 */
	public String getAlign() {
		return getAttr(Attrs.align, "center").asString();
	}

	/**
	 * Sets the horizontal alignment of the title. Can be one of "left",
	 * "center" and "right".
	 * <p>
	 * Defaults to <code>center</code>.
	 */
	public void setAlign(String align) {
		if (!("left".equals(align) || "center".equals(align) || "right"
				.equals(align)))
			throw new IllegalArgumentException("Unsupported align: [" + align
					+ "]");
		setAttr(Attrs.align, align, "center");
	}

	/**
	 * Returns when the title is floating, the plot area will not move to make
	 * space for it.
	 * <p>
	 * Defaults to false.
	 */
	public boolean isFloating() {
		return getAttr(Attrs.floating, false).asBoolean();
	}

	/**
	 * Sets when the title is floating, the plot area will not move to make
	 * space for it.
	 * <p>
	 * Defaults to false.
	 */
	public void setFloating(boolean floating) {
		setAttr(Attrs.floating, floating);
	}

	/**
	 * Returns the margin between the title and the plot area, or if a subtitle
	 * is present, the margin between the subtitle and the plot area.
	 * <p>
	 * Defaults to 15.
	 */
	public Number getMargin() {
		return getAttr(Attrs.margin, 15).asNumber();
	}

	/**
	 * Sets the margin between the title and the plot area, or if a subtitle is
	 * present, the margin between the subtitle and the plot area.
	 * <p>
	 * Defaults to 15.
	 */
	public void setMargin(Number margin) {
		setAttr(Attrs.margin, margin, 15);
	}

	/**
	 * Returns CSS styles for the title. Use this for font styling, but use
	 * align, x and y for text alignment.
	 * <p>
	 * Defaults to:
	 * 
	 * <pre>
	 * color: #3E576F; fontSize: 16px;
	 * </pre>
	 */
	public <K, V> Map<K, V> getStyle() {
		if (!containsKey(Attrs.style)) {
			setStyle("color: #3E576F; fontSize: 16px;");
		}
		return Generics.cast(getAttr(Attrs.style));
	}

	/**
	 * Sets CSS styles for the title. Use this for font styling, but use align,
	 * x and y for text alignment.
	 * <p>
	 * Defaults to:
	 * 
	 * <pre>
	 * color: #3E576F; fontSize: 16px
	 * </pre>
	 */
	public void setStyle(String style) {
		setStyle(MapsHelper.parse(new LinkedHashMap<String, String>(), style, ':', ';', '\''));
	}

	public <K, V> void setStyle(Map<K, V> style) {
		setAttr(Attrs.style, style, NOT_NULL_VALUE);
	}

	/**
	 * Returns the title of the chart. To disable the title, set the text to
	 * null.
	 * <p>
	 * Defaults to <code>"Chart title</code>.
	 */
	public String getText() {
		return getAttr(Attrs.text, "Chart title").asString();
	}

	/**
	 * Sets the title of the chart. To disable the title, set the text to null.
	 * <p>
	 * Defaults to <code>"Chart title</code>.
	 */
	public void setText(String text) {
		setAttr(Attrs.text, text, "Chart title");
	}

	/**
	 * Returns whether to <a
	 * href="http://docs.highcharts.com/#formatting$html">use HTML</a> to render
	 * the text.
	 * <p>
	 * Default: false.
	 * 
	 * @return useHTML whether to use HTML to render the text
	 */
	public boolean isUseHTML() {
		return getAttr(Attrs.useHTML, false).asBoolean();
	}

	/**
	 * Sets whether to <a href="http://docs.highcharts.com/#formatting$html">use
	 * HTML</a> to render the text.
	 * 
	 * @param useHTML
	 *            whether to use HTML to render the text
	 */
	public void setUseHTML(boolean useHTML) {
		setAttr(Attrs.useHTML, useHTML);
	}

	/**
	 * Returns the vertical alignment of the title. Can be one of "top",
	 * "middle" and "bottom". When a value is given, the title behaves as
	 * floating.
	 * <p>
	 * Defaults to "top".
	 */
	public String getVerticalAlign() {
		return getAttr(Attrs.verticalAlign, "top").asString();
	}

	/**
	 * Sets the vertical alignment of the title. Can be one of "top", "middle"
	 * and "bottom". When a value is given, the title behaves as floating.
	 * <p>
	 * Defaults to "top".
	 */
	public void setVerticalAlign(String verticalAlign) {
		if (!("top".equals(verticalAlign) || "middle".equals(verticalAlign) || "bottom"
				.equals(verticalAlign)))
			throw new IllegalArgumentException("Unsupported vertical align: ["
					+ verticalAlign + "]");
		setAttr(Attrs.verticalAlign, verticalAlign, "top");
	}

	/**
	 * Returns the x position of the title relative to the alignment within
	 * {@link Chart#setSpacingLeft(Number)} and {@link Chart#setSpacingRight(Number)}.
	 * <p>
	 * Defaults to 0.
	 */
	public Number getX() {
		return getAttr(Attrs.x, 0).asNumber();
	}

	/**
	 * Sets the x position of the title relative to the alignment within
	 * {@link Chart#setSpacingLeft(Number)} and {@link Chart#setSpacingRight(Number)}.
	 * <p>
	 * Defaults to 0.
	 */
	public void setX(Number x) {
		setAttr(Attrs.x, x, 0);
	}

	/**
	 * Returns the y position of the title relative to the alignment within
	 * {@link Chart#setSpacingTop(Number)} and
	 * {@link Chart#setSpacingBottom(Number)}.
	 * <p>
	 * Defaults to 15.
	 */
	public Number getY() {
		return getAttr(Attrs.y, 15).asNumber();
	}

	/**
	 * Sets the y position of the title relative to the alignment within
	 * {@link Chart#setSpacingTop(Number)} and
	 * {@link Chart#setSpacingBottom(Number)}.
	 * <p>
	 * Defaults to 15.
	 */
	public void setY(Number y) {
		setAttr(Attrs.y, y, 15);
	}
}