/** Subtitle.java.

	Purpose:
		
	Description:
		
	History:
		9:24:43 AM Jan 13, 2014, Created by RaymondChao

Copyright (C) 2014 Potix Corporation. All Rights Reserved.
 */
package org.zkoss.chart;

import java.util.LinkedHashMap;
import java.util.Map;

import org.zkoss.chart.util.DynamicalAttribute;
import org.zkoss.chart.util.MapsHelper;
import org.zkoss.lang.Generics;

/**
 * The chart's subtitle
 * <p>
 * All the options in this class support {@link DynamicalAttribute}.
 * 
 * @author jumperchen
 */
public class Subtitle extends Optionable {
	protected enum Attrs implements PlotAttribute, DynamicalAttribute {
		align, floating, style, text, useHTML, verticalAlign, x, y
	}

	/**
	 * Returns the horizontal alignment of the subtitle. Can be one of "left",
	 * "center" and "right".
	 * <p>
	 * Default: "center".
	 * 
	 * @return align the horizontal alignment of the subtitle
	 */
	public String getAlign() {
		return getAttr(Attrs.align, "center").asString();
	}

	/**
	 * Sets the horizontal alignment of the subtitle. Can be one of "left",
	 * "center" and "right".
	 * 
	 * @param align
	 *            the horizontal alignment of the subtitle
	 */
	public void setAlign(String align) {
		if (!("left".equals(align) || "center".equals(align) || "right"
				.equals(align)))
			throw new IllegalArgumentException("Unsupported align: [" + align
					+ "]");
		setAttr(Attrs.align, align, "center");
	}

	/**
	 * Returns when the subtitle is floating, the plot area will not move to
	 * make space for it.
	 * <p>
	 * Default: false.
	 */
	public boolean isFloating() {
		return getAttr(Attrs.floating, false).asBoolean();
	}

	/**
	 * Sets when the subtitle is floating, the plot area will not move to make
	 * space for it.
	 */
	public void setFloating(boolean floating) {
		setAttr(Attrs.floating, floating);
	}

	/**
	 * Returns CSS styles for the title. Exact positioning of the title can be
	 * achieved by changing the margin property, or by adding
	 * <code>position: "absolute"</code> and left and top properties.
	 * <p>
	 * Defaults to:
	 * 
	 * <pre>
	 * color: #3E576F;
	 * </pre>
	 * <p>
	 */
	public <K, V> Map<K, V> getStyle() {
		if (!containsKey(Attrs.style)) {
			setStyle("color: #3E576F;");
		}
		return Generics.cast(getAttr(Attrs.style, null).asValue());
	}

	/**
	 * Sets CSS styles for the title. Exact positioning of the title can be
	 * achieved by changing the margin property, or by adding
	 * <code>position: "absolute"</code> and left and top properties.
	 * <p>
	 * Defaults to:
	 * 
	 * <pre>
	 * color: #3E576F;
	 * </pre>
	 * <p>
	 */
	public void setStyle(String style) {
		setStyle(MapsHelper.parse(new LinkedHashMap<String, String>(), style, ':', ';', '\''));
	}

	/**
	 * Sets CSS styles for the title. Exact positioning of the title can be
	 * achieved by changing the margin property, or by adding
	 * <code>position: "absolute"</code> and left and top properties.
	 * <p>
	 * Defaults to:
	 * 
	 * <pre>
	 * color: #3E576F;
	 * </pre>
	 * <p>
	 */
	public <K, V> void setStyle(Map<K, V> style) {
		setAttr(Attrs.style, style, NOT_NULL_VALUE);
	}

	/**
	 * Returns the subtitle of the chart.
	 * <p>
	 * Default: null.
	 */
	public String getText() {
		return getAttr(Attrs.text, null).asString();
	}

	/**
	 * Sets the subtitle of the chart.
	 * 
	 * @param text
	 *            the subtitle of the chart
	 */
	public void setText(String text) {
		setAttr(Attrs.text, text);
	}

	/**
	 * Returns whether to <a
	 * href="http://docs.highcharts.com/#formatting$html">use HTML</a> to render
	 * the text.
	 * <p>
	 * Default: false.
	 */
	public boolean isUseHTML() {
		return getAttr(Attrs.useHTML, false).asBoolean();
	}

	/**
	 * Sets whether to <a href="http://docs.highcharts.com/#formatting$html">use
	 * HTML</a> to render the text.
	 */
	public void setUseHTML(boolean useHTML) {
		setAttr(Attrs.useHTML, useHTML);
	}

	/**
	 * Returns the vertical alignment of the title. Can be one of "top",
	 * "middle" and "bottom". When a value is given, the title behaves as
	 * floating.
	 * <p>
	 * Default: top.
	 */
	public String getVerticalAlign() {
		return getAttr(Attrs.verticalAlign, "top").asString();
	}

	/**
	 * Sets the vertical alignment of the title. Can be one of "top", "middle"
	 * and "bottom". When a value is given, the title behaves as floating.
	 */
	public void setVerticalAlign(String verticalAlign) {
		if (!("top".equals(verticalAlign) || "middle".equals(verticalAlign) || "bottom"
				.equals(verticalAlign)))
			throw new IllegalArgumentException("Unsupported vertical align: ["
					+ verticalAlign + "]");
		setAttr(Attrs.verticalAlign, verticalAlign, "top");
	}

	/**
	 * Returns the x position of the subtitle relative to the alignment within
	 * {@link Chart#setSpacingLeft(Number)} and
	 * {@link Chart#setSpacingRight(Number)}.
	 * <p>
	 * Default: 0.
	 */
	public Number getX() {
		return getAttr(Attrs.x, 0).asNumber();
	}

	/**
	 * Sets the x position of the subtitle relative to the alignment within
	 * {@link Chart#setSpacingLeft(Number)} and
	 * {@link Chart#setSpacingRight(Number)}.
	 */
	public void setX(Number x) {
		setAttr(Attrs.x, x, 0);
	}

	/**
	 * Returns the y position of the subtitle relative to the alignment within
	 * {@link Chart#setSpacingTop(Number)} and
	 * {@link Chart#setSpacingBottom(Number)}. By default the subtitle is laid
	 * out below the title unless the title is floating.
	 * <p>
	 * Default: null.
	 */
	public Number getY() {
		return getAttr(Attrs.y, null).asNumber();
	}

	/**
	 * Sets the y position of the subtitle relative to the alignment within
	 * {@link Chart#setSpacingTop(Number)} and
	 * {@link Chart#setSpacingBottom(Number)}. By default the subtitle is laid
	 * out below the title unless the title is floating.
	 */
	public void setY(Number y) {
		setAttr(Attrs.y, y);
	}

}