/** State.java.

	Purpose:
		
	Description:
		
	History:
		12:46:14 PM Feb 26, 2014, Created by jumperchen

Copyright (C) 2014 Potix Corporation. All Rights Reserved.
 */
package org.zkoss.chart;

import org.zkoss.chart.util.DynamicalAttribute;

/**
 * A state options.
 * <p>
 * All the options in this class support {@link DynamicalAttribute}.
 * 
 * @author jumperchen
 */
public class State extends Optionable {

	private enum Attrs implements PlotAttribute, DynamicalAttribute {
		enabled, fillColor, lineColor, lineWidth, radius, marker, brightness
	}

	private String _name;
	
	/**
	 * Constructs with a name. Like "hover" or "select"
	 */
	public State(String name) {
		_name = name;
	}
	
	/**
	 * Returns the name of this state. Like "hover" or "select"
	 */
	public String getName() {
		return _name;
	}
	/**
	 * Returns enable or disable visible feedback for selection.
	 * 
	 * @return enabled enable or disable visible feedback for selection.
	 */
	public boolean isEnabled() {
		return getAttr(Attrs.enabled, true).asBoolean();
	}

	/**
	 * Sets enable or disable visible feedback for selection. 'enabled enable or
	 * disable visible feedback for selection.
	 */
	public void setEnabled(boolean enabled) {
		setAttr(Attrs.enabled, enabled);
	}

	/**
	 * Returns the fill color of the point marker.
	 * <p>
	 * Default: null.
	 */
	public Color getFillColor() {
		return (Color) getAttr(Attrs.fillColor, null).asValue();
	}

	/**
	 * Sets the fill color of the point marker.
	 */
	public void setFillColor(Color color) {
		setAttr(Attrs.fillColor, color);
	}

	/**
	 * Sets the fill color of the point marker.
	 */
	public void setFillColor(String color) {
		setFillColor(new Color(color));
	}

	/**
	 * Sets the fill color of the point marker.
	 */
	public void setFillColor(LinearGradient color) {
		setFillColor(new Color(color));
	}

	/**
	 * Sets the fill color of the point marker.
	 */
	public void setFillColor(RadialGradient color) {
		setFillColor(new Color(color));
	}

	/**
	 * Returns the color of the point marker's outline. When <code>null</code>,
	 * the series' or point's color is used.
	 * <p>
	 * Default: "#000000"
	 */
	public Color getLineColor() {
		if (!containsKey(Attrs.lineColor)) {
			setLineColor("#000000");
		}
		return (Color) getAttr(Attrs.lineColor);
	}

	/**
	 * Sets the color of the point marker's outline. When <code>null</code>, the
	 * series' or point's color is used.
	 */
	public void setLineColor(Color color) {
		setAttr(Attrs.lineColor, color, NOT_NULL_VALUE);
	}

	/**
	 * Sets the color of the point marker's outline. When <code>null</code>, the
	 * series' or point's color is used.
	 */
	public void setLineColor(String color) {
		setLineColor(new Color(color));
	}

	/**
	 * Sets the color of the point marker's outline. When <code>null</code>, the
	 * series' or point's color is used.
	 */
	public void setLineColor(LinearGradient color) {
		setLineColor(new Color(color));
	}

	/**
	 * Sets the color of the point marker's outline. When <code>null</code>, the
	 * series' or point's color is used.
	 */
	public void setLineColor(RadialGradient color) {
		setLineColor(new Color(color));
	}

	/**
	 * Returns the width of the point marker's outline.
	 */
	public Number getLineWidth() {
		return getAttr(Attrs.lineWidth, 0).asNumber();
	}

	/**
	 * Sets the width of the point marker's outline.
	 */
	public void setLineWidth(Number lineWidth) {
		setAttr(Attrs.lineWidth, lineWidth, NOT_NULL_VALUE);
	}

	/**
	 * Returns the radius of the point marker. In hover state, it defaults to
	 * the normal state's radius + 2.
	 */
	public Number getRadius() {
		return getAttr(Attrs.radius, null).asNumber();
	}

	/**
	 * Sets the radius of the point marker. In hover state, it defaults to the
	 * normal state's radius + 2.
	 */
	public void setRadius(Number radius) {
		setAttr(Attrs.radius, radius);
	}
	
	/**
	 * Sets the brightness number
	 */
	public void setBrightness(Number brightness) {
		setAttr(Attrs.brightness, brightness, NOT_NULL_VALUE);
	}

	/**
	 * Returns the brightness number
	 */
	public Number getBrightness(Number brightness) {
		return getAttr(Attrs.brightness, null).asNumber();
	}

	/**
	 * Returns the marker of this state
	 */
	public Marker getMarker() {
		Marker marker = (Marker) getAttr(Attrs.marker);
		if (marker == null) {
			marker = new Marker();
			setMarker(marker);
		}
		return marker;
	}

	/**
	 * Sets the marker of this state
	 * @param marker
	 */
	public void setMarker(Marker marker) {
		setAttr(Attrs.marker, marker);
	}
}
