/** StackLabels.java.

	Purpose:
		
	Description:
		
	History:
		9:23:13 AM Jan 13, 2014, Created by RaymondChao

Copyright (C) 2014 Potix Corporation. All Rights Reserved.
 */

package org.zkoss.chart;

import java.util.LinkedHashMap;
import java.util.Map;

import org.zkoss.chart.util.DynamicalAttribute;
import org.zkoss.chart.util.MapsHelper;
import org.zkoss.lang.Generics;

/**
 * The stack labels show the total value for each bar in a stacked column or bar
 * chart. The label will be placed on top of positive columns and below negative
 * columns. In case of an inverted column chart or a bar chart the label is
 * placed to the right of positive bars and to the left of negative bars.
 * 
 * <p>
 * All the options in this class support {@link DynamicalAttribute}.
 * 
 * @author jumperchen
 * @author RaymondChao
 */
public class StackLabels extends Optionable {
	private enum Attrs implements PlotAttribute, DynamicalAttribute {
		align,
		enabled,
		format,
		formatter,
		rotation,
		style,
		textAlign,
		useHTML,
		verticalAlign,
		x,
		y
	}

	/**
	 * Returns the horizontal alignment of the stack total label. Can be one of
	 * <code>"left"</code>, <code>"center"</code> or <code>"right"</code>. The
	 * default value is calculated at runtime and depends on orientation and
	 * whether the stack is positive or negative.
	 * <p>
	 * Default: null.
	 */
	public String getAlign() {
		return getAttr(Attrs.align, null).asString();
	}

	/**
	 * Sets the horizontal alignment of the stack total label. Can be one of
	 * <code>"left"</code>, <code>"center"</code> or <code>"right"</code>. The
	 * default value is calculated at runtime and depends on orientation and
	 * whether the stack is positive or negative.
	 */
	public void setAlign(String align) {
		setAttr(Attrs.align, align);
	}

	/**
	 * Returns whether to enable or disable the stack total labels.
	 * <p>
	 * Default: false.
	 */
	public boolean isEnabled() {
		return getAttr(Attrs.enabled, false).asBoolean();
	}

	/**
	 * Sets whether to enable or disable the stack total labels.
	 */
	public void setEnabled(boolean enabled) {
		setAttr(Attrs.enabled, enabled);
	}

	/**
	 * Returns a <a href="http://docs.highcharts.com/#formatting">format
	 * string</a> for the data label. Available variables are the same as for
	 * <code>formatter</code>.
	 * <p>
	 * Default: "{total}".
	 */
	public String getFormat() {
		return getAttr(Attrs.format, "{total}").asString();
	}

	/**
	 * Sets a <a href="http://docs.highcharts.com/#formatting">format string</a>
	 * for the data label. Available variables are the same as for
	 * <code>formatter</code>.
	 * <p>
	 * Default: "{total}".
	 */
	public void setFormat(String format) {
		setAttr(Attrs.format, format, "{total}");
	}

	/**
	 * Returns rotation of the labels in degrees.
	 * <p>
	 * Default: 0.
	 */
	public Number getRotation() {
		return getAttr(Attrs.rotation, 0).asNumber();
	}

	/**
	 * Sets rotation of the labels in degrees.
	 */
	public void setRotation(Number rotation) {
		setAttr(Attrs.rotation, rotation, 0);
	}

	/**
	 * Returns CSS styles for the label.
	 * <p>
	 * Defaults to:
	 * 
	 * <pre>
	 * color: #666; font-size: 11px; line-height: 14px;
	 * </pre>
	 */
	public <K, V> Map<K, V> getStyle() {
		if (!containsKey(Attrs.style)) {
			setStyle("color: #666; font-size: 11px; line-height: 14px;");
		}
		return Generics.cast(getAttr(Attrs.style));
	}

	/**
	 * Sets CSS styles for the label.
	 * <p>
	 * Defaults to:
	 * 
	 * <pre>
	 * color: #666; font-size: 11px; line-height: 14px;
	 * </pre>
	 */
	public void setStyle(String style) {
		setStyle(MapsHelper.parse(new LinkedHashMap<String, String>(), style, ':', ';', '\''));
	}

	/**
	 * Sets CSS styles for the label.
	 * <p>
	 * Defaults to:
	 * 
	 * <pre>
	 * color: #666; font-size: 11px; line-height: 14px;
	 * </pre>
	 */
	public <K, V> void setStyle(Map<K, V> style) {
		setAttr(Attrs.style, style, NOT_NULL_VALUE);
	}

	/**
	 * Returns the text alignment for the label. While <code>align</code>
	 * determines where the texts anchor point is placed with regards to the
	 * stack, <code>textAlign</code> determines how the text is aligned against
	 * its anchor point. Possible values are <code>"left"</code>,
	 * <code>"center"</code> and <code>"right"</code>. The default value is
	 * calculated at runtime and depends on orientation and whether the stack is
	 * positive or negative.
	 * <p>
	 * Default: null.
	 */
	public String getTextAlign() {
		return getAttr(Attrs.textAlign, null).asString();
	}

	/**
	 * Sets the text alignment for the label. While <code>align</code>
	 * determines where the texts anchor point is placed with regards to the
	 * stack, <code>textAlign</code> determines how the text is aligned against
	 * its anchor point. Possible values are <code>"left"</code>,
	 * <code>"center"</code> and <code>"right"</code>. The default value is
	 * calculated at runtime and depends on orientation and whether the stack is
	 * positive or negative.
	 * <p>
	 * Default: null.
	 */
	public void setTextAlign(String textAlign) {
		if (!("left".equals(textAlign) || "center".equals(textAlign) || "right"
				.equals(textAlign)))
			throw new IllegalArgumentException("Unsupported textAlign: ["
					+ textAlign + "]");
		setAttr(Attrs.textAlign, textAlign);
	}

	/**
	 * Returns whether to <a
	 * href="http://docs.highcharts.com/#formatting$html">use HTML</a> to render
	 * the labels.
	 * <p>
	 * Default: false.
	 */
	public boolean isUseHTML() {
		return getAttr(Attrs.useHTML, false).asBoolean();
	}

	/**
	 * Sets whether to <a href="http://docs.highcharts.com/#formatting$html">use
	 * HTML</a> to render the labels.
	 */
	public void setUseHTML(boolean useHTML) {
		setAttr(Attrs.useHTML, useHTML);
	}

	/**
	 * Returns defines the vertical alignment of the stack total label. Can be
	 * one of <code>"top"</code>, <code>"middle"</code> or <code>"bottom"</code>
	 * . The default value is calculated at runtime and depends on orientation
	 * and whether the stack is positive or negative.
	 * <p>
	 * Default: null.
	 */
	public String getVerticalAlign() {
		return getAttr(Attrs.verticalAlign, null).asString();
	}

	/**
	 * Sets defines the vertical alignment of the stack total label. Can be one
	 * of <code>"top"</code>, <code>"middle"</code> or <code>"bottom"</code>.
	 * The default value is calculated at runtime and depends on orientation and
	 * whether the stack is positive or negative.
	 */
	public void setVerticalAlign(String verticalAlign) {
		if (!("top".equals(verticalAlign) || "middle".equals(verticalAlign) || "bottom"
				.equals(verticalAlign)))
			throw new IllegalArgumentException("Unsupported vertical align: ["
					+ verticalAlign + "]");
		setAttr(Attrs.verticalAlign, verticalAlign);
	}

	/**
	 * Returns the x position offset of the label relative to the left of the
	 * stacked bar. The default value is calculated at runtime and depends on
	 * orientation and whether the stack is positive or negative.
	 * <p>
	 * Default: null.
	 */
	public Number getX() {
		return getAttr(Attrs.x, null).asNumber();
	}

	/**
	 * Sets the x position offset of the label relative to the left of the
	 * stacked bar. The default value is calculated at runtime and depends on
	 * orientation and whether the stack is positive or negative.
	 */
	public void setX(Number x) {
		setAttr(Attrs.x, x);
	}

	/**
	 * Returns the y position offset of the label relative to the tick position
	 * on the axis. The default value is calculated at runtime and depends on
	 * orientation and whether the stack is positive or negative.
	 * <p>
	 * Default: null.
	 */
	public Number getY() {
		return getAttr(Attrs.y, null).asNumber();
	}

	/**
	 * Sets the y position offset of the label relative to the tick position on
	 * the axis. The default value is calculated at runtime and depends on
	 * orientation and whether the stack is positive or negative.
	 */
	public void setY(Number y) {
		setAttr(Attrs.y, y);
	}

}