/** Series.java.

	Purpose:
		
	Description:
		
	History:
		2:06:58 PM Jan 9, 2014, Created by jumperchen

Copyright (C) 2014 Potix Corporation. All Rights Reserved.
 */
package org.zkoss.chart;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.zkoss.chart.OptionDataEvent.EventType;
import org.zkoss.chart.plotOptions.DataLabels;
import org.zkoss.chart.plotOptions.SeriesPlotOptions;
import org.zkoss.chart.util.AnyVal;
import org.zkoss.chart.util.DeferredCall;
import org.zkoss.chart.util.DynamicalAttribute;
import org.zkoss.chart.util.JSFunction;
import org.zkoss.json.JSONObject;
import org.zkoss.lang.Generics;
import org.zkoss.lang.Objects;

/**
 * The Series object is the JavaScript representation of each line, area series,
 * pie etc. The object can be accessed in a number of ways. All series and point
 * event handlers give a reference to the series object. The chart object has a
 * series property that is a collection of all the chart's series. The point
 * objects also have the same reference. Another way to reference the series
 * programmatically is by id.
 * <p>
 * All the options in this class support {@link DynamicalAttribute}.
 * 
 * @author jumperchen
 */
public class Series extends Optionable implements OptionDataListener {
	private enum DAttrs implements PlotAttribute, DynamicalAttribute {
		color, dataLabels,
		// dataParser, Deprecated
		// dataURL, Deprecated
		index,
		legendIndex,
		marker,
		name,
		stack,
		type,
		xAxis,
		yAxis,
		zIndex,
		id
	}

	private enum Attrs implements PlotAttribute {
		data, visible,

		// internal use
		point
	}

	private SeriesPlotOptions plotOptions;

	public Series() {
	}

	/**
	 * Constructs the series with id and a list of point data.
	 */
	public Series(String id, List<Point> data) {
		setId(id);
		setData(data.toArray(new Point[0]));
	}

	/**
	 * Constructs the series with id and point data.
	 */
	public Series(String id, Point... data) {
		setId(id);
		setData(data);
	}

	/**
	 * Returns an id for the series.
	 * <p>
	 * Default: null.
	 */
	public String getId() {
		return getAttr(DAttrs.id, null).asString();
	}

	/**
	 * Sets an id for the series.
	 */
	public void setId(String id) {
		setAttr(DAttrs.id, id);
	}

	/**
	 * Sets the plot options to the series.
	 */
	public void setPlotOptions(SeriesPlotOptions plotOptions) {
		SeriesPlotOptions old = this.plotOptions;
		if (!Objects.equals(old, plotOptions)) {
			if (old != null)
				old.removeOptionDataListener(this);

			if (plotOptions != null) {
				plotOptions.addOptionDataListener(this);
				this.plotOptions = plotOptions;
			}
		}
	}

	/**
	 * Returns the plot options to the series. If null, it will create a new
	 * one.
	 */
	public SeriesPlotOptions getPlotOptions() {
		if (plotOptions == null) {
			plotOptions = new SeriesPlotOptions();
			setPlotOptions(plotOptions);
		}
		return plotOptions;
	}

	/**
	 * Add a point at the end of the current points list.
	 * 
	 * @param point
	 */
	public void addPoint(Point point) {
		addPoint(point, true, false, true);
	}

	/**
	 * Add a point to the end of the current points list.
	 * 
	 * @param point
	 *            the point
	 * @param redraw
	 *            whether to redraw the chart
	 * @param shift
	 *            If shift is true, a point is shifted off the start of the
	 *            series as one is appended to the end.
	 * @param animation
	 *            Whether to apply animation, and optionally animation
	 *            configuration
	 */
	public void addPoint(Point point, boolean redraw, boolean shift,
			boolean animation) {
		addPoint(point, redraw, shift, animation ? new Animation()
				: Animation.NONE);
	}

	/**
	 * Add a point at the end of the current points list.
	 * 
	 * @param point
	 *            the point
	 * @param redraw
	 *            whether to redraw the chart
	 * @param shift
	 *            If shift is true, a point is shifted off the start of the
	 *            series as one is appended to the end.
	 * @param animation
	 *            Whether to apply animation, and optionally animation
	 *            configuration
	 */
	public void addPoint(final Point point, final boolean redraw,
			final boolean shift, final Animation animation) {
		List<Point> list = getData();
		if (list == null)
			list = new LinkedList<Point>();
		list.add(point);

		// monitor if data changed.
		point.addOptionDataListener(this);
		point.setSeries(this);
		setAttr(Attrs.data, list);

		fireEvent(EventType.ADDED, Attrs.point.toString(), point,
				new DeferredCall() {
					public void execute(JSFunction func) {
						func.callFunction("addPoint", point, redraw, shift,
								animation);
					}
				});
	}

	/**
	 * Add a number point to the end of the current points list.
	 */
	public void addPoint(Number y) {
		if (y == null)
			addPoint(new Point());
		else
			addPoint(new Point(y));
	}

	/**
	 * Add a number point to the end of the current points list.
	 */
	public void addPoint(double y) {
		addPoint(new Point(y));
	}

	/**
	 * Add a point with x and y numbers to the end of the current points list.
	 */
	public void addPoint(Number x, Number y) {
		addPoint(new Point(x, y));
	}

	/**
	 * Add a point with x and y numbers to the end of the current points list.
	 */
	public void addPoint(double x, double y) {
		addPoint(new Point(x, y));
	}

	/**
	 * Add a point with x, low, and high numbers to the end of the current
	 * points list.
	 */
	public void addPoint(Number x, Number low, Number high) {
		addPoint(new Point(x, low, high));
	}

	/**
	 * Add a point with x, low, and high numbers to the end of the current
	 * points list.
	 */
	public void addPoint(double x, double low, double high) {
		addPoint(new Point(x, low, high));
	}

	/**
	 * Add a point with name and number to the end of the current points list.
	 */
	public void addPoint(String name, Number y) {
		addPoint(new Point(name, y));
	}

	/**
	 * Add a point with name and number to the end of the current points list.
	 */
	public void addPoint(String name, double y) {
		addPoint(new Point(name, y));
	}

	/**
	 * Sets a list of point data
	 */
	public <T extends Number> void setData(List<T> data) {
		Point[] points = new Point[data.size()];
		int i = 0;
		for (T t : data) {
			if (t == null)
				points[i++] = new Point();
			else
				points[i++] = new Point(t);
		}
		setData(points);
	}

	/**
	 * Sets an array of point data
	 */
	public void setData(Point... data) {
		for (Point p : data) {
			// monitor if data changed.
			p.addOptionDataListener(this);
			p.setSeries(this);
		}

		// use ArrayList instead of fixed ArrayList
		final List<Point> list = new ArrayList<Point>(Arrays.asList(data));
		setAttr(Attrs.data, list);

		fireEvent(EventType.CHANGED, Attrs.data.toString(), data,
				new DeferredCall() {
					public void execute(JSFunction func) {
						func.callFunction("setData", list);
					}
				});
	}

	/**
	 * Sets an array of number data
	 */
	public void setData(Number... data) {
		setData(Arrays.asList(data));
	}

	/**
	 * Sets an array of number data
	 */
	public void setData(Double... data) {
		setData(Arrays.asList(data));
	}

	/**
	 * Sets an array of number data
	 */
	public void setData(Integer... data) {
		setData(Arrays.asList(data));
	}

	/**
	 * Returns the list of point data
	 */
	public List<Point> getData() {
		return Generics.cast(getAttr(Attrs.data, null).asValue());
	}

	/**
	 * Returns the point in the list from the given index.
	 */
	public Point getPoint(int index) {
		return getData().get(index);
	}

	/**
	 * Sets the index of the series in the chart, affecting the internal index
	 * in the chart.series array, the visible Z index as well as the order in
	 * the legend.
	 */
	public void setIndex(int index) {
		if (index < 0) {
			removeKey(DAttrs.index, true);
		} else {
			setAttr(DAttrs.index, index);
		}
	}

	/**
	 * Returns the index of the series in the chart, affecting the internal
	 * index in the chart.series array, the visible Z index as well as the order
	 * in the legend.
	 * <p>
	 * Default: -1, it means depended on the implementaion.
	 */
	public int getIndex() {
		return getAttr(DAttrs.index, -1).asInt();
	}

	/**
	 * Sets the option of data labels.
	 * 
	 * @param dataLabels
	 * @see SeriesPlotOptions#setDataLabels(DataLabels)
	 */
	public void setDataLabels(DataLabels dataLabels) {
		getPlotOptions().setDataLabels(dataLabels);
	}

	/**
	 * Returns the option of data labels.
	 * 
	 * @see SeriesPlotOptions#getDataLabels()
	 */
	public DataLabels getDataLabels() {
		return getPlotOptions().getDataLabels();
	}

	/**
	 * Returns individual color for the series. By default the color is pulled
	 * from the global <code>colors</code> array.
	 * <p>
	 * Default: null.
	 */
	public Color getColor() {
		return (Color) getAttr(DAttrs.color, null).asValue();
	}

	/**
	 * Sets individual color for the series. By default the color is pulled from
	 * the global <code>colors</code> array.
	 */
	public void setColor(Color color) {
		setAttr(DAttrs.color, color);
	}

	/**
	 * Sets individual color for the series. By default the color is pulled from
	 * the global <code>colors</code> array.
	 */
	public void setColor(String color) {
		setColor(new Color(color));
	}

	/**
	 * Sets individual color for the series. By default the color is pulled from
	 * the global <code>colors</code> array.
	 */
	public void setColor(LinearGradient color) {
		setColor(new Color(color));
	}

	/**
	 * Sets individual color for the series. By default the color is pulled from
	 * the global <code>colors</code> array.
	 */
	public void setColor(RadialGradient color) {
		setColor(new Color(color));
	}

	/**
	 * Sets the sequential index of the series in the legend.
	 */
	public void setLegendIndex(int legendIndex) {
		setAttr(DAttrs.legendIndex, legendIndex);
	}

	/**
	 * Returns the sequential index of the series in the legend.
	 */
	public int getLegendIndex() {
		return getAttr(DAttrs.legendIndex, -1).asInt();
	}

	/**
	 * Returns the marker for this series.
	 */
	public Marker getMarker() {
		Marker marker = (Marker) getAttr(DAttrs.marker);
		if (marker == null) {
			marker = new Marker();
			setAttr(DAttrs.marker, marker);
		}
		return marker;
	}

	/**
	 * Sets the marker for this series.
	 */
	public void setMarker(Marker marker) {
		setAttr(DAttrs.marker, marker);
	}

	/**
	 * Sets the name of the series as shown in the legend, tooltip etc.
	 */
	public void setName(String name) {
		setAttr(DAttrs.name, name);
	}

	/**
	 * Returns the name of the series as shown in the legend, tooltip etc.
	 */
	public String getName() {
		return getAttr(DAttrs.name, null).asString();
	}

	/**
	 * Sets the option allows grouping series in a stacked chart. The stack
	 * option can be a string or a number or anything else, as long as the
	 * grouped series' stack options match each other.
	 */
	public void setStack(String stack) {
		setAttr(DAttrs.stack, stack);
	}

	/**
	 * Returns the option allows grouping series in a stacked chart. The stack
	 * option can be a string or a number or anything else, as long as the
	 * grouped series' stack options match each other.
	 */
	public String getStack() {
		return getAttr(DAttrs.stack, null).asString();
	}

	/**
	 * The type of series. Can be one of area, areaspline, bar, column, line,
	 * pie, scatter, spline, arearange, areasplinerange and columnrange.
	 */
	public void setType(String type) {
		setAttr(DAttrs.type, type);
	}

	/**
	 * The type of series. Can be one of area, areaspline, bar, column, line,
	 * pie, scatter, spline, arearange, areasplinerange and columnrange.
	 */
	public String getType() {
		return getAttr(DAttrs.type, null).asString();
	}

	/**
	 * When using dual or multiple x axes, this number defines which xAxis the
	 * particular series is connected to. It refers to either the axis id or the
	 * index of the axis in the xAxis array, with 0 being the first. Defaults to
	 * 0.
	 */
	public void setXAxis(int xAxis) {
		setAttr(DAttrs.xAxis, xAxis);
	}

	/**
	 * When using dual or multiple x axes, this number defines which xAxis the
	 * particular series is connected to. It refers to either the axis id or the
	 * index of the axis in the xAxis array, with 0 being the first. Defaults to
	 * 0.
	 */
	public void setXAxis(String xAxis) {
		setAttr(DAttrs.xAxis, xAxis);
	}

	/**
	 * When using dual or multiple x axes, this number defines which xAxis the
	 * particular series is connected to. It refers to either the axis id or the
	 * index of the axis in the xAxis array, with 0 being the first. Defaults to
	 * 0.
	 */
	public Object getXAxis() {
		return getAttr(DAttrs.xAxis, 0).asValue();
	}

	/**
	 * When using dual or multiple y axes, this number defines which yAxis the
	 * particular series is connected to. It refers to either the axis id or the
	 * index of the axis in the yAxis array, with 0 being the first. Defaults to
	 * 0.
	 */
	public void setYAxis(int yAxis) {
		setAttr(DAttrs.yAxis, yAxis);
	}

	/**
	 * When using dual or multiple y axes, this number defines which yAxis the
	 * particular series is connected to. It refers to either the axis id or the
	 * index of the axis in the yAxis array, with 0 being the first. Defaults to
	 * 0.
	 */
	public void setYAxis(String yAxis) {
		setAttr(DAttrs.yAxis, yAxis);
	}

	/**
	 * When using dual or multiple y axes, this number defines which yAxis the
	 * particular series is connected to. It refers to either the axis id or the
	 * index of the axis in the yAxis array, with 0 being the first. Defaults to
	 * 0.
	 */
	public Object getYAxis() {
		return getAttr(DAttrs.yAxis, 0).asValue();
	}

	/**
	 * Define the visual z index of the series.
	 */
	public void setZIndex(int zIndex) {
		setAttr(DAttrs.zIndex, zIndex);
	}

	/**
	 * Define the visual z index of the series.
	 */
	public int getZIndex() {
		return getAttr(DAttrs.zIndex, -1).asInt();
	}

	/**
	 * Hides the series if visible. If the {@link Chart#isIgnoreHiddenSeries()}
	 * option is true,the chart is redrawn without this series.
	 */
	public void hide() {
		setAttr(Attrs.visible, false);
		fireEvent(EventType.CHANGED, "hide", this, new DeferredCall() {
			public void execute(JSFunction func) {
				func.callFunction("hide");
			}
		});
	}

	/**
	 * Returns whether the series is visible or not.
	 * <p>
	 * Default: true
	 */
	public boolean isVisible() {
		return getAttr(Attrs.visible, true).asBoolean();
	}

	/**
	 * Sets whether the serries is visible.
	 */
	public void setVisible(final boolean visible) {
		if (setAttr(Attrs.visible, visible, true)) {
			fireEvent(EventType.CHANGED, "visible", this, new DeferredCall() {
				public void execute(JSFunction func) {
					func.callFunction("setVisible", visible);
				}
			});
		}
	}

	/**
	 * Shows the series if hidden.
	 */
	public void show() {
		setAttr(Attrs.visible, true);
		fireEvent(EventType.CHANGED, "show", this, new DeferredCall() {
			public void execute(JSFunction func) {
				func.callFunction("show");
			}

		});
	}

	public void remove() {
		fireEvent(EventType.DESTROYED, "series", this, new DeferredCall() {
			public void execute(JSFunction func) {
				func.callFunction("remove");
			}
		});
		clearOptonDataListener();
	}

	public void select() {
		fireEvent(EventType.CHANGED, "select", this, new DeferredCall() {
			public void execute(JSFunction func) {
				func.callFunction("select");
			}

		});
	}

	public void onChange(OptionDataEvent event) {
		final Optionable target = event.getTarget();
		event.setCurrentTarget(this);
		if (target instanceof Point) {
			final int index = getData().indexOf(target);
			if (event.hasJSFunctionCall()) {
				event.addJSFunctionCall(new DeferredCall() {
					public void execute(JSFunction func) {
						func.callArray("data", index);
					}

				});
			}
			if (event.getType() == OptionDataEvent.EventType.DESTROYED) {
				List<Point> list = getData();
				if (list != null)
					list.remove(index);
			}
		} else {
			event.addJSFunctionCall(new DeferredCall() {
				public void execute(JSFunction func) {
					if (target instanceof DataLabels) {
						JSONObject json = new JSONObject();
						json.put("dataLabels", target);
						func.callFunction("update", json);
					} else {
						func.callFunction("update", target);
					}
				}
			});
		}

		fireEvent(event);
	}

	public String toJSONString() {
		if (plotOptions != null) {
			// we don't use merge(options) here
			Map<String, AnyVal<Object>> clone = new LinkedHashMap<String, AnyVal<Object>>(
					options);
			clone.putAll(plotOptions.options);
			return JSONObject.toJSONString(clone);
		}
		return super.toJSONString();
	}
}
