/* Point.java

	Purpose:
		
	Description:
		
	History:
		Thu, Jan 16, 2014  2:39:56 PM, Created by RaymondChao

Copyright (C) 2014 Potix Corporation. All Rights Reserved.

 */
package org.zkoss.chart;

import org.zkoss.chart.OptionDataEvent.EventType;
import org.zkoss.chart.plotOptions.DataLabels;
import org.zkoss.chart.plotOptions.GaugeDialPlotOptions;
import org.zkoss.chart.util.DeferredCall;
import org.zkoss.chart.util.DynamicalAttribute;
import org.zkoss.chart.util.JSFunction;
import org.zkoss.chart.util.OptionsList;
import org.zkoss.json.JSONArray;
import org.zkoss.json.JSONObject;
import org.zkoss.json.JavaScriptValue;

/**
 * The Point object can be accessed in a number of ways. In the {@link Series}
 * object all the points are accessed by the {@link Series#setData(Double...)}.
 * <p>
 * All the options in this class support {@link DynamicalAttribute}.
 * 
 * @author jumperchen
 * @author RaymondChao
 */
public class Point extends Optionable {
	private enum Attrs implements PlotAttribute {
		selected,
		sliced
	}
	private enum DAttrs implements PlotAttribute, DynamicalAttribute {
		color,
		dataLabels,
		events,
		id,
		legendIndex,
		marker,
		name,
		selected,
		sliced,
		x,
		y,
		dial,
		drilldown,

		isIntermediateSum,
		isSum,
		visible,

		// extra DAttrs internal use only
		low,
		high

	}

	private boolean _valuesOnly = true;
	private Series _series;

	/**
	 * Default constructor
	 */
	public Point() {
	}

	/* package */void setSeries(Series s) {
		_series = s;
	}

	/* package */Series getSeries() {
		return _series;
	}

	/**
	 * Constructs with a number
	 */
	public Point(Number y) {
		setY(y);
	}

	/**
	 * Constructs with a pair number (x, y)
	 */
	public Point(Number x, Number y) {
		setX(x);
		setY(y);
	}

	/**
	 * Constructs with a name and number
	 * 
	 * @param name
	 * @param y
	 */
	public Point(String name, Number y) {
		setName(name);
		setY(y);
	}

	/**
	 * Constructs with a name, number, and the id of the drilldown data
	 * 
	 * @param name
	 * @param y
	 * @param drilldown
	 */
	public Point(String name, Number y, String drilldown) {
		setName(name);
		setY(y);
		setDrilldown(drilldown);
	}

	/**
	 * Constructs with three numbers, x, low, and high.
	 */
	public Point(Number x, Number low, Number high) {
		setX(x);
		setLow(low);
		setHigh(high);
	}

	/**
	 * Returns the option of data labels.
	 */
	public DataLabels getDataLabels() {
		DataLabels dataLabels = (DataLabels) this.getAttr(DAttrs.dataLabels);

		if (dataLabels == null) {
			dataLabels = new DataLabels();
			setDataLabels(dataLabels);
		}
		return dataLabels;
	}

	/**
	 * Sets the option of data labels.
	 * 
	 * @see DataLabels
	 */
	public void setDataLabels(DataLabels dataLabels) {
		_valuesOnly = false;
		setAttr(DAttrs.dataLabels, dataLabels);
	}

	/**
	 * Returns whether the series is visible or not.
	 * <p>
	 * Default: true
	 */
	public boolean isVisible() {
		return getAttr(DAttrs.visible, true).asBoolean();
	}

	/**
	 * Sets whether the serries is visible.
	 */
	public void setVisible(final boolean visible) {
		_valuesOnly = false;
		setAttr(DAttrs.visible, visible, true);
	}

	/**
	 * Sets the low number, if any.
	 */
	public void setLow(Number low) {
		setAttr(DAttrs.low, low, NOT_NULL_VALUE);
	}

	/**
	 * Returns the low number, if any.
	 * <p>
	 * Default: null
	 */
	public Number getLow() {
		return getAttr(DAttrs.low, null).asNumber();
	}

	/**
	 * Sets the high number, if any.
	 */
	public void setHigh(Number high) {
		setAttr(DAttrs.high, high, NOT_NULL_VALUE);
	}

	/**
	 * Returns the high number, if any.
	 * <p>
	 * Default: null
	 */
	public Number getHigh() {
		return getAttr(DAttrs.high, null).asNumber();
	}

	/**
	 * Returns the dial data for gauge chart type only.
	 */
	public GaugeDialPlotOptions getDial() {
		GaugeDialPlotOptions dial = (GaugeDialPlotOptions) getAttr(DAttrs.dial);
		if (dial == null) {
			dial = new GaugeDialPlotOptions();
			setDial(dial);
		}
		return dial;
	}

	/**
	 * Sets the dial data for gauge chart type only.
	 */
	public void setDial(GaugeDialPlotOptions dial) {
		setAttr(DAttrs.dial, dial);
	}

	/**
	 * Sets to enable the drilldown.
	 */
	public void setDrilldown(boolean enable) {
		setDrilldown(String.valueOf(enable));
	}

	/**
	 * Sets the id of the drilldown data.
	 */
	public void setDrilldown(String seriesId) {
		_valuesOnly = false;
		setAttr(DAttrs.drilldown, seriesId);
	}

	/**
	 * Returns the id of drilldown
	 * <p>
	 * Default: null
	 */
	public String getDrilldown() {
		return getAttr(DAttrs.drilldown, null).asString();
	}


	/**
	 * Returns individual color for the point. By default the color is pulled
	 * from the global <code>colors</code> array.
	 * <p>
	 * Default: null.
	 */
	public Color getColor() {
		return (Color) getAttr(DAttrs.color, null).asValue();
	}

	/**
	 * Sets individual color for the point. By default the color is pulled from
	 * the global <code>colors</code> array.
	 */
	public void setColor(Color color) {
		_valuesOnly = false;
		setAttr(DAttrs.color, color);
	}

	/**
	 * Sets individual color for the point. By default the color is pulled from
	 * the global <code>colors</code> array.
	 */
	public void setColor(String color) {
		setColor(new Color(color));
	}

	/**
	 * Sets individual color for the point. By default the color is pulled from
	 * the global <code>colors</code> array.
	 */
	public void setColor(LinearGradient color) {
		setColor(new Color(color));
	}

	/**
	 * Sets individual color for the point. By default the color is pulled from
	 * the global <code>colors</code> array.
	 */
	public void setColor(RadialGradient color) {
		setColor(new Color(color));
	}

	/**
	 * Returns an id for the point.
	 * <p>
	 * Default: null.
	 */
	public String getId() {
		return getAttr(DAttrs.id, null).asString();
	}

	/**
	 * Sets an id for the point.
	 */
	public void setId(String id) {
		_valuesOnly = false;
		setAttr(DAttrs.id, id);
	}

	/**
	 * Returns waterfall series only. Whether to acts as a summary column for
	 * the values added or subtracted since the last intermediate sum.
	 * <p>
	 * Default: false.
	 */
	public boolean isIntermediateSum() {
		return getAttr(DAttrs.isIntermediateSum, false).asBoolean();
	}

	/**
	 * Sets waterfall series only. Whether to acts as a summary column for the
	 * values added or subtracted since the last intermediate sum.
	 */
	public void setIntermediateSum(boolean sum) {
		_valuesOnly = false;
		setAttr(DAttrs.isIntermediateSum, sum);
	}

	/**
	 * Returns waterfall series only. Whether to display the total sum across
	 * the entire series.
	 * <p>
	 * Default: false.
	 */
	public boolean isSum() {
		return getAttr(DAttrs.isSum, false).asBoolean();
	}

	/**
	 * Sets waterfall series only. Whether to display the total sum across the
	 * entire series.
	 */
	public void setSum(boolean sum) {
		_valuesOnly = false;
		setAttr(DAttrs.isSum, sum);
	}

	/**
	 * Returns pies only. The sequential index of the pie slice in the legend.
	 * <p>
	 * Default: 0.
	 */
	public int getLegendIndex() {
		return getAttr(DAttrs.legendIndex, 0).asInt();
	}

	/**
	 * Sets pies only. The sequential index of the pie slice in the legend.
	 */
	public void setLegendIndex(int legendIndex) {
		_valuesOnly = false;
		setAttr(DAttrs.legendIndex, legendIndex);
	}

	/**
	 * Returns the data marker
	 * <p>
	 * Default: null.
	 */
	public Marker getMarker() {
		Marker marker = (Marker) getAttr(DAttrs.marker);
		if (marker == null) {
			_valuesOnly = false;
			marker = new Marker();
			setAttr(DAttrs.marker, marker);
		}
		return marker;
	}

	/**
	 * Sets the data marker
	 */
	public void setMarker(Marker marker) {
		_valuesOnly = false;
		setAttr(DAttrs.marker, marker);
	}

	/**
	 * Returns the name of the point as shown in the legend, tooltip, dataLabel
	 * etc.
	 * <p>
	 * Default: null.
	 */
	public String getName() {
		return getAttr(DAttrs.name, null).asString();
	}

	/**
	 * Sets the name of the point as shown in the legend, tooltip, dataLabel
	 * etc.
	 */
	public void setName(String name) {
		_valuesOnly = false;
		setAttr(DAttrs.name, name);
	}
	
	/**
	 * Returns whether the point is selected or not.
	 * <p>
	 * Default: false.
	 */
	public boolean isSelected() {
		_valuesOnly = false;
		return getAttr(Attrs.selected, false).asBoolean();
	}

	/**
	 * Sets whether the point is selected or not.
	 */
	public void setSelected(boolean selected) {
		select(selected, false);
	}

	/**
	 * Sets whether the point is selected or not.
	 */
	public void setSelectedDirectly(boolean selected) {
		_valuesOnly = false;
		setAttr(Attrs.selected, selected);
		setSlicedDirectly(selected);
	}

	/**
	 * Returns pie series only. Whether to display a slice offset from the
	 * center.
	 * <p>
	 * Default: false.
	 */
	public boolean isSliced() {
		_valuesOnly = false;
		return getAttr(Attrs.sliced, false).asBoolean();
	}

	/**
	 * Sets pie series only. Whether to display a slice offset from the center.
	 */
	public void setSliced(boolean sliced) {
		slice(sliced);
	}
	
	/**
	 * Sets pie series only. Whether to display a slice offset from the center.
	 */
	public void setSlicedDirectly(boolean sliced) {
		_valuesOnly = false;
		setAttr(Attrs.sliced, sliced);
	}

	/**
	 * Returns the x value of the point.
	 * <p>
	 * Default: null.
	 */
	public Number getX() {
		return (Number) getAttr(DAttrs.x, null).asValue();
	}

	/**
	 * Sets the x value of the point.
	 */
	public void setX(Number x) {
		setAttr(DAttrs.x, x, NOT_NULL_VALUE);
	}

	/**
	 * Returns the y value of the point.
	 * <p>
	 * Default: null.
	 */
	public Number getY() {
		return (Number) getAttr(DAttrs.y, null).asValue();
	}

	/**
	 * Sets the y value of the point.
	 */
	public void setY(Number y) {
		setAttr(DAttrs.y, y, NOT_NULL_VALUE);
	}

	/**
	 * Removes the point from the series
	 */
	public void remove() {
		fireEvent(EventType.DESTROYED, "point", this, new DeferredCall() {
			public void execute(JSFunction func) {
				func.callFunction("remove");
			}

		});
		clearOptonDataListener();
		_series = null;
	}

	/**
	 * Toggle select or unselect the point.
	 */
	public void select() {
		select(!isSelected(), false);
	}
	
	/**
	 * Select or unselect the point.
	 * 
	 * @param select when true, the point is selected. When false, the point is unselected.
	 * @param accumulate when true, the selection is added to other selected points.
	 * When false, other selected points are deselected. Selected points are accumulated on
	 * Control, Shift or Cmd clicking the point.
	 */
	public void select(final boolean select, final boolean accumulate) {
		setSelectedDirectly(select);
		fireEvent(EventType.SELECTED, "accumulate", accumulate, new DeferredCall() {
			public void execute(JSFunction func) {
				func.callFunction("select", select, accumulate, true);
			}
			
		});
	}

	/**
	 * Pie series only. Slice out or set back in a pie chart slice.
	 */
	public void slice() {
		slice(!isSliced());
	}
	
	/**
	 * Pie series only. Slice out or set back in a pie chart slice.
	 */
	public void slice(final boolean sliced) {
		fireEvent(EventType.CHANGED, "slice", this, new DeferredCall() {
			public void execute(JSFunction func) {
				func.callFunction("slice", sliced);
			}

		});
		setSlicedDirectly(sliced);
	}

	/**
	 * Updates with the given values, x, low, and high.
	 * 
	 * @param x
	 * @param low
	 * @param high
	 */
	public void update(final Number x, final Number low, final Number high) {
		setX(x);
		setLow(low);
		setHigh(high);
		OptionDataEvent evt = new OptionDataEvent(this, EventType.CHANGED,
				DAttrs.x.toString(), x, DAttrs.low.toString(), low,
				DAttrs.high.toString(), high);
		evt.addJSFunctionCall(new DeferredCall() {
			public void execute(JSFunction func) {
				func.callFunction(
						"update",
						new JavaScriptValue(JSONArray
								.toJSONString(new Number[] { x, low, high })));
			}
		});
		fireEvent(evt);
	}

	/**
	 * Updates with the given values, x, and y.
	 * 
	 * @param x
	 * @param y
	 */
	public void update(final Number x, final Number y) {
		setX(x);
		setY(y);
		OptionDataEvent evt = new OptionDataEvent(this, EventType.CHANGED,
				DAttrs.x.toString(), x, DAttrs.y.toString(), y);
		evt.addJSFunctionCall(new DeferredCall() {
			public void execute(JSFunction func) {
				func.callFunction(
						"update",
						new JavaScriptValue(JSONArray
								.toJSONString(new Number[] { x, y })));
			}
		});
		fireEvent(evt);
	}

	/**
	 * Updates with the given new value.
	 */
	public void update(final Number value) {
		setY(value);
		fireEvent(EventType.CHANGED, DAttrs.y, value, new DeferredCall() {
			public void execute(JSFunction func) {
				func.callFunction("update", value);
			}
		});
	}

	/**
	 * Encodes this object to a JSON string. It is the same as
	 * {@link #toString()}.
	 */
	@SuppressWarnings("unchecked")
	public String toJSONString() {
		if (options.isEmpty()) {
			return "null";
		} else if (options.size() == 1 && getY() != null) {
			return getY().toString();
		} else if (_valuesOnly) {
			// for backward compatibility
			return OptionsList.toJSONString(options.values());
		} else
			return JSONObject.toJSONString(options);
	}

}