/** PlotLine.java.

	Purpose:
		
	Description:
		
	History:
		6:01:12 PM Jan 13, 2014, Created by RaymondChao

Copyright (C) 2014 Potix Corporation. All Rights Reserved.
 */

package org.zkoss.chart;

import org.zkoss.chart.util.DynamicalAttribute;

/**
 * A plot line stretching across the plot area, marking a specific value on one
 * of the axes.
 * <p>
 * All the options in this class support {@link DynamicalAttribute}.
 * 
 * @author jumperchen
 * @author RaymondChao
 */
public class PlotLine extends Optionable {
	private enum Attrs implements PlotAttribute, DynamicalAttribute {
		color, dashStyle, id, label, value, width, zIndex
	}

	/**
	 * Returns the color of the line.
	 * <p>
	 * Default: null.
	 */
	public Color getColor() {
		return (Color) getAttr(Attrs.color, null).asValue();
	}

	/**
	 * Sets the color of the plot line.
	 */
	public void setColor(Color color) {
		setAttr(Attrs.color, color);
	}

	/**
	 * Sets the color of the plot line.
	 */
	public void setColor(String color) {
		setColor(new Color(color));
	}

	/**
	 * Sets the color of the plot line.
	 */
	public void setColor(LinearGradient color) {
		setColor(new Color(color));
	}

	/**
	 * Sets the color of the plot line.
	 */
	public void setColor(RadialGradient color) {
		setColor(new Color(color));
	}

	/**
	 * Returns the dashing or dot style for the plot line. For possible values
	 * see <a href=
	 * "http://jsfiddle.net/gh/get/jquery/1.7.1/highslide-software/highcharts.com/tree/master/samples/highcharts/plotoptions/series-dashstyle-all/"
	 * > this overview</a>.
	 * <p>
	 * Default: "Solid".
	 */
	public String getDashStyle() {
		return getAttr(Attrs.dashStyle, "Solid").asString();
	}

	/**
	 * Sets the dashing or dot style for the plot line. For possible values see
	 * <a href=
	 * "http://jsfiddle.net/gh/get/jquery/1.7.1/highslide-software/highcharts.com/tree/master/samples/highcharts/plotoptions/series-dashstyle-all/"
	 * > this overview</a>.
	 */
	public void setDashStyle(String dashStyle) {
		if (!("Solid".equalsIgnoreCase(dashStyle) || "ShortDash".equalsIgnoreCase(dashStyle)
				|| "ShortDot".equalsIgnoreCase(dashStyle)
				|| "ShortDashDot".equalsIgnoreCase(dashStyle)
				|| "ShortDashDotDot".equalsIgnoreCase(dashStyle)
				|| "Dot".equalsIgnoreCase(dashStyle) || "Dash".equalsIgnoreCase(dashStyle)
				|| "LongDash".equalsIgnoreCase(dashStyle) || "DashDot".equalsIgnoreCase(dashStyle)
				|| "LongDashDot".equalsIgnoreCase(dashStyle) || "LongDashDotDot"
					.equalsIgnoreCase(dashStyle)))
			throw new IllegalArgumentException("Unsupported style: ["
					+ dashStyle + "]");
		setAttr(Attrs.dashStyle, dashStyle, "Solid");
	}

	/**
	 * Returns an id used for identifying the plot line in
	 * {@link Axis#removePlotLine(String)}
	 * <p>
	 * Default: null.
	 */
	public String getId() {
		return getAttr(Attrs.id, null).asString();
	}

	/**
	 * Sets an id used for identifying the plot line in
	 * {@link Axis#removePlotLine(String)}
	 * <p>
	 * Default: null.
	 */
	public void setId(String id) {
		setAttr(Attrs.id, id);
	}

	/**
	 * Returns text labels for the plot line
	 * <p>
	 * Default: an instance of PlotLabel.
	 */
	public PlotLabel getLabel() {
		PlotLabel label = (PlotLabel) this.getAttr(Attrs.label);

		if (label == null) {
			label = new PlotLabel();
			setLabel(label);
		}
		return label;
	}

	/**
	 * Sets text labels for the plot line
	 */
	public void setLabel(PlotLabel label) {
		setAttr(Attrs.label, label);
	}

	/**
	 * Returns the position of the line in axis units.
	 * <p>
	 * Default: null.
	 */
	public Number getValue() {
		return getAttr(Attrs.value, null).asNumber();
	}

	/**
	 * Sets the position of the line in axis units.
	 */
	public void setValue(Number value) {
		setAttr(Attrs.value, value);
	}

	/**
	 * Returns the width or thickness of the plot line.
	 * <p>
	 * Default: null.
	 */
	public Number getWidth() {
		return getAttr(Attrs.width, null).asNumber();
	}

	/**
	 * Sets the width or thickness of the plot line.
	 */
	public void setWidth(Number width) {
		setAttr(Attrs.width, width);
	}

	/**
	 * Returns the z index of the plot line within the chart.
	 * <p>
	 * Default: null.
	 */
	public Number getZIndex() {
		return getAttr(Attrs.zIndex, null).asNumber();
	}

	/**
	 * Sets the z index of the plot line within the chart.
	 */
	public void setZIndex(Number zIndex) {
		setAttr(Attrs.zIndex, zIndex);
	}

}