/* PlotLabel.java

	Purpose:
		
	Description:
		
	History:
		Fri, Jan 17, 2014 12:03:02 PM, Created by RaymondChao

Copyright (C) 2014 Potix Corporation. All Rights Reserved.

 */
package org.zkoss.chart;

import java.util.LinkedHashMap;
import java.util.Map;

import org.zkoss.chart.util.DynamicalAttribute;
import org.zkoss.chart.util.MapsHelper;
import org.zkoss.lang.Generics;

/**
 * Text labels for the plot bands and plot lines
 * <p>
 * All the options in this class support {@link DynamicalAttribute}.
 * 
 * @author RaymondChao
 * @author jumperchen
 */
public class PlotLabel extends Optionable {
	private enum Attrs implements PlotAttribute, DynamicalAttribute {
		align, rotation, style, text, textAlign, useHTML, verticalAlign, x, y
	}

	/**
	 * Returns horizontal alignment of the label. Can be one of "left", "center"
	 * or "right".
	 * <p>
	 * Default: "center"
	 */
	public String getAlign() {
		return getAttr(Attrs.align, "center").asString();
	}

	/**
	 * Sets horizontal alignment of the label. Can be one of "left", "center" or
	 * "right".
	 */
	public void setAlign(String align) {
		if (!("left".equals(align) || "center".equals(align) || "right"
				.equals(align)))
			throw new IllegalArgumentException("Unsupported align: [" + align
					+ "]");
		setAttr(Attrs.align, align, "center");
	}

	/**
	 * Returns rotation of the text label in degrees.
	 * <p>
	 * Default: 0
	 */
	public Number getRotation() {
		return getAttr(Attrs.rotation, 0).asNumber();
	}

	/**
	 * Sets rotation of the text label in degrees . 'rotation rotation of the
	 * text label in degrees.
	 */
	public void setRotation(Number rotation) {
		setAttr(Attrs.rotation, rotation, 0);
	}

	/**
	 * Returns CSS styles for the text label.
	 */
	public <K, V> Map<K, V> getStyle() {
		return Generics.cast(getAttr(Attrs.style, null).asValue());
	}

	/**
	 * Sets CSS styles for the text label.
	 */
	public void setStyle(String style) {
		setStyle(MapsHelper.parse(new LinkedHashMap<String, String>(), style, ':', ';', '\''));
	}

	/**
	 * Sets CSS styles for the text label.
	 */
	public <K, V> void setStyle(Map<K, V> style) {
		setAttr(Attrs.style, style);
	}

	/**
	 * Returns the string text itself. A subset of HTML is supported.
	 */
	public String getText() {
		return getAttr(Attrs.text, null).asString();
	}

	/**
	 * Sets the string text itself. A subset of HTML is supported.
	 */
	public void setText(String text) {
		setAttr(Attrs.text, text);
	}

	/**
	 * Returns the text alignment for the label. While <code>align</code>
	 * determines where the texts anchor point is placed within the plot band,
	 * <code>textAlign</code> determines how the text is aligned against its
	 * anchor point. Possible values are "left", "center" and "right". Defaults
	 * to the same as the <code>align</code> option.
	 */
	public String getTextAlign() {
		if (containsKey(Attrs.textAlign)) {
			return (String) getAttr(Attrs.textAlign);
		} else {
			return getAlign();
		}
	}

	/**
	 * Sets the text alignment for the label. While <code>align</code>
	 * determines where the texts anchor point is placed within the plot band,
	 * <code>textAlign</code> determines how the text is aligned against its
	 * anchor point. Possible values are "left", "center" and "right". Defaults
	 * to the same as the <code>align</code> option.
	 */
	public void setTextAlign(String textAlign) {
		if (!("left".equals(textAlign) || "center".equals(textAlign) || "right"
				.equals(textAlign)))
			throw new IllegalArgumentException("Unsupported textAlign: ["
					+ textAlign + "]");
		setAttr(Attrs.textAlign, textAlign);
	}

	/**
	 * Returns whether to <a
	 * href="http://docs.highcharts.com/#formatting$html">use HTML</a> to render
	 * the labels.
	 * <p>
	 * Default: false
	 */
	public boolean isUseHTML() {
		return getAttr(Attrs.useHTML, false).asBoolean();
	}

	/**
	 * Sets whether to <a href="http://docs.highcharts.com/#formatting$html">use
	 * HTML</a> to render the labels.
	 * <p>
	 * Default: false
	 */
	public void setUseHTML(boolean useHTML) {
		setAttr(Attrs.useHTML, useHTML);
	}

	/**
	 * Returns vertical alignment of the label relative to the plot band. Can be
	 * one of "top", "middle" or "bottom".
	 * <p>
	 * Default: "top"
	 */
	public String getVerticalAlign() {
		return getAttr(Attrs.verticalAlign, "top").asString();
	}

	/**
	 * Sets vertical alignment of the label relative to the plot band. Can be
	 * one of "top", "middle" or "bottom".
	 */
	public void setVerticalAlign(String verticalAlign) {
		if (!("top".equals(verticalAlign) || "middle".equals(verticalAlign) || "bottom"
				.equals(verticalAlign)))
			throw new IllegalArgumentException("Unsupported verticalAlign: ["
					+ verticalAlign + "]");
		setAttr(Attrs.verticalAlign, verticalAlign, "top");
	}

	/**
	 * Returns horizontal position relative the alignment.
	 * <p>
	 * Default varies by orientation.
	 */
	public Number getX() {
		return getAttr(Attrs.x, null).asNumber();
	}

	/**
	 * Sets horizontal position relative the alignment. Default varies by
	 * orientation.
	 */
	public void setX(Number x) {
		setAttr(Attrs.x, x);
	}

	/**
	 * Returns vertical position of the text baseline relative to the alignment.
	 * <p>
	 * Default varies by orientation.
	 */
	public Number getY() {
		return getAttr(Attrs.y, null).asNumber();
	}

	/**
	 * Sets vertical position of the text baseline relative to the alignment.
	 * Default varies by orientation.
	 */
	public void setY(Number y) {
		setAttr(Attrs.y, y);
	}

}