/** PlotData.java.

	Purpose:
		
	Description:
		
	History:
		12:07:37 PM Jan 9, 2014, Created by jumperchen

Copyright (C) 2014 Potix Corporation. All Rights Reserved.
 */
package org.zkoss.chart;

import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.zkoss.chart.OptionDataEvent.EventType;
import org.zkoss.chart.plotOptions.PlotOptions;
import org.zkoss.chart.util.DeferredCall;
import org.zkoss.chart.util.JSFunction;
import org.zkoss.chart.util.ResponseDataHandler;
import org.zkoss.json.JSONAware;
import org.zkoss.json.JSONObject;
import org.zkoss.zk.au.out.AuInvoke;
import org.zkoss.zk.ui.Executions;
import org.zkoss.zk.ui.UiException;
import org.zkoss.zk.ui.event.EventListener;
import org.zkoss.zk.ui.util.Clients;

/**
 * A plot data used for PlotEngine to generate the result as JSON string.
 * 
 * @author jumperchen
 */
public class PlotData implements JSONAware, OptionDataListener {
	private Map<PlotAttribute, Object> _json = new LinkedHashMap<PlotAttribute, Object>(
			3);
	private Charts charts;

	private enum Attrs implements PlotAttribute {
		series,
		yAxis,
		xAxis,
		pane,
		navigation,
		chart,
		credits,
		labels,
		legend,
		plotOptions,
		subtitle,
		title,
		tooltip,
		loading,
		drilldown,
		colors,
		exporting,
		noData
	}

	public PlotData(Charts owner) {
		charts = owner;
	}
	
	/**
	 * Returns whether to enable the exporting module
	 */
	public Exporting getExporting() {
		Exporting exporting = (Exporting) _json.get(Attrs.exporting);
		if (exporting == null) {
			exporting = new Exporting();
			setExporting(exporting);
		}
		return exporting;
	}
	
	/**
	 * Sets whether to enable the exporting module
	 */
	public void setExporting(Exporting exporting) {
		Exporting old = (Exporting) _json.put(Attrs.exporting, exporting);
		if (old != exporting) {
			addOptionDataListener(exporting);
			removeOptionDataListener(old);
		}
	}
	
	
	/**
	 * Returns the default colors for the chart's series. When all colors are used,
	 * new colors are pulled from the start again.
	 */
	public List<Color> getColors() {
		if (!_json.containsKey(Attrs.colors)) {
			setColors("#2f7ed8", "#0d233a", "#8bbc21",
					"#910000", "#1aadce", "#492970", "#f28f43", "#77a1e5",
					"#c42525", "#a6c96a");
		}
		return (List<Color>) _json.get(Attrs.colors);
	}

	/**
	 * Sets the default colors for the chart's series. When all colors are used,
	 * new colors are pulled from the start again.
	 */
	public void setColors(List<Color> colors) {
		_json.put(Attrs.colors, colors);
	}

	/**
	 * Sets the default colors for the chart's series. When all colors are used,
	 * new colors are pulled from the start again.
	 */
	public void setColors(String... source) {
		Color[] colors = new Color[source.length];
		int i = 0;
		for (String s : source)
			colors[i++] = new Color(s);
		setColors(Arrays.asList(colors));
	}

	private void clearResponseData() {
		ResponseDataHandler handler = getResponseDataHandler(false);
		if (handler != null)
			handler.clear();
	}

	private ResponseDataHandler getResponseDataHandler(boolean create) {
		String key = this.getClass().getName() + "#" + charts.getUuid();
		ResponseDataHandler handler = (ResponseDataHandler) Executions
				.getCurrent().getAttribute(key);
		if (create && handler == null) {
			handler = new ResponseDataHandler();
			Executions.getCurrent().setAttribute(key, handler);
		}
		return handler;
	}

	private ResponseDataHandler addResponseData(OptionDataEvent evt) {
		ResponseDataHandler handler = getResponseDataHandler(true);
		handler.addQueue(evt);
		return handler;
	}

	public void onChange(OptionDataEvent event) {
		final Optionable target = event.getTarget();
		
		if (EventType.SELECTED == event.getType()
				&& event.getOriginTarget() instanceof Point) {
			charts.selectPoint((Point) event.getOriginTarget(), (Boolean) event.getValue("accumulate"));
		}

		if (!charts.isRendered() || !charts.checkLock()) {
			clearResponseData();
			return; // nothing to do while rendering
		}

		// Highcharts didn't support to change plotOptions dynamically.
		if (target instanceof PlotOptions) {
			charts.invalidate();
			clearResponseData();
			return;
		}

		if (target instanceof Series) {
			if (EventType.INITIALIZED == event.getType()) {
				// we have to eval the target at this moment to avoid the replicated "addPoint" command.
				//final String result = target.toJSONString();
				event.addJSFunctionCall(new DeferredCall() {
					public void execute(JSFunction func) {
						//func.evalJavascript("this.addSeries(" + result + ")");
						func.callFunction("this.addSeries", target);
					}
				});
			} else {
				LinkedList<Series> list = (LinkedList<Series>) _json
						.get(Attrs.series);
				final int sIndex = list.indexOf(target);

				// sub options like point
				final boolean hasJS = event.hasJSFunctionCall();
				event.addJSFunctionCall(new DeferredCall() {
					public void execute(JSFunction func) {
						if (!hasJS)
							func.callFunction("update", target);
						func.callArray(Attrs.series.toString(), sIndex);
					}
				}).setJSUpdateCall(!hasJS);
			}
			if (EventType.DESTROYED == event.getType()
					&& event.getOriginTarget() instanceof Series) {
				LinkedList<Series> list = (LinkedList<Series>) _json
						.get(Attrs.series);
				list.remove(target);
			}
		} else if (target instanceof YAxis) {
			if (EventType.INITIALIZED == event.getType()) {
				// we have to eval the target at this moment to avoid the replicated command.
				final String result = target.toJSONString();
				event.addJSFunctionCall(new DeferredCall() {
					public void execute(JSFunction func) {
						func.evalJavascript("this.addAxis(" + result
								+ ", false)");
					}
				});
			} else {

				LinkedList<YAxis> list = (LinkedList<YAxis>) _json
						.get(Attrs.yAxis);
				final int yIndex = list.indexOf(target);

				// sub options like point
				final boolean hasJS = event.hasJSFunctionCall();
				event.addJSFunctionCall(new DeferredCall() {
					public void execute(JSFunction func) {
						if (!hasJS)
							func.callFunction("update", target);
						func.callArray(Attrs.yAxis.toString(), yIndex);
					}
				}).setJSUpdateCall(!hasJS);
			}
			if (EventType.DESTROYED == event.getType()
					&& event.getOriginTarget() instanceof YAxis) {
				LinkedList<YAxis> list = (LinkedList<YAxis>) _json
						.get(Attrs.yAxis);
				list.remove(target);
			}
		} else if (target instanceof XAxis) {
			if (EventType.INITIALIZED == event.getType()) {

				// we have to eval the target at this moment to avoid the replicated command.
				final String result = target.toJSONString();
				event.addJSFunctionCall(new DeferredCall() {
					public void execute(JSFunction func) {
						func.evalJavascript("this.addAxis(" + result
								+ ", true)");
					}
				});
			} else {

				LinkedList<XAxis> list = (LinkedList<XAxis>) _json
						.get(Attrs.xAxis);
				final int xIndex = list.indexOf(target);

				// sub options like point
				final boolean hasJS = event.hasJSFunctionCall();
				event.addJSFunctionCall(new DeferredCall() {
					public void execute(JSFunction func) {
						if (!hasJS)
							func.callFunction("update", target);
						func.callArray(Attrs.xAxis.toString(), xIndex);
					}
				}).setJSUpdateCall(!hasJS);
			}
			if (EventType.DESTROYED == event.getType()
					&& event.getOriginTarget() instanceof XAxis) {
				LinkedList<XAxis> list = (LinkedList<XAxis>) _json
						.get(Attrs.xAxis);
				list.remove(target);
			}
		} else if (target instanceof Title) {
			event.addJSFunctionCall(new DeferredCall() {
				public void execute(JSFunction func) {
					func.callFunction("setTitle", target);
				}
			});
		} else if (target instanceof Subtitle) {
			event.addJSFunctionCall(new DeferredCall() {
				public void execute(JSFunction func) {
					func.callFunction("setTitle", null, target);
				}
			});
		} else if (target instanceof Chart) {
			if ("width".equals(event.getKey())) {
				final Integer width = (Integer) event.getValue(event.getKey());
				final Integer height = (Integer) ((Chart) target).getHeight();
				event.addJSFunctionCall(new DeferredCall() {
					public void execute(JSFunction func) {
						func.callFunction("setSize", width, null);
					}
				});
			} else if ("height".equals(event.getKey())) {
				final Integer height = (Integer) event.getValue(event.getKey());
				final Integer width = (Integer) ((Chart) target).getWidth();
				event.addJSFunctionCall(new DeferredCall() {
					public void execute(JSFunction func) {
						func.callFunction("setSize", null, height);
					}
				});
			}
		}

		switch (event.getType()) {
		case ECHO:
			String attr = event.getKey();
			Object data = event.getValue(attr);
			if (data instanceof EventListener) {
				charts.setAttribute(event.hashCode() + "." + attr, data);
				Clients.response(new AuInvoke(charts, "echo", new Object[] {
						event.hashCode() + "." + attr, event }));
			}
			break;
		default:
			if (event.hasJSFunctionCall())
				charts.smartUpdate("eval", addResponseData(event), false);
			else
				charts.invalidate();
		}
	}

	/**
	 * Returns the series at the index 0
	 */
	public Series getSeries() {
		return getSeries(0);
	}

	/**
	 * Returns the size of the series list
	 */
	public int getSeriesSize() {
		LinkedList<Series> list = (LinkedList<Series>) _json.get(Attrs.series);
		if (list == null)
			return 0;
		else
			return list.size();
	}

	/**
	 * Returns the series from the given index.
	 */
	@SuppressWarnings("unchecked")
	public Series getSeries(int index) {
		LinkedList<Series> list = (LinkedList<Series>) _json.get(Attrs.series);
		if (list == null) {
			list = new LinkedList<Series>();
			_json.put(Attrs.series, list);
		}

		int size = list.size();
		if (size <= index) {
			for (int i = index - size + 1; i > 0; i--) {
				Series series = new Series().addOptionDataListener(this);
				list.add(series);
				onChange(new OptionDataEvent(series, EventType.INITIALIZED,
						Attrs.series.toString(), series));
			}
		}
		return list.get(index);
	}

	/**
	 * Adds the series at the end of the series list
	 */
	public void addSeries(Series series) {
		LinkedList<Series> list = (LinkedList<Series>) _json.get(Attrs.series);
		if (list == null) {
			list = new LinkedList<Series>();
			_json.put(Attrs.series, list);
		}
		series.addOptionDataListener(this);
		list.add(series);
		onChange(new OptionDataEvent(series, EventType.INITIALIZED,
				Attrs.series.toString(), series));
	}

	/**
	 * Returns the xAxis at the index 0
	 */
	public XAxis getXAxis() {
		return getXAxis(0);
	}

	/**
	 * Returns the size of the xAxis list.
	 */
	public int getXAxisSize() {
		LinkedList<XAxis> list = (LinkedList<XAxis>) _json.get(Attrs.xAxis);
		if (list == null)
			return 0;
		else
			return list.size();
	}

	private boolean firstXAxis = true;
	private boolean firstYAxis = true;

	/**
	 * Returns the xAxis from the given index.
	 */
	@SuppressWarnings("unchecked")
	public XAxis getXAxis(int index) {
		LinkedList<XAxis> list = (LinkedList<XAxis>) _json.get(Attrs.xAxis);
		if (list == null) {
			list = new LinkedList<XAxis>();
			_json.put(Attrs.xAxis, list);
		}

		int size = list.size();
		if (size <= index) {
			for (int i = index - size + 1; i > 0; i--) {
				XAxis xAxis = new XAxis().addOptionDataListener(this);
				list.add(xAxis);
				if (firstXAxis) {
					firstXAxis = false; // ignore the first event, because the
					// client has one already.
				} else
					onChange(new OptionDataEvent(xAxis, EventType.INITIALIZED,
							Attrs.series.toString(), xAxis));
			}
		}
		return list.get(index);
	}

	/**
	 * Adds the xAxis at the end of the xAxis list.
	 */
	public void addXAxis(XAxis xAxis) {
		LinkedList<XAxis> list = (LinkedList<XAxis>) _json.get(Attrs.xAxis);
		if (list == null) {
			list = new LinkedList<XAxis>();
			_json.put(Attrs.xAxis, list);
		}
		xAxis.addOptionDataListener(this);
		list.add(xAxis);
		onChange(new OptionDataEvent(xAxis, EventType.INITIALIZED,
				Attrs.xAxis.toString(), xAxis));
	}

	/**
	 * Returns the yAxis at the index 0
	 */
	public YAxis getYAxis() {
		return getYAxis(0);
	}

	/**
	 * Returns the size of yAxis list.
	 */
	public int getYAxisSize() {
		LinkedList<YAxis> list = (LinkedList<YAxis>) _json.get(Attrs.yAxis);
		if (list == null)
			return 0;
		else
			return list.size();
	}

	@SuppressWarnings("unchecked")
	public YAxis getYAxis(int index) {
		LinkedList<YAxis> list = (LinkedList<YAxis>) _json.get(Attrs.yAxis);
		if (list == null) {
			list = new LinkedList<YAxis>();
			_json.put(Attrs.yAxis, list);
		}

		int size = list.size();
		if (size <= index) {
			for (int i = index - size + 1; i > 0; i--) {
				YAxis yAxis = new YAxis().addOptionDataListener(this);
				list.add(yAxis);
				if (firstYAxis) {
					// The test case is "test/api/axis/setTitle.zul"
					firstYAxis = false; // ignore the first event, because the
					// client has one already.
				} else
					onChange(new OptionDataEvent(yAxis, EventType.INITIALIZED,
							Attrs.series.toString(), yAxis));
			}
		}
		return list.get(index);
	}

	/**
	 * Adds the yAxis at the end of yAxis list.
	 */
	public void addYAxis(YAxis yAxis) {
		LinkedList<YAxis> list = (LinkedList<YAxis>) _json.get(Attrs.yAxis);
		if (list == null) {
			list = new LinkedList<YAxis>();
			_json.put(Attrs.yAxis, list);
		}
		yAxis.addOptionDataListener(this);
		list.add(yAxis);
		onChange(new OptionDataEvent(yAxis, EventType.INITIALIZED,
				Attrs.yAxis.toString(), yAxis));
	}

	/**
	 * Returns the no-data options
	 * @since 1.0.1
	 */
	public NoData getNoData() {
		NoData noData = (NoData) _json.get(Attrs.noData);
		if (noData == null) {
			noData = new NoData();
			setNoData(noData);
		}
		return noData;
	}

	/**
	 * Sets the no-data options
	 * @since 1.0.1
	 */
	public void setNoData(NoData noData) {
		NoData old = (NoData) _json.put(Attrs.noData, noData);
		if (old != noData) {
			addOptionDataListener(noData);
			removeOptionDataListener(old);
		}
	}

	/**
	 * Returns the chart options
	 */
	public Chart getChart() {
		Chart chart = (Chart) _json.get(Attrs.chart);
		if (chart == null) {
			chart = new Chart();
			setChart(chart);
		}
		return chart;
	}

	/**
	 * Sets the chart options
	 */
	public void setChart(Chart chart) {
		Chart old = (Chart) _json.put(Attrs.chart, chart);
		if (old != chart) {
			addOptionDataListener(chart);
			removeOptionDataListener(old);
		}
	}
	
	/**
	 * Returns the credits options
	 */
	public Credits getCredits() {
		Credits credits = (Credits) _json.get(Attrs.credits);
		if (credits == null) {
			credits = new Credits();
			setCredits(credits);
		}
		return credits;
	}

	/**
	 * Sets the credits options
	 */
	public void setCredits(Credits credits) {
		Credits old = (Credits) _json.put(Attrs.credits, credits);
		if (old != credits) {
			addOptionDataListener(credits);
			removeOptionDataListener(old);
		}
	}

	/**
	 * Returns the drilldown options
	 */
	public Drilldown getDrilldown() {
		Drilldown drilldown = (Drilldown) _json.get(Attrs.drilldown);
		if (drilldown == null) {
			drilldown = new Drilldown();
			setDrilldown(drilldown);
		}
		return drilldown;
	}

	/**
	 * Sets the drilldown options
	 */
	public void setDrilldown(Drilldown drilldown) {
		Drilldown old = (Drilldown) _json.put(Attrs.drilldown, drilldown);
		if (old != drilldown) {
			addOptionDataListener(drilldown);
			removeOptionDataListener(old);
		}
	}
	
	/**
	 * Returns the labels options
	 */
	public Labels getLabels() {
		Labels labels = (Labels) _json.get("labels");
		if (labels == null) {
			labels = new Labels();
			setLabels(labels);
		}
		return labels;
	}

	/**
	 * Sets the labels options
	 */
	public void setLabels(Labels labels) {
		Labels old = (Labels) _json.put(Attrs.labels, labels);
		if (old != labels) {
			addOptionDataListener(labels);
			removeOptionDataListener(old);
		}
	}

	/**
	 * Returns the legend options
	 */
	public Legend getLegend() {
		Legend legend = (Legend) _json.get(Attrs.legend);
		if (legend == null) {
			legend = new Legend();
			setLegend(legend);
		}
		return legend;
	}

	/**
	 * Sets the legend options
	 */
	public void setLegend(Legend legend) {
		Legend old = (Legend) _json.put(Attrs.legend, legend);
		if (old != legend) {
			addOptionDataListener(legend);
			removeOptionDataListener(old);
		}
	}

	/**
	 * Returns the loading options
	 */
	public Loading getLoading() {
		Loading loading = (Loading) _json.get(Attrs.loading);
		if (loading == null) {
			loading = new Loading();
			setLoading(loading);
		}
		return loading;
	}

	/**
	 * Sets the loading options
	 */
	public void setLoading(Loading loading) {
		Loading old = (Loading) _json.put(Attrs.loading, loading);
		if (old != loading) {
			addOptionDataListener(loading);
			removeOptionDataListener(old);
		}
	}

	/**
	 * Returns the navigation options
	 */
	public Navigation getNavigation() {
		Navigation navigation = (Navigation) _json.get("navigation");
		if (navigation == null) {
			navigation = new Navigation();
			setNavigation(navigation);
		}
		return navigation;
	}

	/**
	 * Sets the navigation options
	 */
	public void setNavigation(Navigation navigation) {
		Navigation old = (Navigation) _json.put(Attrs.navigation, navigation);
		if (old != navigation) {
			addOptionDataListener(navigation);
			removeOptionDataListener(old);
		}
	}

	private void addOptionDataListener(Optionable option) {
		if (option != null)
			option.addOptionDataListener(this);
	}

	private void removeOptionDataListener(Optionable option) {
		if (option != null)
			option.removeOptionDataListener(this);
	}
	
	/**
	 * Returns the plot options
	 */
	public PlotOptions getPlotOptions() {
		PlotOptions plotOptions = (PlotOptions) _json.get(Attrs.plotOptions);
		if (plotOptions == null) {
			plotOptions = new PlotOptions();
			setPlotOptions(plotOptions);
		}
		return plotOptions;
	}

	/**
	 * Sets the plot options
	 */
	public void setPlotOptions(PlotOptions plotOptions) {
		PlotOptions old = (PlotOptions) _json.put(Attrs.plotOptions,
				plotOptions);
		if (old != plotOptions) {
			addOptionDataListener(plotOptions);
			removeOptionDataListener(old);
		}
	}

	/**
	 * Returns the subtitle options
	 */
	public Subtitle getSubtitle() {
		Subtitle subtitle = (Subtitle) _json.get(Attrs.subtitle);
		if (subtitle == null) {
			subtitle = new Subtitle();
			setSubtitle(subtitle);
		}
		return subtitle;
	}

	/**
	 * Sets the subtitle options
	 */
	public void setSubtitle(Subtitle subtitle) {
		Subtitle old = (Subtitle) _json.put(Attrs.subtitle, subtitle);
		if (old != subtitle) {
			addOptionDataListener(subtitle);
			removeOptionDataListener(old);
		}
	}

	/**
	 * Returns the title options
	 */
	public Title getTitle() {
		Title title = (Title) _json.get(Attrs.title);
		if (title == null) {
			title = new Title();
			setTitle(title);
		}
		return title;
	}

	/**
	 * Sets the title options
	 * @param title
	 */
	public void setTitle(Title title) {
		Title old = (Title) _json.put(Attrs.title, title);
		if (old != title) {
			addOptionDataListener(title);
			removeOptionDataListener(old);
		}
	}

	/**
	 * Returns the tooltip options
	 */
	public Tooltip getTooltip() {
		Tooltip tooltip = (Tooltip) _json.get(Attrs.tooltip);
		if (tooltip == null) {
			tooltip = new Tooltip();
			setTooltip(tooltip);
		}
		return tooltip;
	}

	/**
	 * Sets the tooltip options
	 */
	public void setTooltip(Tooltip tooltip) {
		Tooltip old = (Tooltip) _json.put(Attrs.tooltip, tooltip);
		if (old != tooltip) {
			addOptionDataListener(tooltip);
			removeOptionDataListener(old);
		}
	}

	/**
	 * Returns the pane at the index 0
	 */
	public Pane getPane() {
		return getPane(0);
	}

	/**
	 * Return the size of the pane list
	 */
	public int getPaneSize() {
		LinkedList<Pane> list = (LinkedList<Pane>) _json.get(Attrs.pane);
		if (list == null)
			return 0;
		else
			return list.size();
	}

	/**
	 * Returns the pane from the given index
	 */
	@SuppressWarnings("unchecked")
	public Pane getPane(int index) {
		LinkedList<Pane> list = (LinkedList<Pane>) _json.get(Attrs.pane);
		if (list == null) {
			list = new LinkedList<Pane>();
			_json.put(Attrs.pane, list);
		}

		int size = list.size();
		if (size <= index) {
			for (int i = index - size + 1; i > 0; i--) {
				list.add(new Pane());
				charts.invalidate(); // need to redraw
			}
		}
		return list.get(index);
	}

	/**
	 * Merges the current plot data with the given one.
	 */
	protected PlotData merge(PlotData other) {
		for (Map.Entry<PlotAttribute, Object> me : other._json.entrySet()) {
			Object o = _json.get(me.getKey());
			if (o == null) {
				_json.put(me.getKey(), me.getValue());
			} else {
				if (o instanceof List) {
					List<Optionable> lo = (List<Optionable>) o;
					List<Optionable> lo2 = (List<Optionable>) me.getValue();
					final int sizeOfLo = lo.size();
					for (int i = 0; i < lo2.size(); i++) {
						if (i >= sizeOfLo) {
							lo.add(lo2.get(i));
						} else {
							lo.get(i).merge(lo2.get(i));
						}
					}
				} else if (o instanceof Optionable) {
					((Optionable) o).merge((Optionable) me.getValue());
				} else {
					throw new UiException("Unknown type: [" + o + "]");
				}
			}
		}
		return this;
	}

	public String toJSONString() {
		return JSONObject.toJSONString(_json);
	}

	public int hashCode() {
		return toJSONString().hashCode();
	}

	public boolean equals(Object o) {
		if (this == o) {
			return true;
		} else if (o instanceof PlotData) {
			PlotData op = (PlotData) o;
			return toJSONString().equals(op.toJSONString());
		}
		return false;
	}

}
