/* PlotBand.java

	Purpose:
		
	Description:
		
	History:
		Fri, Jan 17, 2014 12:02:18 PM, Created by RaymondChao

Copyright (C) 2014 Potix Corporation. All Rights Reserved.

 */
package org.zkoss.chart;

import org.zkoss.chart.util.DynamicalAttribute;

/**
 * A colored band stretching across the plot area marking an interval on the
 * axis. In a gauge, a plot band on the Y axis (value axis) will stretch along
 * the perimiter of the gauge.
 * <p>
 * All the options in this class support {@link DynamicalAttribute}.
 * 
 * @author RaymondChao
 * @author jumperchen
 */
public class PlotBand extends Optionable {
	private enum Attrs implements PlotAttribute, DynamicalAttribute {
		color, from, id, innerRadius, label, outerRadius, to, zIndex
	}

	/**
	 * Default constructor
	 */
	public PlotBand() {
	}

	/**
	 * Constructs with the values, from, to, and color.
	 */
	public PlotBand(Number from, Number to, String color) {
		setFrom(from);
		setTo(to);
		setColor(color);
	}

	/**
	 * Returns the color of the plot band.
	 * <p>
	 * Default: null.
	 */
	public Color getColor() {
		return (Color) getAttr(Attrs.color, null).asValue();
	}

	/**
	 * Sets the color of the plot band.
	 */
	public void setColor(Color color) {
		setAttr(Attrs.color, color);
	}

	/**
	 * Sets the color of the plot band.
	 */
	public void setColor(String color) {
		setColor(new Color(color));
	}

	/**
	 * Sets the color of the plot band.
	 */
	public void setColor(LinearGradient color) {
		setColor(new Color(color));
	}

	/**
	 * Sets the color of the plot band.
	 */
	public void setColor(RadialGradient color) {
		setColor(new Color(color));
	}

	/**
	 * Returns the start position of the plot band in axis units.
	 */
	public Number getFrom() {
		return getAttr(Attrs.from, null).asNumber();
	}

	/**
	 * Sets the start position of the plot band in axis units. 'from the start
	 * position of the plot band in axis units.
	 */
	public void setFrom(Number from) {
		setAttr(Attrs.from, from);
	}

	/**
	 * Returns an id used for identifying the plot band in
	 * {@link Axis#removePlotBand(String)}.
	 */
	public String getId() {
		return getAttr(Attrs.id, null).asString();
	}

	/**
	 * Sets an id used for identifying the plot band in
	 * {@link Axis#removePlotBand(String)}.
	 */
	public void setId(String id) {
		setAttr(Attrs.id, id);
	}

	/**
	 * Returns the inner radius.
	 */
	public Object getInnerRadius() {
		return getAttr(Attrs.innerRadius, null).asValue();
	}

	/**
	 * Sets the inner radius.
	 */
	public void setInnerRadius(String innerRadius) {
		setAttr(Attrs.innerRadius, innerRadius);
	}

	/**
	 * Sets the inner radius.
	 */
	public void setInnerRadius(Number innerRadius) {
		setAttr(Attrs.innerRadius, innerRadius);
	}

	/**
	 * Returns text labels for the plot bands
	 * <p>
	 * Default: an instance of PlotLabel.
	 */
	public PlotLabel getLabel() {
		PlotLabel label = (PlotLabel) this.getAttr(Attrs.label);

		if (label == null) {
			label = new PlotLabel();
			setLabel(label);
		}
		return label;
	}

	/**
	 * Sets text labels for the plot bands
	 */
	public void setLabel(PlotLabel label) {
		setAttr(Attrs.label, label);
	}

	/**
	 * Returns the outer radius
	 * <p>
	 * Default: null.
	 */
	public Object getOuterRadius() {
		return getAttr(Attrs.outerRadius, null).asValue();
	}

	/**
	 * Sets the outer radius
	 */
	public void setOuterRadius(String outerRadius) {
		setAttr(Attrs.outerRadius, outerRadius);
	}

	/**
	 * Sets the outer radius
	 */
	public void setOuterRadius(Number outerRadius) {
		setAttr(Attrs.outerRadius, outerRadius);
	}

	/**
	 * Returns the end position of the plot band in axis units.
	 */
	public Number getTo() {
		return getAttr(Attrs.to, null).asNumber();
	}

	/**
	 * Sets the end position of the plot band in axis units.
	 */
	public void setTo(Number to) {
		setAttr(Attrs.to, to);
	}

	/**
	 * Returns the z index of the plot band within the chart.
	 * <p>
	 * Default: null
	 */
	public Number getZIndex() {
		return getAttr(Attrs.zIndex, null).asNumber();
	}

	/**
	 * Sets the z index of the plot band within the chart.
	 */
	public void setZIndex(Number zIndex) {
		setAttr(Attrs.zIndex, zIndex);
	}

}