/* Pane.java

	Purpose:
		
	Description:
		
	History:
		Thu, Jan 16, 2014  4:36:06 PM, Created by RaymondChao

Copyright (C) 2014 Potix Corporation. All Rights Reserved.

 */
package org.zkoss.chart;

import java.util.Arrays;
import java.util.List;

import org.zkoss.chart.util.DynamicalAttribute;
import org.zkoss.lang.Generics;

/**
 * Applies only to polar charts and angular gauges. This configuration object
 * holds general options for the combined X and Y axes set. Each xAxis or yAxis
 * can reference the pane by index.
 * <p>
 * All the options in this class support {@link DynamicalAttribute}.
 * 
 * @author RaymondChao
 * @author jumperchen
 */
public class Pane extends Optionable {
	private enum Attrs implements PlotAttribute, DynamicalAttribute {
		background, center, endAngle, startAngle, size
	}

	/**
	 * Returns a list of objects, for backgrounds. Sub options include
	 * <code>backgroundColor</code> (which can be solid or gradient),
	 * <code>innerWidth</code>, <code>outerWidth</code>,
	 * <code>borderWidth</code>, <code>borderColor</code>.
	 */
	public List<PaneBackground> getBackground() {
		return Generics.cast(getAttr(Attrs.background));
	}

	/**
	 * Sets a list of objects, for backgrounds. Sub options include
	 * <code>backgroundColor</code> (which can be solid or gradient),
	 * <code>innerWidth</code>, <code>outerWidth</code>,
	 * <code>borderWidth</code>, <code>borderColor</code>. 'background an
	 * object, or array of objects, for backgrounds. Sub options include
	 * <code>backgroundColor</code> (which can be solid or gradient),
	 * <code>innerWidth</code>, <code>outerWidth</code>,
	 * <code>borderWidth</code>, <code>borderColor</code>.
	 */
	public void setBackground(List<PaneBackground> background) {
		setAttr(Attrs.background, background);
	}

	/**
	 * Returns the center of a polar chart or angular gauge, given as an array
	 * of [x, y] positions. Positions can be given as integers that transform to
	 * pixels, or as percentages of the plot area size.
	 * 
	 * @return center the center of a polar chart or angular gauge, given as an
	 *         array of [x, y] positions. Positions can be given as integers
	 *         that transform to pixels, or as percentages of the plot area
	 *         size.
	 */
	public <T> List<T> getCenter() {
		return Generics.cast(getAttr(Attrs.center, null).asValue());
	}

	/**
	 * Sets the center of a polar chart or angular gauge, given as an array of
	 * [x, y] positions. Positions can be given as integers that transform to
	 * pixels, or as percentages of the plot area size. 'center the center of a
	 * polar chart or angular gauge, given as an array of [x, y] positions.
	 * Positions can be given as integers that transform to pixels, or as
	 * percentages of the plot area size.
	 */
	public <T> void setCenter(List<T> center) {
		setAttr(Attrs.center, center);
	}
	
	public void setCenter(Object... center) {
		setCenter(Arrays.asList(center));
	}

	/**
	 * Returns the end angle of the polar X axis or gauge value axis, given in
	 * degrees where 0 is north.
	 * <p>
	 * Defaults to {@link #getStartAngle()} + 360.
	 */
	public Number getEndAngle() {
		if (containsKey(Attrs.endAngle)) {
			return (Number) getAttr(Attrs.endAngle);
		} else {
			return ((Integer)getStartAngle()) + 360;
		}
	}

	/**
	 * Sets the end angle of the polar X axis or gauge value axis, given in
	 * degrees where 0 is north.
	 * <p>
	 * Defaults to {@link #getStartAngle()} + 360.
	 */
	public void setEndAngle(Number endAngle) {
		setAttr(Attrs.endAngle, endAngle, NOT_NULL_VALUE);
	}

	/**
	 * Returns the start angle of the polar X axis or gauge axis, given in
	 * degrees where 0 is north.
	 * <p> Defaults to 0.
	 */
	public Number getStartAngle() {
		return getAttr(Attrs.startAngle, 0).asNumber();
	}

	/**
	 * Sets the start angle of the polar X axis or gauge axis, given in degrees
	 * where 0 is north.
	 * <p> Defaults to 0.
	 */
	public void setStartAngle(Number startAngle) {
		setAttr(Attrs.startAngle, startAngle, 0);
	}

	/**
	 * Returns the size of the pane.
	 * <p> Default: null
	 */
	public Object getSize() {
		return getAttr(Attrs.size, null).asValue();
	}

	/**
	 * Sets the size of the pane.
	 * @param size
	 */
	public void setSize(Number size) {
		setAttr(Attrs.size, size);
	}

	/**
	 * Sets the size of the pane.
	 * @param size
	 */
	public void setSize(String size) {
		setAttr(Attrs.size, size);
	}

}