/** Optionable.java.

	Purpose:
		
	Description:
		
	History:
		2:13:42 PM Jan 9, 2014, Created by jumperchen

Copyright (C) 2014 Potix Corporation. All Rights Reserved.
 */
package org.zkoss.chart;

import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

import org.zkoss.chart.OptionDataEvent.EventType;
import org.zkoss.chart.util.AnyVal;
import org.zkoss.chart.util.DeferredCall;
import org.zkoss.chart.util.DynamicalAttribute;
import org.zkoss.chart.util.JSFunction;
import org.zkoss.io.Serializables;
import org.zkoss.json.JSONAware;
import org.zkoss.json.JSONObject;
import org.zkoss.json.JSONValue;
import org.zkoss.lang.Classes;
import org.zkoss.lang.Objects;
import org.zkoss.zk.ui.event.Event;
import org.zkoss.zk.ui.event.EventListener;

/**
 * A skeletal implementation for Highcharts's optionable data.
 * 
 * @author jumperchen
 */
abstract public class Optionable implements JSONAware, OptionDataListener {

	protected Map<String, AnyVal<Object>> options = new LinkedHashMap<String, AnyVal<Object>>(
			5);

	private Set<OptionDataListener> _listeners = new HashSet<OptionDataListener>();

	/**
	 * An instance for setAttr to check the default is not a null value.
	 */
	protected static String NOT_NULL_VALUE = "ZnotNullValueKChart";

	/**
	 * Triggers the option change event.
	 */
	protected <T extends Optionable> T fireEvent(EventType type,
			PlotAttribute key, Object value) {
		return fireEvent(type, key.toString(), value, null);
	}

	/**
	 * Triggers the option change event.
	 */
	protected <T extends Optionable> T fireEvent(EventType type,
			PlotAttribute key, Object value, DeferredCall call) {
		return fireEvent(type, key.toString(), value, call);
	}

	/**
	 * Triggers the option change event.
	 */
	protected <T extends Optionable> T fireEvent(EventType type, String key,
			Object value, DeferredCall call) {
		final OptionDataEvent evt = new OptionDataEvent(this, type, key, value,
				call);
		return fireEvent(evt);
	}

	/**
	 * Triggers the option change event.
	 */
	protected <T extends Optionable> T fireEvent(EventType type, String key,
			Object value, String key2, Object value2, Object... pairs) {
		final OptionDataEvent evt = new OptionDataEvent(this, type, key, value,
				key2, value2, pairs);
		return fireEvent(evt);
	}

	/**
	 * Triggers the option change event.
	 */
	protected <T extends Optionable> T fireEvent(OptionDataEvent event) {
		for (OptionDataListener l : _listeners)
			l.onChange(event);
		return (T) this;
	}

	/**
	 * Adds the option data listener
	 */
	public <T extends Optionable> T addOptionDataListener(OptionDataListener l) {
		if (l == null)
			throw new NullPointerException();
		_listeners.add(l);
		return (T) this;
	}

	/**
	 * Removes the option data listener
	 */
	public <T extends Optionable> T removeOptionDataListener(
			OptionDataListener l) {
		_listeners.remove(l);
		return (T) this;
	}

	/**
	 * Clear up all the option data listeners
	 */
	public <T extends Optionable> T clearOptonDataListener() {
		_listeners.clear();
		return (T) this;
	}

	/**
	 * Returns a client state after a client callback in the event listener.
	 */
	public void getClientState(final String attr, EventListener<Event> listener) {
		fireEvent(EventType.ECHO, attr, listener, new DeferredCall() {
			public void execute(JSFunction func) {
				func.evalAttribute(attr);
			}
		});
	}

	// Serializable//
	private synchronized void writeObject(java.io.ObjectOutputStream s)
			throws java.io.IOException {
		s.defaultWriteObject();

		Serializables.smartWrite(s, _listeners);
		Serializables.smartWrite(s, options);
	}

	private void readObject(java.io.ObjectInputStream s)
			throws java.io.IOException, ClassNotFoundException {
		s.defaultReadObject();

		_listeners = new LinkedHashSet<OptionDataListener>();
		options = new LinkedHashMap<String, AnyVal<Object>>();
		Serializables.smartRead(s, _listeners);
		Serializables.smartRead(s, options);
	}

	public Object clone() {
		final Optionable clone;
		try {
			clone = (Optionable) super.clone();
		} catch (CloneNotSupportedException e) {
			throw new InternalError();
		}
		clone._listeners = new LinkedHashSet<OptionDataListener>();
		clone.options = new LinkedHashMap<String, AnyVal<Object>>(options);
		return clone;
	}

	/**
	 * Sets the value of the attribute name.
	 * 
	 * @param key
	 *            the name of the attribute
	 * @param value
	 *            the value of the attribute
	 * @return boolean true if changed
	 */
	protected boolean setAttr(PlotAttribute key, Object value) {
		return setAttr(key, value, null);
	}

	/**
	 * Sets the value of the attribute name with the given default value, which
	 * is used for comparing the null value case when the default value is not
	 * null.
	 * 
	 * @param key
	 *            the name of the attribute
	 * @param value
	 *            the value of the attribute
	 * @param defaultValue
	 *            defaultValue for specifying the null value to check whether
	 *            the value is not the same as T defaultValue
	 * @return boolean true if changed
	 */
	protected <T> boolean setAttr(PlotAttribute key, Object value,
			T defaultValue) {
		final String strKey = key.toString();
		AnyVal<Object> old = options.get(strKey);
		Object oldValue = old != null ? old.asValue() : defaultValue;
		if (!Objects.equals(oldValue, value)) {
			options.put(strKey, new AnyVal<Object>(value));

			if (value instanceof Optionable)
				((Optionable) value).addOptionDataListener(this);

			if (key instanceof DynamicalAttribute)
				fireEvent(EventType.CHANGED, key, value);

			// just in case
			if (oldValue instanceof Optionable)
				((Optionable) oldValue).removeOptionDataListener(this);

			return true;
		}
		return false;
	}

	/**
	 * Sets the color value of the attribute name with the given default value,
	 * which is used for comparing the null value case when the default value is
	 * not null.
	 * 
	 * @param key
	 *            the name of the attribute
	 * @param value
	 *            the value of the color
	 * @param defaultValue
	 *            defaultValue for specifying the null value to check whether
	 *            the value is not the same as T defaultValue
	 * @return boolean true if changed
	 */
	protected <T> boolean setAttr(PlotAttribute key, Color value, T defaultValue) {
		final String strKey = key.toString();
		AnyVal<Object> old = options.get(strKey);

		final String oldColor = old != null ? old.asValue().toString() : JSONValue
				.toJSONString(defaultValue);
		
		final String newColor = JSONValue.toJSONString(value);

		if (!Objects.equals(oldColor, newColor)) {
			options.put(strKey, new AnyVal<Object>(value));

			if (value instanceof Optionable)
				((Optionable) value).addOptionDataListener(this);

			if (key instanceof DynamicalAttribute)
				fireEvent(EventType.CHANGED, key, value);

			if (old != null) {
				Object oldValue = old.asValue();
				// just in case
				if (oldValue instanceof Optionable)
					((Optionable) oldValue).removeOptionDataListener(this);
			}
			return true;
		}
		return false;
	}

	/**
	 * Returns the <code>AnyVal</code> object of the element according to the
	 * attribute name. If not found, sets with the given default value, if any.
	 * 
	 * @param key
	 *            the attribute name
	 * @param defaultValue
	 *            if the value is not found, the defaultValue will be returned.
	 */
	@SuppressWarnings("unchecked")
	protected <T> AnyVal<T> getAttr(PlotAttribute key, T defaultValue) {
		final String strKey = key.toString();
		if (options.containsKey(strKey)) {
			return (AnyVal<T>) options.get(strKey);
		} else {
			// only invoke setter for non-Number and String Object
			if (defaultValue != null
					&& !(defaultValue instanceof String || defaultValue instanceof Number)) {
				if (defaultValue instanceof Class) {
					try {
						Object o = Classes.newInstance((Class) defaultValue,
								null);
						setAttr(key, o);
					} catch (Exception e) {
						e.printStackTrace();
					}
				} else
					setAttr(key, defaultValue);
				return (AnyVal<T>) options.get(strKey);
			} else
				return new AnyVal<T>(defaultValue);
		}
	}

	/**
	 * Returns <tt>true</tt> if this options contains a mapping for the
	 * specified key.
	 */
	protected boolean containsKey(PlotAttribute key) {
		return options.containsKey(key.toString());
	}

	/**
	 * Removes the mapping for a key from this options if it is present.
	 */
	protected Object removeKey(PlotAttribute key, boolean triggerEvent) {
		AnyVal<Object> result = options.remove(key.toString());
		Object val = null;

		if (result != null)
			val = result.asValue();

		if (triggerEvent) {
			if (key instanceof DynamicalAttribute)
				fireEvent(EventType.REMOVED, key, result);
		}
		return val;
	}

	/**
	 * Merges the all value from the given object, if the key is the same, the
	 * value will be overridden.
	 * 
	 * @param other
	 */
	protected Optionable merge(Optionable other) {
		options.putAll(other.options);
		return this;
	}

	/**
	 * Returns the value of the element according to the attribute name.
	 * 
	 * @param key
	 *            the attribute name.
	 */
	protected Object getAttr(PlotAttribute key) {
		return getAttr(key, null).asValue();
	}

	/**
	 * Adds an extra attribute to client side, if any.
	 * 
	 * @return the old value, if any.
	 */
	public Object addExtraAttr(String key, JSONAware value) {
		AnyVal<Object> old = options.put(key, new AnyVal<Object>(value));
		if (old != null)
			return old.asValue();
		return null;
	}

	/**
	 * Removes the extra attribute to client side from the given key, if any.
	 * <p>
	 * Invoking this method won't trigger any event to notify the client side,
	 * please use {@link Charts#invalidate()} if you want to notify the client
	 * side to sync with the value.
	 * 
	 * @return the old value, if any.
	 */
	public Object removeExtraAttr(String key) {
		AnyVal<Object> remove = options.remove(key);
		if (remove != null)
			return remove.asValue();
		return null;
	}

	/**
	 * Encodes this object to a JSON string. It is the same as
	 * {@link #toString()}.
	 */
	public String toJSONString() {
		return JSONObject.toJSONString(options);
	}

	/**
	 * Encodes this object to a JSON string. It is the same as
	 * {@link #toJSONString()}.
	 */
	public String toString() {
		return toJSONString();
	}

	public void onChange(OptionDataEvent event) {
		event.setCurrentTarget(this);
		fireEvent(event);
	}
}
