/** OptionDataEvent.java.

	Purpose:
		
	Description:
		
	History:
		2:29:40 PM Jan 17, 2014, Created by jumperchen

Copyright (C) 2014 Potix Corporation. All Rights Reserved.
*/
package org.zkoss.chart;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.Set;

import org.zkoss.chart.util.DeferredCall;
import org.zkoss.chart.util.JSFunction;
import org.zkoss.json.JSONAware;

/**
 * Defines an event that encapsulates changes to a highcharts's optionable data. 
 * @author jumperchen
 */
public class OptionDataEvent implements JSONAware {
	
	/**
	 * Defines some states of OptionDataEvent for {@link Optionable}'s attributes
	 * @author jumperchen
	 */
	public enum EventType {
		/**
		 * Represents the event is in a initialized state
		 */
		INITIALIZED,
		/**
		 * Represents the event is in a changed state.
		 */
		CHANGED,
		/**
		 * Represents the event is in an added state.
		 */
		ADDED,
		/**
		 * Represents the event is in a removed state.
		 */
		REMOVED,
		/**
		 * Represents the event is in a selected state.
		 */
		SELECTED,
		/**
		 * Represents the event is in a destroyed state.
		 */
		DESTROYED,
		/**
		 * Represents the event is in an echo state.
		 */
		ECHO,
		/**
		 * Represents the event is in an invoke state.
		 */
		INVOKE
	}

	private Optionable _target;
	private Optionable _currentTarget;
	private EventType _type;
	private HashMap<String, Object> _attrMap;
	private LinkedList<DeferredCall> _funcs = new LinkedList<DeferredCall>();
	
	public OptionDataEvent(Optionable optionable, EventType type, Object... pair) {
		this._target = optionable;
		this._type = type;
		_attrMap = new HashMap<String, Object>(2);
		if (pair.length == 3) {
			_attrMap.put((String)pair[0], pair[1]);
			
			if (pair[2] instanceof Object[]) {
				Object[] pairs = (Object[]) pair[2];
				
				for (int i = 0; i < pairs.length; i+=2)
					_attrMap.put((String)pairs[i], pairs[i+1]);
				
			} else if (pair[2] instanceof DeferredCall) {
				_funcs.add((DeferredCall) pair[2]);
			}
		} else if (pair.length > 3) {
			for (int i = 0; i < pair.length; i+=2)
				_attrMap.put((String)pair[i], pair[i+1]);
		} else {
			_attrMap.put((String)pair[0], pair[1]);
		}
	}
	/**
	 * Returns the origin target.
	 */
	public Optionable getOriginTarget() {
		return _target;
	}
	/**
	 * Sets the current target, which is used for {@link OptionDataListener} implementation
	 * @param current the current target when the listener received, if any.
	 */
	public void setCurrentTarget(Optionable current) {
		_currentTarget = current;
	}
	/**
	 * Returns the current target.
	 * @see OptionDataEvent#getOriginTarget()
	 */
	public Optionable getTarget() {
		return _currentTarget != null ? _currentTarget : getOriginTarget();
	}
	/**
	 * Returns the type of the event.
	 * @see EventType#ADDED
	 * @see EventType#CHANGED
	 * @see EventType#REMOVED
	 * @see EventType#DESTROYED
	 * @see EventType#ECHO
	 */
	public EventType getType() {
		return _type;
	}
	/**
	 * Returns the value of the given key that triggered with the event.
	 * @param key the property key in {@link Optionable}
	 */
	public Object getValue(String key) {
		return _attrMap.get(key);
	}
	/**
	 * Returns the first key that triggered with the event
	 */
	public String getKey() {
		return _attrMap.keySet().iterator().next();
	}
	/**
	 * Returns the keys that triggered with the event
	 */
	public Set<String> getKeys() {
		return _attrMap.keySet();
	}
	/**
	 * Adds the Javascript function call to update at client side.
	 */
	public OptionDataEvent addJSFunctionCall(DeferredCall call) {
		_funcs.add(call);
		return this;
	}
	private boolean _updateCall;
	
	/**
	 * Sets whether the JSFunction call is for updating the whole optionable data.
	 * (like {@link Axis} and {@link Series}
	 * @param updateCall
	 */
	public OptionDataEvent setJSUpdateCall(boolean updateCall) {
		_updateCall = updateCall;
		return this;
	}
	
	/**
	 * Returns whether the JSFunction call has a command for whole updating.
	 */
	public boolean isJSUpldateCall() {
		return _updateCall;
	}
	/**
	 * Returns wthere the event contines with some Javascript function calls.
	 * @return true if contains
	 */
	public boolean hasJSFunctionCall() {
		return !_funcs.isEmpty();
	}
	public String toJSONString() {
		JSFunction func = new JSFunction();
		for (DeferredCall f : _funcs) {
			f.execute(func);
		}
		
		if (_type == EventType.ECHO) {
			return func.reverse().toReturnFunction();
		} else if (_type == EventType.INITIALIZED) {
			return func.reverse().toExecFunction();
		} else {
			return func.reverse().toWrapFunction();
		}
	}
	public String toString() {
		return toJSONString();
	}
}
