/* NavigationButton.java

	Purpose:
		
	Description:
		
	History:
		Fri, Jan 17, 2014 12:37:33 PM, Created by RaymondChao

Copyright (C) 2014 Potix Corporation. All Rights Reserved.

 */
package org.zkoss.chart;

import java.util.Map;

import org.zkoss.chart.util.DynamicalAttribute;
import org.zkoss.lang.Generics;

/**
 * A collection of options for buttons appearing in the exporting module.
 * <p>
 * All the options in this class support {@link DynamicalAttribute}.
 * 
 * @author jumperchen
 * @author RaymondChao
 */

public class NavigationButton extends Optionable {
	private enum Attrs implements PlotAttribute, DynamicalAttribute {
		align,
		enabled,
		height,
		symbolFill,
		symbolSize,
		symbolStroke,
		symbolStrokeWidth,
		symbolX,
		symbolY,
		text,
		theme,
		verticalAlign,
		width,
		y
	}

	/**
	 * Returns alignment for the buttons.
	 * <p>
	 * Default: right
	 */
	public String getAlign() {
		return getAttr(Attrs.align, "right").asString();
	}

	/**
	 * Sets alignment for the buttons.
	 */
	public void setAlign(String align) {
		setAttr(Attrs.align, align, "right");
	}

	/**
	 * Returns whether to enable buttons.
	 * <p>
	 * Default: true
	 */
	public boolean isEnabled() {
		return getAttr(Attrs.enabled, true).asBoolean();
	}

	/**
	 * Sets whether to enable buttons.
	 */
	public void setEnabled(boolean enabled) {
		setAttr(Attrs.enabled, enabled);
	}

	/**
	 * Returns pixel height of the buttons.
	 * <p>
	 * Default: 20
	 */
	public Number getHeight() {
		return getAttr(Attrs.height, 20).asNumber();
	}

	/**
	 * Sets pixel height of the buttons.
	 */
	public void setHeight(Number height) {
		setAttr(Attrs.height, height, 20);
	}

	/**
	 * Returns fill color for the symbol within the button.
	 * <p>
	 * Default: "#E0E0E0"
	 */
	public Color getSymbolFill() {
		if (!containsKey(Attrs.symbolFill)) {
			setSymbolFill("#E0E0E0");
		}
		return (Color) getAttr(Attrs.symbolFill);
	}

	/**
	 * Sets fill color for the symbol within the button.
	 */
	public void setSymbolFill(Color color) {
		setAttr(Attrs.symbolFill, color, NOT_NULL_VALUE);
	}

	/**
	 * Sets fill color for the symbol within the button.
	 */
	public void setSymbolFill(String color) {
		setSymbolFill(new Color(color));
	}

	/**
	 * Sets fill color for the symbol within the button.
	 */
	public void setSymbolFill(LinearGradient color) {
		setSymbolFill(new Color(color));
	}

	/**
	 * Sets fill color for the symbol within the button.
	 */
	public void setSymbolFill(RadialGradient color) {
		setSymbolFill(new Color(color));
	}

	/**
	 * Returns the pixel size of the symbol on the button.
	 * <p>
	 * Default: 14
	 */
	public Number getSymbolSize() {
		return getAttr(Attrs.symbolSize, 14).asNumber();
	}

	/**
	 * Sets the pixel size of the symbol on the button.
	 */
	public void setSymbolSize(Number symbolSize) {
		setAttr(Attrs.symbolSize, symbolSize, 14);
	}

	/**
	 * Returns the color of the symbol's stroke or line.
	 * <p>
	 * Default: "#666"
	 */
	public Color getSymbolStroke() {
		if (!containsKey(Attrs.symbolStroke)) {
			setSymbolStroke("#666");
		}
		return (Color) getAttr(Attrs.symbolStroke);
	}

	/**
	 * Sets the color of the symbol's stroke or line.
	 */
	public void setSymbolStroke(Color color) {
		setAttr(Attrs.symbolStroke, color, NOT_NULL_VALUE);
	}

	/**
	 * Sets the color of the symbol's stroke or line.
	 */
	public void setSymbolStroke(String color) {
		setSymbolStroke(new Color(color));
	}

	/**
	 * Sets the color of the symbol's stroke or line.
	 */
	public void setSymbolStroke(LinearGradient color) {
		setSymbolStroke(new Color(color));
	}

	/**
	 * Sets the color of the symbol's stroke or line.
	 */
	public void setSymbolStroke(RadialGradient color) {
		setSymbolStroke(new Color(color));
	}

	/**
	 * Returns the pixel stroke width of the symbol on the button.
	 * <p>
	 * Default: 1
	 */
	public Number getSymbolStrokeWidth() {
		return getAttr(Attrs.symbolStrokeWidth, 1).asNumber();
	}

	/**
	 * Sets the pixel stroke width of the symbol on the button.
	 */
	public void setSymbolStrokeWidth(Number symbolStrokeWidth) {
		setAttr(Attrs.symbolStrokeWidth, symbolStrokeWidth, 1);
	}

	/**
	 * Returns the x position of the center of the symbol inside the button.
	 * <p>
	 * Default: 12.5
	 */
	public Number getSymbolX() {
		return getAttr(Attrs.symbolX, 12.5).asNumber();
	}

	/**
	 * Sets the x position of the center of the symbol inside the button.
	 */
	public void setSymbolX(Number symbolX) {
		setAttr(Attrs.symbolX, symbolX, 12.5);
	}

	/**
	 * Returns the y position of the center of the symbol inside the button.
	 * <p>
	 * Default: 10.5
	 */
	public Number getSymbolY() {
		return getAttr(Attrs.symbolY, 10.5).asNumber();
	}

	/**
	 * Sets the y position of the center of the symbol inside the button.
	 */
	public void setSymbolY(Number symbolY) {
		setAttr(Attrs.symbolY, symbolY, 10.5);
	}

	/**
	 * Returns a text string to add to the individual button.
	 * <p>
	 * Default: null
	 */
	public String getText() {
		return getAttr(Attrs.text, null).asString();
	}

	/**
	 * Sets a text string to add to the individual button.
	 */
	public void setText(String text) {
		setAttr(Attrs.text, text);
	}

	/**
	 * Returns a collection of attributes for the button. The object takes SVG
	 * attributes like <code>fill</code>, <code>stroke</code>,
	 * <code>stroke-width</code> or <code>r</code>, the border radius. The theme
	 * also supports <code>style</code>, a collection of CSS properties for the
	 * text. Equivalent attributes for the hover state are given in
	 * <code>theme.states.hover</code>.
	 */
	public <K, V> Map<K, V> getTheme() {
		return Generics.cast(getAttr(Attrs.theme, null).asValue());
	}

	/**
	 * Sets a collection of attributes for the button. The object takes SVG
	 * attributes like <code>fill</code>, <code>stroke</code>,
	 * <code>stroke-width</code> or <code>r</code>, the border radius. The theme
	 * also supports <code>style</code>, a collection of CSS properties for the
	 * text. Equivalent attributes for the hover state are given in
	 * <code>theme.states.hover</code>.
	 */
	public <K, V> void setTheme(Map<K, V> theme) {
		setAttr(Attrs.theme, theme);
	}

	/**
	 * Returns the vertical alignment of the buttons. Can be one of "top",
	 * "middle" or "bottom".
	 * <p>
	 * Default: top
	 */
	public String getVerticalAlign() {
		return getAttr(Attrs.verticalAlign, "top").asString();
	}

	/**
	 * Sets the vertical alignment of the buttons. Can be one of "top", "middle"
	 * or "bottom".
	 */
	public void setVerticalAlign(String verticalAlign) {
		if (!("top".equals(verticalAlign) || "middle".equals(verticalAlign) || "bottom"
				.equals(verticalAlign)))
			throw new IllegalArgumentException("Unsupported vertical align: ["
					+ verticalAlign + "]");
		setAttr(Attrs.verticalAlign, verticalAlign, "top");
	}

	/**
	 * Returns the pixel width of the button.
	 * <p>
	 * Default: 24
	 */
	public Number getWidth() {
		return getAttr(Attrs.width, 24).asNumber();
	}

	/**
	 * Sets the pixel width of the button.
	 */
	public void setWidth(Number width) {
		setAttr(Attrs.width, width, 24);
	}

	/**
	 * Returns the vertical offset of the button's position relative to its
	 * <code>verticalAlign</code>.
	 * <p>
	 * Default: 0
	 */
	public Number getY() {
		return getAttr(Attrs.y, 0).asNumber();
	}

	/**
	 * Sets the vertical offset of the button's position relative to its
	 * <code>verticalAlign</code>.
	 */
	public void setY(Number y) {
		setAttr(Attrs.y, y, 0);
	}

}