/* Marker.java

	Purpose:
		
	Description:
		
	History:
		Tue, Jan 14, 2014  5:41:03 PM, Created by RaymondChao

Copyright (C) 2014 Potix Corporation. All Rights Reserved.

 */
package org.zkoss.chart;

import org.zkoss.chart.util.DynamicalAttribute;

/**
 * The options for the marker
 * <p>
 * All the options in this class support {@link DynamicalAttribute}.
 * 
 * @author RaymondChao
 */
public class Marker extends Optionable {
	private enum Attrs implements PlotAttribute, DynamicalAttribute {
		enabled, fillColor, lineColor, lineWidth, radius, states, symbol
	}

	/**
	 * Returns enable or disable the point marker.
	 * <p>
	 * Default: true
	 */
	public boolean isEnabled() {
		return getAttr(Attrs.enabled, true).asBoolean();
	}

	/**
	 * Sets enable or disable the point marker.
	 */
	public void setEnabled(boolean enabled) {
		setAttr(Attrs.enabled, enabled);
	}

	/**
	 * Returns the fill color of the point marker. When <code>null</code>, the
	 * series' or point's color is used.
	 * <p>
	 * Default: null
	 */
	public Color getFillColor() {
		return (Color) getAttr(Attrs.fillColor, null).asValue();
	}

	/**
	 * Sets the fill color of the point marker. When <code>null</code>, the
	 * series' or point's color is used.
	 */
	public void setFillColor(Color color) {
		setAttr(Attrs.fillColor, color);
	}

	/**
	 * Sets the fill color of the point marker. When <code>null</code>, the
	 * series' or point's color is used.
	 */
	public void setFillColor(String color) {
		setFillColor(new Color(color));
	}

	/**
	 * Sets the fill color of the point marker. When <code>null</code>, the
	 * series' or point's color is used.
	 */
	public void setFillColor(LinearGradient color) {
		setFillColor(new Color(color));
	}

	/**
	 * Sets the fill color of the point marker. When <code>null</code>, the
	 * series' or point's color is used.
	 */
	public void setFillColor(RadialGradient color) {
		setFillColor(new Color(color));
	}

	/**
	 * Returns the color of the point marker's outline. When <code>null</code>,
	 * the series' or point's color is used.
	 * <p>
	 * Default: "#FFFFFF"
	 */
	public Color getLineColor() {
		if (!containsKey(Attrs.lineColor)) {
			setLineColor("#FFFFFF");
		}
		return (Color) getAttr(Attrs.lineColor);
	}

	/**
	 * Sets the color of the point marker's outline. When <code>null</code>, the
	 * series' or point's color is used. 'lineColor the color of the point
	 * marker's outline. When <code>null</code>, the series' or point's color is
	 * used.
	 */
	public void setLineColor(Color color) {
		setAttr(Attrs.lineColor, color, NOT_NULL_VALUE);
	}

	/**
	 * Sets the color of the point marker's outline. When <code>null</code>, the
	 * series' or point's color is used. 'lineColor the color of the point
	 * marker's outline. When <code>null</code>, the series' or point's color is
	 * used.
	 */
	public void setLineColor(String color) {
		setLineColor(new Color(color));
	}

	/**
	 * Sets the color of the point marker's outline. When <code>null</code>, the
	 * series' or point's color is used. 'lineColor the color of the point
	 * marker's outline. When <code>null</code>, the series' or point's color is
	 * used.
	 */
	public void setLineColor(LinearGradient color) {
		setLineColor(new Color(color));
	}

	/**
	 * Sets the color of the point marker's outline. When <code>null</code>, the
	 * series' or point's color is used. 'lineColor the color of the point
	 * marker's outline. When <code>null</code>, the series' or point's color is
	 * used.
	 */
	public void setLineColor(RadialGradient color) {
		setLineColor(new Color(color));
	}

	/**
	 * Returns the width of the point marker's outline.
	 * <p>
	 * Default: 0
	 */
	public Number getLineWidth() {
		return getAttr(Attrs.lineWidth, 0).asNumber();
	}

	/**
	 * Sets the width of the point marker's outline. 'lineWidth the width of the
	 * point marker's outline.
	 */
	public void setLineWidth(Number lineWidth) {
		setAttr(Attrs.lineWidth, lineWidth, 0);
	}

	/**
	 * Returns the radius of the point marker.
	 * <p>
	 * Default: 4
	 */
	public Number getRadius() {
		return getAttr(Attrs.radius, 4).asNumber();
	}

	/**
	 * Sets the radius of the point marker. 'radius the radius of the point
	 * marker.
	 */
	public void setRadius(Number radius) {
		setAttr(Attrs.radius, radius, 4);
	}

	/**
	 * Returns a wrapper object for all the series options in specific states.
	 */
	public States getStates() {
		States states = (States) getAttr(Attrs.states);
		if (states == null) {
			states = new States();
			setStates(states);
		}
		return states;
	}

	/**
	 * Sets a wrapper object for all the series options in specific states.
	 */

	public void setStates(States states) {
		setAttr(Attrs.states, states);
	}

	/**
	 * Returns a predefined shape or symbol for the marker. When null, the
	 * symbol is pulled from options.symbols. Other possible values are
	 * "circle", "square", "diamond", "triangle" and "triangle-down". </p>
	 * <p>
	 * Additionally, the URL to a graphic can be given on this form:
	 * "url(graphic.png)". Note that for the image to be applied to exported
	 * charts, its URL needs to be accessible by the export server.
	 * </p>
	 * <p>
	 * Custom callbacks for symbol path generation can also be added to
	 * <code>Highcharts.SVGRenderer.prototype.symbols</code>. The callback is
	 * then used by its method name, as shown in the demo.
	 * </p>
	 */
	public String getSymbol() {
		return getAttr(Attrs.symbol, null).asString();
	}

	/**
	 * Sets a predefined shape or symbol for the marker. When null, the symbol
	 * is pulled from options.symbols. Other possible values are "circle",
	 * "square", "diamond", "triangle" and "triangle-down". </p>
	 * <p>
	 * Additionally, the URL to a graphic can be given on this form:
	 * "url(graphic.png)". Note that for the image to be applied to exported
	 * charts, its URL needs to be accessible by the export server.
	 * </p>
	 * <p>
	 * Custom callbacks for symbol path generation can also be added to
	 * <code>Highcharts.SVGRenderer.prototype.symbols</code>. The callback is
	 * then used by its method name, as shown in the demo.
	 * </p>
	 */
	public void setSymbol(String symbol) {
		if (symbol != null
				&& !("circle".equals(symbol) || "square".equals(symbol)
						|| "diamond".equals(symbol)
						|| "triangle".equals(symbol) || "triangle-down"
							.equals(symbol) || symbol.matches("url(.*)")))
			throw new IllegalArgumentException("Unsupported symbol: [" + symbol + "]");
		setAttr(Attrs.symbol, symbol);
	}

}