/** Loading.java.

	Purpose:
		
	Description:
		
	History:
		10:10:46 AM Jan 28, 2014, Created by jumperchen

Copyright (C) 2014 Potix Corporation. All Rights Reserved.
 */
package org.zkoss.chart;

import java.util.LinkedHashMap;
import java.util.Map;

import org.zkoss.chart.util.DynamicalAttribute;
import org.zkoss.chart.util.MapsHelper;
import org.zkoss.lang.Generics;

/**
 * The loading options control the appearance of the loading screen that covers
 * the plot area on chart operations. This screen only appears after an explicit
 * call to {@link Charts#showLoading(String)}.
 * 
 * <p> All the options in this class support {@link DynamicalAttribute}.
 * @author jumperchen
 */
public class Loading extends Optionable {
	private enum Attrs implements PlotAttribute, DynamicalAttribute {
		hideDuration, labelStyle, showDuration, style
	}

	/**
	 * Sets the duration in milliseconds of the fade out effect.
	 * <p>
	 * Defaults to 100.
	 * 
	 * @param hideDuration
	 */
	public void setHideDuration(Number hideDuration) {
		setAttr(Attrs.hideDuration, hideDuration, 100);
	}

	/**
	 * Returns the duration in milliseconds of the fade out effect.
	 * <p>
	 * Defaults to 100.
	 */
	public Number getHideDuration() {
		return getAttr(Attrs.hideDuration, 100).asNumber();
	}

	/**
	 * Sets CSS styles for the loading label span. Defaults to:
	 * 
	 * <pre>
	 * 	fontWeight: bold; position: relative; top: 1em;
	 * </pre>
	 * 
	 * @param labelStyle
	 * @see #setLabelStyle(Map)
	 */
	public void setLabelStyle(String labelStyle) {
		setLabelStyle(MapsHelper.parse(new LinkedHashMap<String, String>(), labelStyle, ':', ';', '\''));
	}

	/**
	 * Sets CSS styles for the loading label span. Defaults to:
	 * 
	 * <pre>
	 * 	fontWeight: bold; position: relative; top: 1em;
	 * </pre>
	 * 
	 * @param labelStyle
	 * @see #setLabelStyle(String)
	 */
	public <K, V> void setLabelStyle(Map<K, V> labelStyle) {
		setAttr(Attrs.labelStyle, labelStyle);
	}

	/**
	 * Returns CSS styles for the loading label span
	 */
	public <K, V> Map<K, V> getLabelStyle() {
		return Generics.cast(getAttr(Attrs.labelStyle, null).asValue());
	}

	/**
	 * Sets the duration in milliseconds of the fade in effect.
	 * <p>
	 * Defaults to 100.
	 * 
	 * @param showDuration
	 */
	public void setShowDuration(Number showDuration) {
		setAttr(Attrs.showDuration, showDuration, 100);
	}

	/**
	 * Returns the duration in milliseconds of the fade in effect.
	 * <p>
	 * Defaults to 100.
	 */
	public Number getShowDuration() {
		return getAttr(Attrs.showDuration, 100).asNumber();
	}

	/**
	 * Returns CSS styles for the label. When labels are rotated they are
	 * rendered using vector graphic techniques and not all styles are
	 * applicable. Default:
	 * 
	 * <pre>
	 * position: absolute; backgroundColor: white; opacity: 0.5; textAlign: center
	 * </pre>
	 * <p>
	 * 
	 * @return style CSS styles for the label
	 */
	public <K, V> Map<K, V> getStyle() {
		return Generics.cast(getAttr(Attrs.style, null).asValue());
	}

	/**
	 * Sets CSS styles for the loading screen that covers the plot area.
	 * Defaults to:
	 * 
	 * <pre>
	 * position: absolute; backgroundColor: white; opacity: 0.5; textAlign: center
	 * </pre>
	 * 
	 * @param style
	 *            CSS styles for the label
	 */
	public void setStyle(String style) {
		setStyle(MapsHelper.parse(new LinkedHashMap<String, String>(), style, ':', ';', '\''));
	}

	/**
	 * Sets CSS styles for the loading screen that covers the plot area.
	 * Defaults to:
	 * 
	 * <pre>
	 * position: absolute; backgroundColor: white; opacity: 0.5; textAlign: center
	 * </pre>
	 * 
	 * @param style
	 *            CSS styles for the label
	 */
	public <K, V> void setStyle(Map<K, V> style) {
		setAttr(Attrs.style, style);
	}
}
