/** LinearGradient.java.

	Purpose:
		
	Description:
		
	History:
		12:17:03 PM Jan 16, 2014, Created by jumperchen

Copyright (C) 2014 Potix Corporation. All Rights Reserved.
 */
package org.zkoss.chart;

import java.util.Arrays;
import java.util.List;

import org.zkoss.chart.util.DynamicalAttribute;
import org.zkoss.json.JSONArray;
import org.zkoss.json.JSONObject;
import org.zkoss.lang.Generics;

/**
 * A linear gradient is a smooth transition from one color to another. In
 * addition, several color transitions can be applied to the same element.
 *
 * <p> All the options in this class support {@link DynamicalAttribute}. 
 * @author jumperchen
 */
public class LinearGradient extends Optionable {
	private enum Attrs implements PlotAttribute, DynamicalAttribute {
		linearGradient, stops
	}

	/**
	 * The x1, x2, y1,y2 attributes of the linearGradient define the start and
	 * end position of the gradient
	 */
	public LinearGradient(double x1, double y1, double x2, double y2) {
		// for pixel range
		if (x2 > 1 || y2 > 1) {
			JSONArray json = new JSONArray();
			json.add(x1);
			json.add(y1);
			json.add(x2);
			json.add(y2);
			setAttr(Attrs.linearGradient, json);
		} else {
			JSONObject json = new JSONObject();
			json.put("x1", x1);
			json.put("y1", y1);
			json.put("x2", x2);
			json.put("y2", y2);
			setAttr(Attrs.linearGradient, json);
		}
	}

	/**
	 * Adds the color range for a gradient can be composed of two or more colors.
	 * Each color is specified with a offset in order.
	 */
	public void addStop(double index, String color) {
		JSONArray json = (JSONArray) getAttr(Attrs.stops);
		if (json == null) {
			json = new JSONArray();
		}
		JSONArray arr = new JSONArray();
		arr.add(index);
		arr.add(color);
		json.add(arr);
		setAttr(Attrs.stops, json);
	}
	
	/**
	 * Sets the color range for a gradient can be composed of two colors.
	 * It is the same as<br/>
	 * <pre><code>
	 * setStops(Arrays.asList(new Object[][]{
				new Object[]{0, color1},
				new Object[]{1, color2}
		}));
	 * </code></pre>
	 *
	 * @see #setStops(Object... stops)
	 */
	public <T> void setStops(String color1, String color2) {
	    setStops(new Object[]{0, color1}, new Object[]{1, color2});
	}

	/**
	 * Sets the color range for a gradient can be composed of two or more colors.
	 * Each color is specified with a offset in order.
	 * <p>
	 * For example,
	 * <pre><code>
	 * setStops(Arrays.asList(new Object[][]{
				new Object[]{0, "rgb(255, 255, 255)"},
				new Object[]{1, "rgb(200, 200, 255)"}
		}));
	 * </code></pre>
	 */
	public <T> void setStops(List<T> stops) {
		setAttr(Attrs.stops, stops);
	}

	/**
	 * Sets the color range for a gradient can be composed of two or more colors.
	 * Each color is specified with a offset in order.
	 * <p>
	 * For example,
	 * <pre><code>
	 * setStops(new Object[]{0, "rgb(255, 255, 255)"},
				new Object[]{1, "rgb(200, 200, 255)"});
	 * </code></pre>
	 */
	public void setStops(Object... stops) {
		setStops(Arrays.asList(stops));
	}

	/**
	 * Returns the all stops
	 */
	public <T> T getStops() {
		return Generics.cast(getAttr(Attrs.stops, null).asValue());
	}
}
