/** LegendNavigation.java.

	Purpose:
		
	Description:
		
	History:
		9:45:33 AM Jan 13, 2014, Created by RaymondChao

Copyright (C) 2014 Potix Corporation. All Rights Reserved.
 */
package org.zkoss.chart;

import java.util.LinkedHashMap;
import java.util.Map;

import org.zkoss.chart.util.DynamicalAttribute;
import org.zkoss.chart.util.MapsHelper;
import org.zkoss.lang.Generics;

/**
 * Options for the paging or navigation appearing when the legend is overflown.
 * When legend.useHTML is enabled, navigation is disabled.
 * <p>
 * All the options in this class support {@link DynamicalAttribute}.
 * 
 * @author jumperchen
 * @author RaymondChao
 */
public class LegendNavigation extends Optionable {
	private enum Attrs implements PlotAttribute, DynamicalAttribute {
		activeColor, animation, arrowSize, inactiveColor, style
	}

	/**
	 * Returns the color for the active up or down arrow in the legend page
	 * navigation.
	 * <p>
	 * Default: "#3E576F".
	 */
	public Color getActiveColor() {
		if (!containsKey(Attrs.activeColor)) {
			setActiveColor("#3E576F");
		}
		return (Color) getAttr(Attrs.activeColor);
	}

	/**
	 * Sets the color for the active up or down arrow in the legend page
	 * navigation.
	 */
	public void setActiveColor(Color color) {
		setAttr(Attrs.activeColor, color, NOT_NULL_VALUE);
	}

	/**
	 * Sets the color for the active up or down arrow in the legend page
	 * navigation.
	 */
	public void setActiveColor(String color) {
		setActiveColor(new Color(color));
	}

	/**
	 * Sets the color for the active up or down arrow in the legend page
	 * navigation.
	 */
	public void setActiveColor(LinearGradient color) {
		setActiveColor(new Color(color));
	}

	/**
	 * Sets the color for the active up or down arrow in the legend page
	 * navigation.
	 */
	public void setActiveColor(RadialGradient color) {
		setActiveColor(new Color(color));
	}

	/**
	 * Returns whether to animate the pages when navigating up or down. A value
	 * of <code>true</code> applies the default navigation given in the
	 * chart.animation option. Additional options can be given as an object
	 * containing values for easing and duration. .
	 * <p>
	 * Default: true.
	 */
	public boolean isAnimation() {
		return getAnimation() != Animation.NONE;
	}

	/**
	 * Returns whether to animate the pages when navigating up or down. A value
	 * of <code>true</code> applies the default navigation given in the
	 * chart.animation option
	 * <p>
	 * Default: an instance of {@link Animation}
	 */
	public Animation getAnimation() {
		return (Animation) getAttr(Attrs.animation,
				org.zkoss.chart.Animation.class).asValue();
	}

	/**
	 * Sets whether to animate the pages when navigating up or down. A value of
	 * <code>true</code> applies the default navigation given in the
	 * chart.animation option. Additional options can be given as an object
	 * containing values for easing and duration.
	 * 
	 * @see #setAnimation(Animation)
	 * @see Animation
	 */
	public void setAnimation(boolean animation) {
		setAnimation(animation ? new Animation() : Animation.NONE);
	}

	/**
	 * Sets whether to animate the pages when navigating up or down. A value of
	 * <code>true</code> applies the default navigation given in the
	 * chart.animation option. Additional options can be given as an object
	 * containing values for easing and duration.
	 * 
	 * @see #setAnimation(boolean)
	 * @see Animation
	 */
	public void setAnimation(Animation animation) {
		setAttr(Attrs.animation, animation == null ? Animation.NONE : animation);
	}

	/**
	 * Returns the pixel size of the up and down arrows in the legend paging
	 * navigation.
	 * <p>
	 * Default: 12.
	 */
	public Number getArrowSize() {
		return getAttr(Attrs.arrowSize, 12).asNumber();
	}

	/**
	 * Sets the pixel size of the up and down arrows in the legend paging
	 * navigation.
	 */
	public void setArrowSize(Number arrowSize) {
		setAttr(Attrs.arrowSize, arrowSize, 12);
	}

	/**
	 * Returns the color of the inactive up or down arrow in the legend page
	 * navigation.
	 * <p>
	 * Default: "#CCC".
	 */
	public Color getIninactiveColor() {
		if (!containsKey(Attrs.inactiveColor)) {
			setIninactiveColor("#CCC");
		}
		return (Color) getAttr(Attrs.inactiveColor);
	}

	/**
	 * Sets the color of the inactive up or down arrow in the legend page
	 * navigation.
	 */
	public void setIninactiveColor(Color color) {
		setAttr(Attrs.inactiveColor, color, NOT_NULL_VALUE);
	}

	/**
	 * Sets the color of the inactive up or down arrow in the legend page
	 * navigation.
	 */
	public void setIninactiveColor(String color) {
		setIninactiveColor(new Color(color));
	}

	/**
	 * Sets the color of the inactive up or down arrow in the legend page
	 * navigation.
	 */
	public void setIninactiveColor(LinearGradient color) {
		setIninactiveColor(new Color(color));
	}

	/**
	 * Sets the color of the inactive up or down arrow in the legend page
	 * navigation.
	 */
	public void setIninactiveColor(RadialGradient color) {
		setIninactiveColor(new Color(color));
	}

	/**
	 * Returns the text styles for the legend page navigation.
	 * <p>
	 * Default: null.
	 */
	public <K, V> Map<K, V> getStyle() {
		return Generics.cast(getAttr(Attrs.style, null).asValue());
	}

	/**
	 * Sets the text styles for the legend page navigation.
	 * 
	 * @see #setStyle(Map)
	 */
	public void setStyle(String style) {
		setStyle(MapsHelper.parse(new LinkedHashMap<String, String>(), style, ':', ';', '\''));
	}

	/**
	 * Sets the text styles for the legend page navigation.
	 * 
	 * @see #setStyle(String)
	 */
	public <K, V> void setStyle(Map<K, V> style) {
		setAttr(Attrs.style, style);
	}

}