/** Legend.java.

	Purpose:
		
	Description:
		
	History:
		9:40:20 AM Jan 13, 2014, Created by RaymondChao

Copyright (C) 2014 Potix Corporation. All Rights Reserved.
 */
package org.zkoss.chart;

import java.util.LinkedHashMap;
import java.util.Map;

import org.zkoss.chart.util.DynamicalAttribute;
import org.zkoss.chart.util.MapsHelper;
import org.zkoss.lang.Generics;

/**
 * The legend is a box containing a symbol and name for each series item or
 * point item in the chart.
 * <p>
 * All the options in this class support {@link DynamicalAttribute}.
 * 
 * @author jumperchen
 * @author RaymondChao
 */
public class Legend extends Optionable {

	private enum Attrs implements PlotAttribute, DynamicalAttribute {
		align,
		backgroundColor,
		borderColor,
		borderRadius,
		borderWidth,
		enabled,
		floating,
		itemDistance,
		itemHiddenStyle,
		itemHoverStyle,
		itemMarginBottom,
		itemMarginTop,
		itemStyle,
		itemWidth,
		labelFormat,
		layout,
		margin,
		maxHeight,
		navigation,
		padding,
		reversed,
		rtl,
		shadow,
		style,
		symbolHeight,
		symbolPadding,
		symbolRadius,
		symbolWidth,
		title,
		useHTML,
		verticalAlign,
		width,
		x,
		y
	}

	/**
	 * Returns the horizontal alignment of the legend box within the chart area.
	 * Valid values are <code>"left"</code>, <code>"center"</code> and
	 * <code>"right"</code>.
	 * <p>
	 * Default: "center".
	 */
	public String getAlign() {
		return getAttr(Attrs.align, "center").asString();
	}

	/**
	 * Sets the horizontal alignment of the legend box within the chart area.
	 * Valid values are <code>"left"</code>, <code>"center"</code> and
	 * <code>"right"</code>.
	 */
	public void setAlign(String align) {
		if (!("left".equals(align) || "center".equals(align) || "right"
				.equals(align)))
			throw new IllegalArgumentException("Unsupported align: [" + align
					+ "]");
		setAttr(Attrs.align, align, "center");
	}

	/**
	 * Returns the background color of the legend, filling the rounded corner
	 * border.
	 * <p>
	 * Default: null.
	 */
	public Color getBackgroundColor() {
		return (Color) getAttr(Attrs.backgroundColor, null).asValue();
	}

	/**
	 * Sets the background color of the legend, filling the rounded corner
	 * border.
	 */
	public void setBackgroundColor(Color color) {
		setAttr(Attrs.backgroundColor, color);
	}

	/**
	 * Sets the background color of the legend, filling the rounded corner
	 * border.
	 */
	public void setBackgroundColor(String color) {
		setBackgroundColor(new Color(color));
	}

	/**
	 * Sets the background color of the legend, filling the rounded corner
	 * border.
	 */
	public void setBackgroundColor(LinearGradient color) {
		setBackgroundColor(new Color(color));
	}

	/**
	 * Sets the background color of the legend, filling the rounded corner
	 * border.
	 */
	public void setBackgroundColor(RadialGradient color) {
		setBackgroundColor(new Color(color));
	}

	/**
	 * Returns the color of the drawn border around the legend.
	 * <p>
	 * Default: "#909090".
	 */
	public Color getBorderColor() {
		if (!containsKey(Attrs.borderColor)) {
			setBorderColor("#909090");
		}
		return (Color) getAttr(Attrs.borderColor);
	}

	/**
	 * Sets the color of the drawn border around the legend.
	 */
	public void setBorderColor(Color color) {
		setAttr(Attrs.borderColor, color, NOT_NULL_VALUE);
	}

	/**
	 * Sets the color of the drawn border around the legend.
	 */
	public void setBorderColor(String color) {
		setBorderColor(new Color(color));
	}

	/**
	 * Sets the color of the drawn border around the legend.
	 */
	public void setBorderColor(LinearGradient color) {
		setBorderColor(new Color(color));
	}

	/**
	 * Sets the color of the drawn border around the legend.
	 */
	public void setBorderColor(RadialGradient color) {
		setBorderColor(new Color(color));
	}
	
	/**
	 * Returns the border corner radius of the legend.
	 * <p>
	 * Default: 5.
	 * 
	 * @return borderRadius the border corner radius of the legend
	 */
	public Number getBorderRadius() {
		return getAttr(Attrs.borderRadius, 5).asNumber();
	}

	/**
	 * Sets the border corner radius of the legend.
	 * 
	 * @param borderRadius
	 *            the border corner radius of the legend
	 */
	public void setBorderRadius(Number borderRadius) {
		setAttr(Attrs.borderRadius, borderRadius, 5);
	}

	/**
	 * Returns the width of the drawn border around the legend.
	 * <p>
	 * Default: 1.
	 * 
	 * @return borderWidth the width of the drawn border around the legend
	 */
	public Number getBorderWidth() {
		return getAttr(Attrs.borderWidth, 1).asNumber();
	}

	/**
	 * Sets the width of the drawn border around the legend.
	 */
	public void setBorderWidth(Number borderWidth) {
		setAttr(Attrs.borderWidth, borderWidth, 1);
	}

	/**
	 * Returns enable or disable the legend.
	 * <p>
	 * Default: true.
	 * 
	 * @return enabled enable or disable the legend
	 */
	public boolean isEnabled() {
		return getAttr(Attrs.enabled, true).asBoolean();
	}

	/**
	 * Sets enable or disable the legend.
	 * 
	 * @param enabled
	 *            enable or disable the legend
	 */
	public void setEnabled(boolean enabled) {
		setAttr(Attrs.enabled, enabled);
	}

	/**
	 * Returns when the legend is floating, the plot area ignores it and is
	 * allowed to be placed below it.
	 * <p>
	 * Default: false.
	 * 
	 * @return floating when the legend is floating, the plot area ignores it
	 *         and is allowed to be placed below it
	 */
	public boolean isFloating() {
		return getAttr(Attrs.floating, false).asBoolean();
	}

	/**
	 * Sets when the legend is floating, the plot area ignores it and is allowed
	 * to be placed below it.
	 * 
	 * @param floating
	 *            when the legend is floating, the plot area ignores it and is
	 *            allowed to be placed below it
	 */
	public void setFloating(boolean floating) {
		setAttr(Attrs.floating, floating);
	}

	/**
	 * Returns in a legend with horizontal layout, the itemDistance defines the
	 * pixel distance between each item.
	 * <p>
	 * Default: 8.
	 * 
	 * @return itemDistance in a legend with horizontal layout, the itemDistance
	 *         defines the pixel distance between each item
	 */
	public Number getItemDistance() {
		return getAttr(Attrs.itemDistance, 8).asNumber();
	}

	/**
	 * Sets in a legend with horizontal layout, the itemDistance defines the
	 * pixel distance between each item.
	 * 
	 * @param itemDistance
	 *            in a legend with horizontal layout, the itemDistance defines
	 *            the pixel distance between each item
	 */
	public void setItemDistance(Number itemDistance) {
		setAttr(Attrs.itemDistance, itemDistance, 8);
	}

	/**
	 * Returns CSS styles for each legend item when the corresponding series or
	 * point is hidden. Only a subset of CSS is supported, notably those options
	 * related to text. Properties are inherited from <code>style</code> unless
	 * overridden here. Defaults to:
	 * 
	 * <pre>
	 * color: #CCC;
	 * </pre>
	 * <p>
	 */
	public <K, V> Map<K, V> getItemHiddenStyle() {
		if (!containsKey(Attrs.itemHiddenStyle)) {
			setItemHiddenStyle("color: #CCC;");
		}
		return Generics.cast(getAttr(Attrs.itemHiddenStyle));
	}

	/**
	 * Sets CSS styles for each legend item when the corresponding series or
	 * point is hidden. Only a subset of CSS is supported, notably those options
	 * related to text. Properties are inherited from <code>style</code> unless
	 * overridden here. Defaults to:
	 * 
	 * <pre>
	 * color: #CCC;
	 * </pre>
	 * 
	 * @see #setItemHiddenStyle(Map)
	 */
	public void setItemHiddenStyle(String itemHiddenStyle) {
		setItemHiddenStyle(MapsHelper.parse(new LinkedHashMap<String, String>(), itemHiddenStyle, ':', ';', '\''));
	}

	/**
	 * Sets CSS styles for each legend item when the corresponding series or
	 * point is hidden. Only a subset of CSS is supported, notably those options
	 * related to text. Properties are inherited from <code>style</code> unless
	 * overridden here. Defaults to:
	 * 
	 * <pre>
	 * color: #CCC;
	 * </pre>
	 * 
	 * @see #setItemHiddenStyle(String)
	 */
	public <K, V> void setItemHiddenStyle(Map<K, V> itemHiddenStyle) {
		setAttr(Attrs.itemHiddenStyle, itemHiddenStyle, NOT_NULL_VALUE);
	}

	/**
	 * Returns CSS styles for each legend item in hover mode. Only a subset of
	 * CSS is supported, notably those options related to text. Properties are
	 * inherited from <code>style</code> unless overridden here. Defaults to:
	 * 
	 * <pre>
	 * color: #000;
	 * </pre>
	 * <p>
	 */
	public <K, V> Map<K, V> getItemHoverStyle() {
		if (!containsKey(Attrs.itemHoverStyle)) {
			setItemHoverStyle("color: #CCC;");
		}
		return Generics.cast(getAttr(Attrs.itemHoverStyle));
	}

	/**
	 * Sets CSS styles for each legend item in hover mode. Only a subset of CSS
	 * is supported, notably those options related to text. Properties are
	 * inherited from <code>style</code> unless overridden here. Defaults to:
	 * 
	 * <pre>
	 * color: #000;
	 * </pre>
	 * 
	 * @see #setItemHoverStyle(Map)
	 */
	public void setItemHoverStyle(String itemHoverStyle) {
		setItemHoverStyle(MapsHelper.parse(new LinkedHashMap<String, String>(), itemHoverStyle, ':', ';', '\''));
	}

	/**
	 * Sets CSS styles for each legend item in hover mode. Only a subset of CSS
	 * is supported, notably those options related to text. Properties are
	 * inherited from <code>style</code> unless overridden here. Defaults to:
	 * 
	 * <pre>
	 * color: #000;
	 * </pre>
	 * 
	 * @see #setItemHoverStyle(String)
	 */
	public <K, V> void setItemHoverStyle(Map<K, V> itemHoverStyle) {
		setAttr(Attrs.itemHoverStyle, itemHoverStyle, NOT_NULL_VALUE);
	}

	/**
	 * Returns the pixel bottom margin for each legend item.
	 * <p>
	 * Default: 0
	 */
	public Number getItemMarginBottom() {
		return getAttr(Attrs.itemMarginBottom, 0).asNumber();
	}

	/**
	 * Sets the pixel bottom margin for each legend item.
	 */
	public void setItemMarginBottom(Number itemMarginBottom) {
		setAttr(Attrs.itemMarginBottom, itemMarginBottom, 0);
	}

	/**
	 * Returns the pixel top margin for each legend item.
	 * <p>
	 * Default: 0
	 */
	public Number getItemMarginTop() {
		return getAttr(Attrs.itemMarginTop, 0).asNumber();
	}

	/**
	 * Sets the pixel top margin for each legend item.
	 */
	public void setItemMarginTop(Number itemMarginTop) {
		setAttr(Attrs.itemMarginTop, itemMarginTop, 0);
	}

	/**
	 * Returns CSS styles for each legend item. Only a subset of CSS is
	 * supported, notably those options related to text. Defaults to:
	 * 
	 * <pre>
	 * cursor: pointer; color: #274b6d; fontSize: 12px;
	 * </pre>
	 * 
	 * @return itemStyle cSS styles for each legend item
	 */
	public <K, V> Map<K, V> getItemStyle() {
		if (!containsKey(Attrs.itemStyle)) {
			setItemStyle("cursor: pointer; color: #274b6d; fontSize: 12px;");
		}
		return Generics.cast(getAttr(Attrs.itemStyle));
	}

	/**
	 * Sets CSS styles for each legend item. Only a subset of CSS is supported,
	 * notably those options related to text. Defaults to:
	 * 
	 * <pre>
	 * cursor: pointer; color: #274b6d; fontSize: 12px;
	 * </pre>
	 * 
	 * @see #setItemStyle(Map)
	 */
	public void setItemStyle(String itemStyle) {
		setItemStyle(MapsHelper.parse(new LinkedHashMap<String, String>(), itemStyle, ':', ';', '\''));
	}

	/**
	 * Sets CSS styles for each legend item. Only a subset of CSS is supported,
	 * notably those options related to text. Defaults to:
	 * 
	 * <pre>
	 * cursor: pointer; color: #274b6d; fontSize: 12px;
	 * </pre>
	 * 
	 * @see #setItemStyle(String)
	 */
	public <K, V> void setItemStyle(Map<K, V> itemStyle) {
		setAttr(Attrs.itemStyle, itemStyle, NOT_NULL_VALUE);
	}

	/**
	 * Returns the width for each legend item. This is useful in a horizontal
	 * layout with many items when you want the items to align vertically. .
	 * <p>
	 * Default: null.
	 */
	public Number getItemWidth() {
		return getAttr(Attrs.itemWidth, null).asNumber();
	}

	/**
	 * Sets the width for each legend item. This is useful in a horizontal
	 * layout with many items when you want the items to align vertically. .
	 * 
	 * @param itemWidth
	 *            the width for each legend item
	 */
	public void setItemWidth(Number itemWidth) {
		setAttr(Attrs.itemWidth, itemWidth);
	}

	/**
	 * Returns a <a href="http://docs.highcharts.com#formatting">format
	 * string</a> for each legend label. Available variables relates to
	 * properties on the series, or the point in case of pies.
	 * <p>
	 * Default: "{name}".
	 */
	public String getLabelFormat() {
		return getAttr(Attrs.labelFormat, "{name}").asString();
	}

	/**
	 * Sets a <a href="http://docs.highcharts.com#formatting">format string</a>
	 * for each legend label. Available variables relates to properties on the
	 * series, or the point in case of pies.
	 */
	public void setLabelFormat(String labelFormat) {
		setAttr(Attrs.labelFormat, labelFormat, "{name}");
	}

	/**
	 * Returns the layout of the legend items. Can be one of "horizontal" or
	 * "vertical".
	 * <p>
	 * Default: "horizontal".
	 */
	public String getLayout() {
		return getAttr(Attrs.layout, "horizontal").asString();
	}

	/**
	 * Sets the layout of the legend items. Can be one of "horizontal" or
	 * "vertical".
	 * 
	 * @param layout
	 *            the layout of the legend items
	 */
	public void setLayout(String layout) {
		if (!("horizontal".equals(layout) || "vertical".equals(layout)))
			throw new IllegalArgumentException("Unsupported Layout: [" + layout
					+ "]");

		setAttr(Attrs.layout, layout, "horizontal");
	}

	/**
	 * Returns if the plot area sized is calculated automatically and the legend
	 * is not floating, the legend margin is the space between the legend and
	 * the axis labels or plot area.
	 * <p>
	 * Default: 15.
	 */
	public Number getMargin() {
		return getAttr(Attrs.margin, 15).asNumber();
	}

	/**
	 * Sets if the plot area sized is calculated automatically and the legend is
	 * not floating, the legend margin is the space between the legend and the
	 * axis labels or plot area.
	 */
	public void setMargin(Number margin) {
		setAttr(Attrs.margin, margin, 15);
	}

	/**
	 * Returns maximum pixel height for the legend. When the maximum height is
	 * extended, navigation will show.
	 * 
	 * @return maxHeight maximum pixel height for the legend
	 */
	public Number getMaxHeight() {
		return getAttr(Attrs.maxHeight, 0).asNumber();
	}

	/**
	 * Sets maximum pixel height for the legend. When the maximum height is
	 * extended, navigation will show.
	 * 
	 * @param maxHeight
	 *            maximum pixel height for the legend
	 */
	public void setMaxHeight(Number maxHeight) {
		setAttr(Attrs.maxHeight, maxHeight, 0);
	}

	/**
	 * Returns options for the paging or navigation appearing when the legend is
	 * overflown. When {@link #setUseHTML(boolean)} is enabled, navigation is
	 * disabled.
	 * <p>
	 * Default: an instance of {@link LegendNavigation}
	 * 
	 * @see LegendNavigation
	 */
	public LegendNavigation getNavigation() {
		LegendNavigation navigation = (LegendNavigation) getAttr(Attrs.navigation);
		if (navigation == null) {
			navigation = new LegendNavigation();
			setNavigation(navigation);
		}
		return navigation;
	}

	/**
	 * Sets options for the paging or navigation appearing when the legend is
	 * overflown. When {@link #setUseHTML(boolean)} is enabled, navigation is
	 * disabled.
	 * 
	 * @see LegendNavigation
	 */
	public void setNavigation(LegendNavigation navigation) {
		setAttr(Attrs.navigation, navigation);
	}

	/**
	 * Returns the inner padding of the legend box.
	 * <p>
	 * Default: 8.
	 * 
	 * @return padding the inner padding of the legend box
	 */
	public Number getPadding() {
		return getAttr(Attrs.padding, 8).asNumber();
	}

	/**
	 * Sets the inner padding of the legend box.
	 * 
	 * @param padding
	 *            the inner padding of the legend box
	 */
	public void setPadding(Number padding) {
		setAttr(Attrs.padding, padding, 8);
	}

	/**
	 * Returns whether to reverse the order of the legend items compared to the
	 * order of the series or points as defined in the configuration object.
	 * <p>
	 * Default: false.
	 */
	public boolean isReversed() {
		return getAttr(Attrs.reversed, false).asBoolean();
	}

	/**
	 * Sets whether to reverse the order of the legend items compared to the
	 * order of the series or points as defined in the configuration object.
	 */
	public void setReversed(boolean reversed) {
		setAttr(Attrs.reversed, reversed);
	}

	/**
	 * Returns whether to show the symbol on the right side of the text rather
	 * than the left side. This is common in Arabic and Hebraic.
	 * <p>
	 * Default: false.
	 */
	public boolean isRtl() {
		return getAttr(Attrs.rtl, false).asBoolean();
	}

	/**
	 * Sets whether to show the symbol on the right side of the text rather than
	 * the left side. This is common in Arabic and Hebraic.
	 */
	public void setRtl(boolean rtl) {
		setAttr(Attrs.rtl, rtl);
	}

	/**
	 * Returns whether to apply a drop shadow to the legend. Requires that
	 * backgroundColor be set.
	 * <p>
	 * Default: false.
	 * 
	 * @see Shadow
	 */
	public boolean isShadow() {
		return getShadow() != Shadow.NONE;
	}

	/**
	 * Returns whether to apply a drop shadow to the legend. Requires that
	 * backgroundColor be set.
	 * <p>
	 * Default: {@link Shadow#NONE}.
	 * 
	 * @see Shadow
	 */
	public Shadow getShadow() {
		return (Shadow) getAttr(Attrs.shadow, Shadow.NONE).asValue();
	}

	/**
	 * Sets whether to apply a drop shadow to the legend. Requires that
	 * backgroundColor be set.
	 */
	public void setShadow(boolean shadow) {
		setShadow(shadow ? new Shadow() : Shadow.NONE);
	}

	/**
	 * Sets whether to apply a drop shadow to the legend. Requires that
	 * backgroundColor be set.
	 * 
	 * @see Shadow
	 */
	public void setShadow(Shadow shadow) {
		setAttr(Attrs.shadow, shadow);
	}

	/**
	 * Returns the pixel height of the legend item symbol.
	 * <p>
	 * Default: 12.
	 */
	public Number getSymbolHeight() {
		return getAttr(Attrs.symbolHeight, 12).asNumber();
	}

	/**
	 * Sets the pixel height of the legend item symbol.
	 */
	public void setSymbolHeight(Number symbolHeight) {
		setAttr(Attrs.symbolHeight, symbolHeight, 12);
	}

	/**
	 * Returns the pixel padding between the legend item symbol and the legend
	 * item text.
	 * <p>
	 * Default: 5.
	 */
	public Number getSymbolPadding() {
		return getAttr(Attrs.symbolPadding, 5).asNumber();
	}

	/**
	 * Sets the pixel padding between the legend item symbol and the legend item
	 * text.
	 */
	public void setSymbolPadding(Number symbolPadding) {
		setAttr(Attrs.symbolPadding, symbolPadding, 5);
	}

	/**
	 * Returns the pixel radius of the legend item symbol.
	 * <p>
	 * Default: 2.
	 */
	public Number getSymbolRadius() {
		return getAttr(Attrs.symbolRadius, 2).asNumber();
	}

	/**
	 * Sets the pixel radius of the legend item symbol.
	 */
	public void setSymbolRadius(Number symbolRadius) {
		setAttr(Attrs.symbolRadius, symbolRadius, 2);
	}

	/**
	 * Returns the pixel width of the legend item symbol.
	 * <p>
	 * Default: 16.
	 * 
	 * @return symbolWidth the pixel width of the legend item symbol
	 */
	public Number getSymbolWidth() {
		return getAttr(Attrs.symbolWidth, 16).asNumber();
	}

	/**
	 * Sets the pixel width of the legend item symbol.
	 */
	public void setSymbolWidth(Number symbolWidth) {
		setAttr(Attrs.symbolWidth, symbolWidth, 16);
	}

	/**
	 * Returns a title to be added on top of the legend.
	 * <p>
	 * Default: .
	 * 
	 * @return title a title to be added on top of the legend
	 */
	public LegendTitle getTitle() {
		LegendTitle title = (LegendTitle) getAttr(Attrs.title);
		if (title == null) {
			title = new LegendTitle();
			setTitle(title);
		}
		return title;
	}

	/**
	 * Sets a title to be added on top of the legend.
	 * 
	 * @param title
	 *            a title to be added on top of the legend
	 */
	public void setTitle(LegendTitle title) {
		setAttr(Attrs.title, title);
	}

	/**
	 * Returns whether to <a
	 * href="http://docs.highcharts.com/#formatting$html">use HTML</a> to render
	 * the legend item texts. When using HTML,
	 * {@link #setNavigation(LegendNavigation)} is disabled.
	 * <p>
	 * Default: false.
	 */
	public boolean isUseHTML() {
		return getAttr(Attrs.useHTML, false).asBoolean();
	}

	/**
	 * Sets whether to <a href="http://docs.highcharts.com/#formatting$html">use
	 * HTML</a> to render the legend item texts. When using HTML,
	 * {@link #setNavigation(LegendNavigation)} is disabled.
	 */
	public void setUseHTML(boolean useHTML) {
		setAttr(Attrs.useHTML, useHTML);
	}

	/**
	 * Returns the vertical alignment of the legend box. Can be one of "top",
	 * "middle" or "bottom". Vertical position can be further determined by the
	 * <code>y</code> option.
	 * <p>
	 * Default: "bottom".
	 */
	public String getVerticalAlign() {
		return getAttr(Attrs.verticalAlign, "bottom").asString();
	}

	/**
	 * Sets the vertical alignment of the legend box. Can be one of "top",
	 * "middle" or "bottom". Vertical position can be further determined by the
	 * <code>y</code> option.
	 */
	public void setVerticalAlign(String verticalAlign) {
		if (!("top".equals(verticalAlign) || "middle".equals(verticalAlign) || "bottom"
				.equals(verticalAlign)))
			throw new IllegalArgumentException("Unsupported vertical align: ["
					+ verticalAlign + "]");
		setAttr(Attrs.verticalAlign, verticalAlign, "bottom");
	}

	/**
	 * Returns the width of the legend box.
	 * <p>
	 * Default: null.
	 * 
	 * @return width the width of the legend box
	 */
	public Number getWidth() {
		return getAttr(Attrs.width, null).asNumber();
	}

	/**
	 * Sets the width of the legend box.
	 */
	public void setWidth(Number width) {
		setAttr(Attrs.width, width);
	}

	/**
	 * Returns the x offset of the legend relative to it's horizontal alignment
	 * <code>align</code> within chart.spacingLeft and chart.spacingRight.
	 * Negative x moves it to the left, positive x moves it to the right.
	 * <p>
	 * Default: 0.
	 */
	public Number getX() {
		return getAttr(Attrs.x, 0).asNumber();
	}

	/**
	 * Sets the x offset of the legend relative to it's horizontal alignment
	 * <code>align</code> within chart.spacingLeft and chart.spacingRight.
	 * Negative x moves it to the left, positive x moves it to the right.
	 */
	public void setX(Number x) {
		setAttr(Attrs.x, x, 0);
	}

	/**
	 * Returns the vertical offset of the legend relative to it's vertical
	 * alignment <code>verticalAlign</code> within chart.spacingTop and
	 * chart.spacingBottom. Negative y moves it up, positive y moves it down.
	 * <p>
	 * Default: 0.
	 */
	public Number getY() {
		return getAttr(Attrs.y, 0).asNumber();
	}

	/**
	 * Sets the vertical offset of the legend relative to it's vertical
	 * alignment <code>verticalAlign</code> within chart.spacingTop and
	 * chart.spacingBottom. Negative y moves it up, positive y moves it down.
	 */
	public void setY(Number y) {
		setAttr(Attrs.y, y, 0);
	}
}