/** Exporting.java.

	Purpose:
		
	Description:
		
	History:
		12:25:28 PM Mar 4, 2014, Created by jumperchen

Copyright (C) 2014 Potix Corporation. All Rights Reserved.
 */
package org.zkoss.chart;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.zkoss.chart.util.AnyVal;
import org.zkoss.chart.util.DynamicalAttribute;
import org.zkoss.lang.Generics;

/**
 * Options for the exporting module. For an overview on the matter, see the <a
 * href
 * ="http://www.highcharts.com/docs/export-module/export-module-overview">docs
 * </a>.
 * <p>
 * All the options in this class support {@link DynamicalAttribute}.
 * 
 * @author jumperchen
 */
public class Exporting extends Optionable {
	private enum Attrs implements PlotAttribute, DynamicalAttribute {
		buttons,
		chartOptions,
		enabled,
		filename,
		formAttributes,
		scale,
		sourceHeight,
		sourceWidth,
		type,
		url,
		width
	}

	/**
	 * Sets an options for the export context button.
	 * 
	 * @param button
	 */
	public void setButtons(ExportingButton button) {
		LinkedHashMap<String, ExportingButton> map = new LinkedHashMap<String, ExportingButton>(
				1);
		map.put("contextButton", button);
		setButtons(map);
	}
	
	/**
	 * Sets multiple buttons within a map
	 * @param buttons
	 */
	public void setButtons(Map<String, ExportingButton> buttons) {
		setAttr(Attrs.buttons, buttons, NOT_NULL_VALUE);
	}

	/**
	 * Returns context button for the export button.
	 */
	public ExportingButton getButtons() {
		return getButtonsMap().get("contextButton");
	}

	/**
	 * Returns a map of buttons for the export button.
	 */
	public Map<String, ExportingButton> getButtonsMap() {
		Map<String, ExportingButton> map = (Map<String, ExportingButton>) getAttr(
				Attrs.buttons, null).asValue();
		if (map == null) {
			ExportingButton button = new ExportingButton();
			setButtons(button);
		}
		return (HashMap<String, ExportingButton>) getAttr(
				Attrs.buttons, null).asValue();
	}

	/**
	 * Sets additional chart options to be merged into an exported chart. For
	 * example, the exported chart can be given a specific width and height, or
	 * a printer-friendly color scheme. Defaults to null.
	 */
	public <K, V> void setChartOptions(Map<K, V>  chartOptions) {
		setAttr(Attrs.chartOptions, chartOptions);
	}

	/**
	 * Returns additional chart options to be merged into an exported chart. For
	 * example, the exported chart can be given a specific width and height, or
	 * a printer-friendly color scheme.
	 * <p>
	 * Defaults to null.
	 */
	public <K, V> Map<K, V> getChartOptions() {
		return Generics.cast(getAttr(Attrs.chartOptions, null).asValue());
	}

	/**
	 * Returns whether to enable buttons.
	 * <p>
	 * Default: true
	 */
	public boolean isEnabled() {
		return getAttr(Attrs.enabled, true).asBoolean();
	}

	/**
	 * Sets whether to enable buttons.
	 */
	public void setEnabled(boolean enabled) {
		setAttr(Attrs.enabled, enabled);
	}

	/**
	 * Returns the filename, without extension, to use for the exported chart.
	 * <p>
	 * Defaults to chart.
	 */
	public String getFilename() {
		return getAttr(Attrs.filename, "chart").asString();
	}

	/**
	 * Sets the filename, without extension, to use for the exported chart.
	 * Defaults to chart.
	 */
	public void setFilename(String filename) {
		setAttr(Attrs.filename, filename, "chart");
	}

	/**
	 * Sets an object containing additional attributes for the POST form that
	 * sends the SVG to the export server. For example, a target can be set to
	 * make sure the generated image is received in another frame, or a custom
	 * enctype or encoding can be set.
	 */
	public <T> void setFormAttributes(List<T> formAttributes) {
		setAttr(Attrs.formAttributes, formAttributes);
	}

	/**
	 * Returns an object containing additional attributes for the POST form that
	 * sends the SVG to the export server. For example, a target can be set to
	 * make sure the generated image is received in another frame, or a custom
	 * enctype or encoding can be set.
	 * <p>
	 * Defaults to null.
	 */
	public <T> List<T> getFormAttributes() {
		return Generics.cast(getAttr(Attrs.formAttributes, null).asValue());
	}

	/**
	 * Returns the scale or zoom factor for the exported image compared to the
	 * on-screen display. While for instance a 600px wide chart may look good on
	 * a website, it will look bad in print. The default scale of 2 makes this
	 * chart export to a 1200px PNG or JPG..
	 * <p>
	 * Default: 2
	 */
	public Number getScale() {
		return getAttr(Attrs.scale, 2).asNumber();
	}

	/**
	 * Sets the scale or zoom factor for the exported image compared to the
	 * on-screen display. While for instance a 600px wide chart may look good on
	 * a website, it will look bad in print. The default scale of 2 makes this
	 * chart export to a 1200px PNG or JPG..
	 * <p>
	 * Default: 2
	 */
	public void setScale(Number scale) {
		setAttr(Attrs.scale, scale, 2);
	}

	/**
	 * Returns the height of the original chart when exported, unless an
	 * explicit chart.width is set. The width exported raster image is then
	 * multiplied by scale.
	 * <p>
	 * Default: null
	 */
	public Number getSourceHeight() {
		return getAttr(Attrs.sourceHeight, null).asNumber();
	}

	/**
	 * Sets the height of the original chart when exported, unless an explicit
	 * chart.width is set. The width exported raster image is then multiplied by
	 * scale.
	 * <p>
	 * Default: null
	 */
	public void setSourceHeight(Number sourceHeight) {
		setAttr(Attrs.sourceHeight, sourceHeight);
	}

	/**
	 * Returns the width of the original chart when exported, unless an explicit
	 * chart.width is set. The width exported raster image is then multiplied by
	 * scale.
	 * <p>
	 * Default: null
	 */
	public Number getSourceWidth() {
		return getAttr(Attrs.sourceWidth, null).asNumber();
	}

	/**
	 * Sets the width of the original chart when exported, unless an explicit
	 * chart.width is set. The width exported raster image is then multiplied by
	 * scale.
	 * <p>
	 * Default: null
	 */
	public void setSourceWidth(Number sourceWidth) {
		setAttr(Attrs.sourceWidth, sourceWidth);
	}

	/**
	 * Sets the default MIME type for exporting if chart.exportChart() is called
	 * without specifying a type option. Possible values are image/png,
	 * image/jpeg, application/pdf and image/svg+xml.
	 * <p>
	 * Defaults to image/png.
	 * 
	 * @param type
	 */
	public void setType(String type) {
		setAttr(Attrs.type, type, "image/png");
	}

	/**
	 * Returns the default MIME type for exporting if chart.exportChart() is
	 * called without specifying a type option. Possible values are image/png,
	 * image/jpeg, application/pdf and image/svg+xml.
	 * <p>
	 * Defaults to image/png.
	 */
	public String getType() {
		return getAttr(Attrs.type, "image/png").asString();
	}

	/**
	 * Sets the URL for the server module converting the SVG string to an image
	 * format. By default this points to Highslide Software's free web service.
	 * <p>
	 * Defaults to http://export.highcharts.com.
	 */
	public void setUrl(String url) {
		setAttr(Attrs.url, url, "http://export.highcharts.com");
	}

	/**
	 * Returns the URL for the server module converting the SVG string to an
	 * image format. By default this points to Highslide Software's free web
	 * service.
	 * <p>
	 * Defaults to http://export.highcharts.com.
	 */
	public String getUrl() {
		return getAttr(Attrs.url, "http://export.highcharts.com").asString();
	}

	/**
	 * Returns the pixel width of charts exported to PNG or JPG. As of
	 * Highcharts 3.0, the default pixel width is a function of the chart.width
	 * or exporting.sourceWidth and the exporting.scale.
	 * <p>Defaults to null.
	 */
	public Number getWidth() {
		return getAttr(Attrs.width, null).asNumber();
	}

	/**
	 * Sets the pixel width of charts exported to PNG or JPG. As of
	 * Highcharts 3.0, the default pixel width is a function of the chart.width
	 * or exporting.sourceWidth and the exporting.scale.
	 * <p>Defaults to null.
	 */
	public void setWidth(Number width) {
		setAttr(Attrs.width, width);
	}
}
