/* Drilldown.java

	Purpose:
		
	Description:
		
	History:
		Tue, Jan 28, 2014 12:21:12 PM, Created by RaymondChao

Copyright (C) 2014 Potix Corporation. All Rights Reserved.

 */
package org.zkoss.chart;

import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.zkoss.chart.util.DynamicalAttribute;
import org.zkoss.chart.util.MapsHelper;
import org.zkoss.lang.Generics;

/**
 * Options for drill down, the concept of inspecting increasingly high
 * resolution data through clicking on chart items like columns or pie slices.
 * 
 * <p> All the options in this class support {@link DynamicalAttribute}.
 * 
 * @author jumperchen
 * @author RaymondChao
 */
public class Drilldown extends Optionable {
	private enum Attrs implements PlotAttribute, DynamicalAttribute {
		activeAxisLabelStyle,
		activeDataLabelStyle,
		animation,
		drillUpButton,
		series
	}

	/**
	 * Returns additional styles to apply to the X axis label for a point that
	 * has drilldown data. By default it is underlined and blue to invite to
	 * interaction. Defaults to:
	 * 
	 * <pre>
	 * cursor: pointer; color: #0d233a; fontWeight: bold; textDecoration: underline;
	 * </pre>
	 */
	public <K, V> Map<K, V> getActiveAxisLabelStyle() {
		if (!containsKey(Attrs.activeAxisLabelStyle)) {
			setActiveAxisLabelStyle("cursor: pointer; color: #0d233a; fontWeight: bold; textDecoration: underline;");
		}
		return Generics.cast(getAttr(Attrs.activeAxisLabelStyle));
	}

	/**
	 * Sets additional styles to apply to the X axis label for a point that has
	 * drilldown data. By default it is underlined and blue to invite to
	 * interaction. Defaults to:
	 * 
	 * <pre>
	 * cursor: pointer; color: #0d233a; fontWeight: bold; textDecoration: underline;
	 * </pre>
	 * 
	 * @see #setActiveAxisLabelStyle(String)
	 */
	public <K, V> void setActiveAxisLabelStyle(Map<K, V> labelStyle) {
		setAttr(Attrs.activeAxisLabelStyle, labelStyle, NOT_NULL_VALUE);
	}

	/**
	 * Sets additional styles to apply to the X axis label for a point that has
	 * drilldown data. By default it is underlined and blue to invite to
	 * interaction. Defaults to:
	 * 
	 * <pre>
	 * cursor: pointer; color: #0d233a; fontWeight: bold; textDecoration: underline;
	 * </pre>
	 * 
	 * @see #setActiveAxisLabelStyle(Map)
	 */
	public void setActiveAxisLabelStyle(String activeAxisLabelStyle) {
		setActiveAxisLabelStyle(MapsHelper.parse(new LinkedHashMap<String, String>(), activeAxisLabelStyle, ':',
				';', '\''));
	}

	/**
	 * Returns additional styles to apply to the data label of a point that has
	 * drilldown data. By default it is underlined and blue to invite to
	 * interaction. Defaults to:
	 * 
	 * <pre>
	 * cursor: pointer; color: #0d233a; fontWeight: bold; textDecoration: underline;
	 * </pre>
	 */
	public <K, V> Map<K, V> getActiveDataLabelStyle() {
		if (!containsKey(Attrs.activeDataLabelStyle)) {
			setActiveDataLabelStyle("cursor: pointer; color: #0d233a; fontWeight: bold; textDecoration: underline;");
		}
		return Generics.cast(getAttr(Attrs.activeDataLabelStyle));
	}

	/**
	 * Sets additional styles to apply to the data label of a point that has
	 * drilldown data. By default it is underlined and blue to invite to
	 * interaction. Defaults to:
	 * 
	 * <pre>
	 * cursor: pointer; color: #0d233a; fontWeight: bold; textDecoration: underline;
	 * </pre>
	 * 
	 * @see #setActiveDataLabelStyle(Map)
	 */
	public void setActiveDataLabelStyle(String activeDataLabelStyle) {
		setActiveDataLabelStyle(MapsHelper.parse(new LinkedHashMap<String, String>(), activeDataLabelStyle, ':',
				';', '\''));
	}

	/**
	 * Sets additional styles to apply to the data label of a point that has
	 * drilldown data. By default it is underlined and blue to invite to
	 * interaction. Defaults to:
	 * 
	 * <pre>
	 * cursor: pointer; color: #0d233a; fontWeight: bold; textDecoration: underline;
	 * </pre>
	 * 
	 * @see #setActiveDataLabelStyle(String)
	 */
	public <K, V> void setActiveDataLabelStyle(Map<K, V> activeDataLabelStyle) {
		setAttr(Attrs.activeDataLabelStyle, activeDataLabelStyle, NOT_NULL_VALUE);
	}

	/**
	 * Returns the overall animation for all drilldown updating.
	 * <p>
	 * Default: an instance of {@link Animation}
	 */
	public Animation getAnimation() {
		return (Animation) getAttr(Attrs.animation,
				org.zkoss.chart.Animation.class).asValue();
	}

	/**
	 * Returns whether enable the animation
	 * <p>
	 * Default: true
	 */
	public boolean isAnimation() {
		return getAnimation() != Animation.NONE;
	}

	/**
	 * Sets the animation for all drilldown animations. Animation of a drilldown
	 * occurs when drilling between a column point and a column series, or a pie
	 * slice and a full pie series. Drilldown can still be used between series
	 * and points of different types, but animation will not occur.
	 * <p>
	 * The animation can either be set as a boolean or a configuration object.
	 * See {@link #setAnimation(Animation)}. If <code>true</code>, it will use
	 * the 'swing' jQuery easing and a duration of 500 ms. If used as an
	 * Animation object, the following properties are supported:
	 * </p>
	 * <dl>
	 * <dt>duration</dt>
	 * <dd>The duration of the animation in milliseconds.</dd>
	 * <dt>easing</dt>
	 * <dd>When using jQuery as the general framework, the easing can be set to
	 * <code>linear</code> or <code>swing</code>. More easing functions are
	 * available with the use of jQuery plug-ins, most notably the jQuery UI
	 * suite. See <a href="http://api.jquery.com/animate/">the jQuery docs</a>.
	 * When using MooTools as the general framework, use the property name
	 * <code>transition</code> instead of <code>easing</code>.</dd>
	 * </dl>
	 * <p>
	 * 
	 * @see #setAnimation(Animation)
	 * @see Animation
	 */
	public void setAnimation(boolean animation) {
		setAnimation(animation ? new Animation() : Animation.NONE);
	}

	/**
	 * Sets the animation for all drilldown animations. Animation of a drilldown
	 * occurs when drilling between a column point and a column series, or a pie
	 * slice and a full pie series. Drilldown can still be used between series
	 * and points of different types, but animation will not occur.
	 * <p>
	 * Properties are supported:
	 * <dl>
	 * <dt>duration</dt>
	 * <dd>The duration of the animation in milliseconds.</dd>
	 * <dt>easing</dt>
	 * <dd>When using jQuery as the general framework, the easing can be set to
	 * <code>linear</code> or <code>swing</code>. More easing functions are
	 * available with the use of jQuery plug-ins, most notably the jQuery UI
	 * suite. See <a href="http://api.jquery.com/animate/">the jQuery docs</a>.
	 * When using MooTools as the general framework, use the property name
	 * <code>transition</code> instead of <code>easing</code>.</dd>
	 * </dl>
	 * <p>
	 * 
	 * @see #setAnimation(boolean)
	 * @see Animation
	 */
	public void setAnimation(Animation animation) {
		setAttr(Attrs.animation, animation == null ? Animation.NONE : animation);
	}

	/**
	 * Returns options for the drill up button that appears when drilling down
	 * on a series. The text for the button is defined in
	 * {@link Lang#getDrillUpText()}.
	 * 
	 * @see DrillUpButton
	 */
	public DrillUpButton getDrillUpButton() {
		return (DrillUpButton) getAttr(Attrs.drillUpButton, null).asValue();
	}

	/**
	 * Sets options for the drill up button that appears when drilling down on a
	 * series. The text for the button is defined in
	 * {@link Lang#setDrillUpText(String)}.
	 * 
	 * @see DrillUpButton
	 */
	public void setDrillUpButton(DrillUpButton drillUpButton) {
		setAttr(Attrs.drillUpButton, drillUpButton);
	}

	/**
	 * Returns an array of series configurations for the drill down. Each series
	 * configuration uses the same syntax as {@link Series}. These drilldown
	 * series are hidden by default. The drilldown series is linked to the
	 * parent series' point by its {@link Series#setId(String)}.
	 */
	public List<Series> getSeries() {
		return Generics.cast(getAttr(Attrs.series, null).asValue());
	}

	/**
	 * Sets an array of series configurations for the drill down. Each series
	 * configuration uses the same syntax as {@link Series}. These drilldown
	 * series are hidden by default. The drilldown series is linked to the
	 * parent series' point by its {@link Series#setId(String)}.
	 */
	public void setSeries(List<Series> series) {
		setAttr(Attrs.series, series);
	}

	/**
	 * Sets an array of series configurations for the drill down. Each series
	 * configuration uses the same syntax as {@link Series}. These drilldown
	 * series are hidden by default. The drilldown series is linked to the
	 * parent series' point by its {@link Series#setId(String)}.
	 * @see #setSeries(List)
	 */
	public void setSeries(Series... series) {
		setSeries(Arrays.asList(series));
	}

}