/* DrillUpButton.java

	Purpose:
		
	Description:
		
	History:
		Tue, Jan 28, 2014 12:21:24 PM, Created by RaymondChao

Copyright (C) 2014 Potix Corporation. All Rights Reserved.

 */
package org.zkoss.chart;

import java.util.Map;

import org.zkoss.chart.util.DynamicalAttribute;
import org.zkoss.lang.Generics;

/**
 * Options for the drill up button that appears when drilling down on a series.
 * The text for the button is defined in {@link Lang#getDrillUpText()}.
 * 
 * @author jumperchen
 * @author RaymondChao
 */
public class DrillUpButton extends Optionable {
	private enum Attrs implements PlotAttribute, DynamicalAttribute {
		position, relativeTo, theme
	}

	/**
	 * Returns positioning options for the button within the
	 * <code>relativeTo</code> box. Available properties are <code>x</code>,
	 * <code>y</code>, <code>align</code> and <code>verticalAlign</code>.
	 * <p>
	 * Default: null
	 */
	public Position getPosition() {
		return (Position) getAttr(Attrs.position, null).asValue();
	}

	/**
	 * Sets positioning options for the button within the
	 * <code>relativeTo</code> box. Available properties are <code>x</code>,
	 * <code>y</code>, <code>align</code> and <code>verticalAlign</code>.
	 */
	public void setPosition(Position position) {
		setAttr(Attrs.position, position);
	}

	/**
	 * Returns what box to align the button to. Can be either "plotBox" or
	 * "spacingBox".
	 * <p>
	 * Default: <code>plotBox</code>
	 */
	public String getRelativeTo() {
		return getAttr(Attrs.relativeTo, "plotBox").asString();
	}

	/**
	 * Sets what box to align the button to. Can be either "plotBox" or
	 * "spacingBox".
	 * <p>
	 * Default: <code>plotBox</code>
	 */
	public void setRelativeTo(String relativeTo) {
		if (!("plotBox".equals(relativeTo) || "spacingBox".equals(relativeTo)))
			throw new IllegalArgumentException("Unsupported value: ["
					+ relativeTo + "]");
		setAttr(Attrs.relativeTo, relativeTo, "plotBox");
	}

	/**
	 * Returns a collection of attributes for the button. The object takes SVG
	 * attributes like <code>fill</code>, <code>stroke</code>,
	 * <code>stroke-width</code> or <code>r</code>, the border radius. The theme
	 * also supports <code>style</code>, a collection of CSS properties for the
	 * text. Equivalent attributes for the hover state are given in
	 * <code>theme.states.hover</code>.
	 */
	public <K, V> Map<K, V> getTheme() {
		return Generics.cast(getAttr(Attrs.theme, null).asValue());
	}

	/**
	 * Sets a collection of attributes for the button. The object takes SVG
	 * attributes like <code>fill</code>, <code>stroke</code>,
	 * <code>stroke-width</code> or <code>r</code>, the border radius. The theme
	 * also supports <code>style</code>, a collection of CSS properties for the
	 * text. Equivalent attributes for the hover state are given in
	 * <code>theme.states.hover</code>.
	 * 
	 * @param theme
	 *            a collection of attributes for the button. The object takes
	 *            SVG attributes like <code>fill</code>, <code>stroke</code>,
	 *            <code>stroke-width</code> or <code>r</code>, the border
	 *            radius. The theme also supports <code>style</code>, a
	 *            collection of CSS properties for the text. Equivalent
	 *            attributes for the hover state are given in
	 *            <code>theme.states.hover</code>.
	 */
	public <K, V> void setTheme(Map<K, V> theme) {
		setAttr(Attrs.theme, theme);
	}
}