/* Credits.java

	Purpose:
		
	Description:
		
	History:
		Fri, Jan 17, 2014 11:41:28 AM, Created by RaymondChao

Copyright (C) 2014 Potix Corporation. All Rights Reserved.

 */
package org.zkoss.chart;

import java.util.LinkedHashMap;
import java.util.Map;

import org.zkoss.chart.util.DynamicalAttribute;
import org.zkoss.chart.util.MapsHelper;
import org.zkoss.lang.Generics;

/**
 * A credits option to put a credits label in the lower right corner of the
 * chart. This can be changed using these options.
 * <p> All the options in this class support {@link DynamicalAttribute}.
 * 
 * @author jumperchen
 * @author RaymondChao
 */
public class Credits extends Optionable {
	private enum Attrs implements PlotAttribute, DynamicalAttribute {
		enabled, href, position, style, text
	}

	/**
	 * Returns whether to show the credits text.
	 * <p>
	 * Default: false
	 */
	public boolean isEnabled() {
		return getAttr(Attrs.enabled, false).asBoolean();
	}

	/**
	 * Sets whether to show the credits text.
	 */
	public void setEnabled(boolean enabled) {
		setAttr(Attrs.enabled, enabled);
	}

	/**
	 * Returns the URL for the credits label.
	 * <p>
	 * Default: "http://www.highcharts.com"
	 */
	public String getHref() {
		return getAttr(Attrs.href, "http://www.highcharts.com").asString();
	}

	/**
	 * Sets the URL for the credits label.
	 */
	public void setHref(String href) {
		setAttr(Attrs.href, href, "http://www.highcharts.com");
	}

	/**
	 * Returns position configuration for the credtis label. Supported
	 * properties are <code>align</code>, <code>verticalAlign</code>,
	 * <code>x</code> and <code>y</code>. Defaults to
	 * 
	 * <pre>
	 * position: {
	 * 	align: 'right',
	 * 	x: -10,
	 * 	verticalAlign: 'bottom',
	 * 	y: -5
	 * }
	 * </pre>
	 */
	public Position getPosition() {
		if (!containsKey(Attrs.position)) {
			Position position = new Position(-10, -5, "right", "bottom");
			setPosition(position);
			return position;
		} else {
			return (Position) getAttr(Attrs.position);
		}
	}

	/**
	 * Sets position configuration for the credtis label. Supported properties
	 * are <code>align</code>, <code>verticalAlign</code>, <code>x</code> and
	 * <code>y</code>. Defaults to
	 * 
	 * <pre>
	 * position: {
	 * 	align: 'right',
	 * 	x: -10,
	 * 	verticalAlign: 'bottom',
	 * 	y: -5
	 * }
	 * </pre>
	 */
	public void setPosition(Position position) {
		setAttr(Attrs.position, position, NOT_NULL_VALUE);
	}

	/**
	 * Returns the CSS styles for the credits label. Defaults to:
	 * 
	 * <pre>
	 * cursor: pointer; color: #909090; fontSize: 10px;
	 * </pre>
	 */
	public <K, V> Map<K, V> getStyle() {
		if (!containsKey(Attrs.style)) {
			setStyle("cursor: pointer; color: #909090; fontSize: 10px;");
		}
		return Generics.cast(getAttr(Attrs.style));
	}

	/**
	 * Sets the CSS styles for the credits label. Defaults to:
	 * 
	 * <pre>
	 * cursor: pointer; color: #909090; fontSize: 10px;
	 * </pre>
	 * 
	 * @see #setStyle(Map)
	 */
	public void setStyle(String style) {
		setStyle(MapsHelper.parse(new LinkedHashMap<String, String>(), style, ':', ';', '\''));
	}

	/**
	 * Sets the CSS styles for the credits label. Defaults to:
	 * 
	 * <pre>
	 * cursor: pointer; color: #909090; fontSize: 10px;
	 * </pre>
	 * 
	 * @see #setStyle(String)
	 */
	public <K, V> void setStyle(Map<K, V> style) {
		setAttr(Attrs.style, style, NOT_NULL_VALUE);
	}

	/**
	 * Returns the text for the credits label.
	 * <p>
	 * Default: "Highcharts.com"
	 */
	public String getText() {
		return getAttr(Attrs.text, "Highcharts.com").asString();
	}

	/**
	 * Sets the text for the credits label.
	 */
	public void setText(String text) {
		setAttr(Attrs.text, text, "Highcharts.com");
	}

}