/* Charts.java

	Purpose:
		
	Description:
		
	History:
		Wed, Jan 08, 2014  5:12:42 PM, Created by jumperchen

Copyright (C)  Potix Corporation. All Rights Reserved.

This program is distributed under LGPL Version 2.1 in the hope that
it will be useful, but WITHOUT ANY WARRANTY.
 */
package org.zkoss.chart;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;

import org.zkoss.chart.impl.AreaPlotImpl;
import org.zkoss.chart.impl.AreaRangePlotImpl;
import org.zkoss.chart.impl.AreaSplinePlotImpl;
import org.zkoss.chart.impl.AreaSplineRangePlotImpl;
import org.zkoss.chart.impl.BarPlotImpl;
import org.zkoss.chart.impl.BubblePlotImpl;
import org.zkoss.chart.impl.ColumnPlotImpl;
import org.zkoss.chart.impl.ColumnRangePlotImpl;
import org.zkoss.chart.impl.ErrorBarPlotImpl;
import org.zkoss.chart.impl.FunnelPlotImpl;
import org.zkoss.chart.impl.GaugePlotImpl;
import org.zkoss.chart.impl.LinePlotImpl;
import org.zkoss.chart.impl.PiePlotImpl;
import org.zkoss.chart.impl.PolarPlotImpl;
import org.zkoss.chart.impl.ScatterPlotImpl;
import org.zkoss.chart.impl.SplinePlotImpl;
import org.zkoss.chart.impl.WaterfallPlotImpl;
import org.zkoss.chart.model.CategoryModel;
import org.zkoss.chart.model.ChartsDataEvent;
import org.zkoss.chart.model.ChartsDataListener;
import org.zkoss.chart.model.ChartsModel;
import org.zkoss.chart.model.ChartsModelProxy;
import org.zkoss.chart.model.DialChartsDataEvent;
import org.zkoss.chart.model.DialChartsDataEventImpl;
import org.zkoss.chart.model.DialModel;
import org.zkoss.chart.model.SingleValueCategoryModel;
import org.zkoss.chart.model.XYModel;
import org.zkoss.chart.plotOptions.PlotOptions;
import org.zkoss.chart.plotOptions.SeriesPlotOptions;
import org.zkoss.chart.util.JSFunction;
import org.zkoss.json.JSONValue;
import org.zkoss.json.JavaScriptValue;
import org.zkoss.lang.Library;
import org.zkoss.lang.Objects;
import org.zkoss.zk.au.AuRequest;
import org.zkoss.zk.au.AuRequests;
import org.zkoss.zk.ui.AbstractComponent;
import org.zkoss.zk.ui.UiException;
import org.zkoss.zk.ui.WebApps;
import org.zkoss.zk.ui.WrongValueException;
import org.zkoss.zk.ui.event.Event;
import org.zkoss.zk.ui.event.EventListener;
import org.zkoss.zk.ui.event.Events;
import org.zkoss.zul.ChartModel;
import org.zkoss.zul.event.ChartDataEvent;
import org.zkoss.zul.impl.Utils;

/**
 * Charts is a ZK component that wraps the <a
 * href="http://www.highcharts.com/">Highcharts.com</a> charting library which
 * is written in pure HTML5/JavaScript, offering intuitive, interactive charts
 * to your web site or web application. Charts component currently supports
 * area, arearange, areaspline, areasplinerange, bar, boxplot, bubble, column,
 * columnrange, errorbar, funnel, gauge, line, pie, scatter, series, spline,
 * waterfall, and polar chart types.
 * <p>
 * Model supports:
 * <ul>
 * <li>CategoryModel: line, area, bar, column, areaspline, waterfall, spline</li>
 * <li>SingleValueCategoryModel: pie, funnel</li>
 * <li>DialModel: gauge</li>
 * <li>XYModel: line, area, bar, column, areaspline, scatter, bubble, waterfall,
 * polar, spline, columnrange, errorbar</li>
 * <li>XYZModel: bubble, arearange, areasplinerange</li>
 * </ul>
 * The model implementations use the same as Charts' API to manipulate the data
 * of the model, so developer can use the Charts' API instead of using model if
 * the chart data are not dynamically changed.
 * </p>
 * 
 * @author jumperchen
 * @author RaymondChao
 */
public class Charts extends AbstractComponent {

	private static final long serialVersionUID = 20140108153712L;
	// chart type
	/**
	 * Chart type: area
	 */
	public static final String AREA = "area";
	/**
	 * Chart type: area range
	 */
	public static final String AREA_RANGE = "arearange";
	/**
	 * Chart type: area spline
	 */
	public static final String AREA_SPLINE = "areaspline";
	/**
	 * Chart type: area spline range
	 */
	public static final String AREA_SPLINE_RANGE = "areasplinerange";
	/**
	 * Chart type: bar
	 */
	public static final String BAR = "bar";
	/**
	 * Chart type: boxplot
	 */
	public static final String BOX_PLOT = "boxplot";
	/**
	 * Chart type: bubble
	 */
	public static final String BUBBLE = "bubble";
	/**
	 * Chart type: column
	 */
	public static final String COLUMN = "column";
	/**
	 * Chart type: column range
	 */
	public static final String COLUMN_RANGE = "columnrange";
	/**
	 * Chart type: error bar
	 */
	public static final String ERROR_BAR = "errorbar";
	/**
	 * Chart type: funnel
	 */
	public static final String FUNNEL = "funnel";
	/**
	 * Chart type: gauge
	 */
	public static final String GAUGE = "gauge";
	/**
	 * Chart type: line
	 */
	public static final String LINE = "line";
	/**
	 * Chart type: pie
	 */
	public static final String PIE = "pie";
	/**
	 * Chart type: scatter
	 */
	public static final String SCATTER = "scatter";
	/**
	 * Chart type: series
	 */
	public static final String SERIES = "series";
	/**
	 * Chart type: spline
	 */
	public static final String SPLINE = "spline";
	/**
	 * Chart type: waterfall
	 */
	public static final String WATERFALL = "waterfall";
	/**
	 * Chart type: polar
	 */
	public static final String POLAR = "polar";

	/** The ZK CSS class. */
	protected String _zclass;

	static {
		// component events
		addClientEvent(Charts.class, Events.ON_CLICK, 0);
		addClientEvent(Charts.class, Events.ON_DOUBLE_CLICK, 0);
		addClientEvent(Charts.class, Events.ON_RIGHT_CLICK, 0);
		addClientEvent(Charts.class, Events.ON_MOUSE_OVER, 0); // not to use
																// CE_DUPLICATE_IGNORE
																// since
																// there is
																// an order
																// issue
		addClientEvent(Charts.class, Events.ON_MOUSE_OUT, 0);
		
		// chart events
		addClientEvent(Charts.class, Events.ON_SELECTION, 0);
		addClientEvent(Charts.class, ChartsEvents.ON_PLOT_CLICK, 0);
		addClientEvent(Charts.class, ChartsEvents.ON_PLOT_CHECKBOX_CLICK, 0);
		addClientEvent(Charts.class, ChartsEvents.ON_PLOT_HIDE, 0);
		addClientEvent(Charts.class, ChartsEvents.ON_PLOT_LEGEND_ITEM_CLICK, 0);
		addClientEvent(Charts.class, ChartsEvents.ON_PLOT_MOUSE_OUT, 0);
		addClientEvent(Charts.class, ChartsEvents.ON_PLOT_MOUSE_OVER, 0);
		addClientEvent(Charts.class, ChartsEvents.ON_PLOT_SHOW, 0);
		addClientEvent(Charts.class, ChartsEvents.ON_PLOT_DRILL_DOWN, 0);
		addClientEvent(Charts.class, ChartsEvents.ON_PLOT_DRILL_UP, 0);
		
		// Update point select information
		addClientEvent(Charts.class, ChartsEvents.ON_PLOT_SELECT, CE_IMPORTANT);
		addClientEvent(Charts.class, ChartsEvents.ON_PLOT_UNSELECT, CE_IMPORTANT);
	}

	private boolean _smartDrawChart; // whether post the smartDraw event
										// already?
	private ChartsDataListener _dataListener;

	// chart data model
	private ChartsModel _model; // chart data model

	private PlotData _plotData;
	private Options _options;
	private Theme _theme;
	
	/** A list of selected points. */
	private transient Set<Point> _selPoints;

	private static final String ZKCHARTS_LOCK = "org.zkoss.chart.rt.lock.key";
	static { Library.setProperty(ZKCHARTS_LOCK, "true"); }
	private static final Random _RGEN = new Random(new java.util.Date().getTime());
	/**package*/ boolean checkLock() {
		return _RGEN.nextInt() > 0 || "true".equals(Library.getProperty(ZKCHARTS_LOCK));
	}
	
	public Charts() {
		_selPoints = new HashSet<Point>();
	}

	/**
	 * Returns the plot data which is a JSON object to collect all chart's
	 * options for Charts JS.
	 */
	public PlotData getPlotData() {
		if (_plotData == null)
			_plotData = new PlotData(this);
		return _plotData;
	}

	/**
	 * Returns the drilldown object.
	 */
	public Drilldown getDrilldown() {
		return getPlotData().getDrilldown();
	}

	/**
	 * Sets the global options for all Charts in the same browser page.
	 */
	public void setOptions(Options options) {
		if (options == null)
			throw new WrongValueException("Global options cannot be null!");

		_options = options; // no need to check, force to update
		invalidate();
	}

	/**
	 * Sets the formats a JavaScript date timestamp (milliseconds since Jan 1st
	 * 1970) into a human readable date string. The format is a subset of the
	 * formats for PHP's <a
	 * href="http://php.net/manual/en/function.strftime.php">strftime</a>
	 * function.
	 * 
	 * @param format
	 *            A string containing some of the formats above.
	 * @param time
	 *            The JavaScript time to format.
	 * @param capitalize
	 *            Whether to capitalize words in the return string.
	 */
	public void setDateFormat(String format, Number time, boolean capitalize) {
		addBeforeInvokeScript("Highcharts.dateFormat('"
				+ JSONValue.toJSONString(format) + "', "
				+ JSONValue.toJSONString(time) + ", "
				+ JSONValue.toJSONString(capitalize) + ")");
		invalidate();
	}

	/**
	 * Formats a JavaScript number with grouped thousands, a fixed amount of
	 * decimals and an optional decimal point. It is a port of PHP's function
	 * with the same name. See PHP number_format for a full explanation of the
	 * parameters.
	 * 
	 * @param number
	 *            The raw number to format.
	 * @param decimals
	 *            The desired number of decimals.
	 * @param decimalPoint
	 *            The decimal point. Defaults to "." or to the string specified
	 *            globally in {@link Lang#setDecimalPoint(String)}.
	 * @param thousandsSep
	 *            The thousands separator. Defaults to "," or to the string
	 *            specified globally in {@link Lang#setThousandsSep(String)}
	 */
	public void setNumberFormat(Number number, Number decimals,
			String decimalPoint, String thousandsSep) {
		addBeforeInvokeScript(String.format(
				"Highcharts.numberFormat(%1$s,%2$s,%3$s,%4$s)",
				JSONValue.toJSONString(number),
				JSONValue.toJSONString(decimals),
				JSONValue.toJSONString(decimalPoint),
				JSONValue.toJSONString(thousandsSep)));

		invalidate();
	}

	private List<JavaScriptValue> _befInvoke = new LinkedList<JavaScriptValue>();

	private void addBeforeInvokeScript(String script) {
		JSFunction jsFunction = new JSFunction();
		jsFunction.evalJavascript(script);
		_befInvoke.add(new JavaScriptValue(jsFunction.toExecFunction()));
	}

	/**
	 * Sets the default colors for the chart's series. When all colors are used,
	 * new colors are pulled from the start again.
	 */
	public void setColors(List<Color> colors) {
		getPlotData().setColors(colors);
	}

	/**
	 * Sets the default colors for the chart's series. When all colors are used,
	 * new colors are pulled from the start again.
	 */
	public void setColors(String... source) {
		Color[] colors = new Color[source.length];
		int i = 0;
		for (String s : source)
			colors[i++] = new Color(s);
		setColors(Arrays.asList(colors));
	}

	/**
	 * Returns the default colors for the chart's series. When all colors are
	 * used, new colors are pulled from the start again.
	 */
	public List<Color> getColors() {
		return getPlotData().getColors();
	}

	/**
	 * Sets the theme options for all Charts in the same browser page.
	 */
	public void setTheme(Theme theme) {
		if (theme == null)
			throw new WrongValueException("Theme options cannot be null!");

		_theme = theme; // no need to check, force to update
		invalidate();
	}

	/**
	 * Returns the global theme for all Charts.
	 * <p>
	 * Default is null
	 */
	public Theme getTheme() {
		return _theme;
	}

	/**
	 * Returns the global options for all Charts.
	 * <p>
	 * Default is null.
	 */
	public Options getOptions() {
		return _options;
	}

	/**
	 * Returns the chart model associated with this chart, or null if this chart
	 * is not associated with any chart data model.
	 */
	public ChartsModel getModel() {
		return _model;
	}

	/**
	 * Sets the chart model associated with this chart. If a non-null model is
	 * assigned, no matter whether it is the same as the previous, it will
	 * always cause re-render.
	 * 
	 * @param model
	 *            the chart model to associate, or null to dissociate any
	 *            previous model.
	 * @exception UiException
	 *                if failed to initialize with the model
	 */
	public void setModel(ChartsModel model) {
		if (_model != model) {
			if (_model != null) {
				_model.removeChartsDataListener(_dataListener);
				for (int i = _plotData.getSeriesSize(); --i >= 0;) {
					_plotData.getSeries(i).remove();
				}
			}
			_model = model;
			initDataListener();
			smartDrawChart();
		}
	}

	public void setModel(ChartModel model) {
		if (model != null) {
			final String version = WebApps.getCurrent().getVersion().replace(".", "");
			int f = Integer.parseInt(String.valueOf(version.charAt(0)));
			if (f < 7) {
				throw new IllegalArgumentException(
						"The version of ChartModel is not supported, please use ZK 7.0.1+ version or new ChartsModel instead. ");
			} else if (f == 7) {
				int s = Integer.parseInt(String.valueOf(version.charAt(1)));
				int t = Integer.parseInt(String.valueOf(version.charAt(2)));
				if (s == 0 && t == 0)
					throw new IllegalArgumentException(
							"The version of ChartModel is not supported, please use ZK 7.0.1+ version or new ChartsModel instead. ");
			}
			setModel(ChartsModelProxy.getInstance(model));
		} else {
			setModel((ChartsModel) null);
		}
	}

	private void initDataListener() {
		if (_dataListener == null) {
			_dataListener = new MyChartsDataListener();
		}
		_model.addChartsDataListener(_dataListener);
	}

	private class MyChartsDataListener implements ChartsDataListener,
			Serializable {
		private static final long serialVersionUID = 20091008183622L;

		public void onChange(ChartsDataEvent event) {

			ChartsModel model = getModel();
			if (model instanceof SingleValueCategoryModel
					|| model instanceof XYModel
					|| model instanceof CategoryModel) {
				switch (event.getType()) {
				case ChartDataEvent.ADDED:
					int sIndex = event.getSeriesIndex();
					if (event.getCategory() instanceof Number) {
						if (event.getData() instanceof Number[]) {
							Number[] numbers = (Number[]) event.getData();
							if (numbers.length == 3)
								getSeries(sIndex).addPoint(numbers[0],
										numbers[1], numbers[2]);
							else if (numbers.length == 2)
								getSeries(sIndex).addPoint(numbers[0],
										numbers[1]);
						} else {
							getSeries(sIndex).addPoint(
									(Number) event.getCategory(),
									(Number) event.getData());
						}
					} else {
						getSeries(sIndex).addPoint(
								(String) event.getCategory(),
								(Number) event.getData());

						if (getXAxis().getCategories() != null) {
							List<Object> categories = getXAxis().getCategories();
							if (!categories.contains(event.getCategory())) {
								List<Object> newList = new ArrayList(categories);
								newList.add(event.getCategory());
								getXAxis().setCategories(categories);
							};
						}
					}
					if (event.getSeries() != null)
						getSeries(sIndex).setName(event.getSeries().toString());
					break;
				case ChartDataEvent.CHANGED:
					int cIndex = event.getCategoryIndex();
					if (event.getData() instanceof Number[]) {
						Number[] numbers = (Number[]) event.getData();
						getSeries(event.getSeriesIndex()).getPoint(cIndex)
								.update(numbers[0], numbers[1], numbers[2]);
					} else if (model instanceof XYModel) {
						getSeries(event.getSeriesIndex()).getPoint(cIndex)
								.update((Number) event.getCategory(),
										(Number) event.getData());
					} else {
						getSeries(event.getSeriesIndex()).getPoint(cIndex)
								.update((Number) event.getData());
					}
					break;
				case ChartDataEvent.REMOVED:
					int cIndex2 = event.getCategoryIndex();
					int sIndex2 = event.getSeriesIndex();
					if (sIndex2 < 0) {
						for (int i = getSeriesSize(); --i >= 0;)
							getSeries(i).remove();

						// reset all
						List<Object> categories = getXAxis().getCategories();
						if (categories != null) {
							getXAxis().setCategories(new ArrayList<Object>()); // force to update
						}
					} else {
						Series series2 = getSeries(sIndex2);
						// xyz model
						if (cIndex2 < 0) {
							series2.remove();
						} else {
							series2.getPoint(cIndex2).remove();
							if (series2.getData().isEmpty())
								series2.remove();
						}
					}
					break;
				}
			} else if (model instanceof DialModel) {
				if (event instanceof DialChartsDataEvent) {
					DialChartsDataEventImpl evt = (DialChartsDataEventImpl) event;
					String key = evt.getPropertyKey();
					if (DialChartsDataEvent.SCALE_VALUE.equals(key)) {
						getSeries().getPoint(0).update((Number) evt.getData());
					} else if (DialChartsDataEvent.SCALE_TEXT.equals(key)) {
						getYAxis().getTitle().setText((String) evt.getData());
					} else if (DialChartsDataEvent.NEEDLE_COLOR.equals(key)) {
						getPlotOptions().getGauge().getDial()
								.setBackgroundColor((String) evt.getData());
					} else {
						smartDrawChart();
						invalidate();
					}
				}
			} else {
				throw new UiException("Unsupported with the model type ["
						+ model + "]");
			}
		}
	}

	/**
	 * Shows the loading dim the chart'ts plot area with the given text after
	 * rendering time, if any.
	 */
	public void showLoading(String text) {
		JSFunction js = new JSFunction();
		js.callFunction("showLoading", text);
		smartUpdate("eval", new JavaScriptValue(js.toWrapFunction()), true);
	}

	/**
	 * Hides the loading dim the chart'ts plot area after rendering time.
	 */
	public void hideLoading() {
		JSFunction js = new JSFunction();
		js.callFunction("hideLoading");
		smartUpdate("eval", new JavaScriptValue(js.toWrapFunction()), true);
	}

	/**
	 * Clears away other elements in the page and prints the chart as it is
	 * displayed.
	 */
	public void print() {
		JSFunction js = new JSFunction();
		js.callFunction("print");
		smartUpdate("eval", new JavaScriptValue(js.toWrapFunction()), true);
	}

	/**
	 * Set the chart's type (Charts.PIE, Charts.BAR, Charts.LINE, etc.).
	 * <p>
	 * Default: line.
	 */
	public void setType(String type) {
		Chart chart = getChart();
		String oldType = chart.getType();
		if (oldType != type) {
			// Highcharts JS didn't have a type for Polar, so we need to do some
			// tricks.
			if (POLAR.equals(type))
				getChart().setPolar(true); // just in case

			chart.setType(type);
			_chartImpl = null; // reset if changed
			smartDrawChart();
		}
	}

	/**
	 * Get the chart's type.
	 * <p>
	 * Default: line.
	 */
	public String getType() {
		return getChart().getType();
	}

	/**
	 * Sets when using multiple axis, the ticks of two or more opposite axes
	 * will automatically be aligned by adding ticks to the axis or axes with
	 * the least ticks. This can be prevented by setting <code>alignTicks</code>
	 * to false. If the grid lines look messy, it's a good idea to hide them for
	 * the secondary axis by setting <code>gridLineWidth</code> to 0.
	 * 
	 * @param alignTicks
	 *            when using multiple axis, the ticks of two or more opposite
	 *            axes will automatically be aligned by adding ticks to the axis
	 *            or axes with the least ticks
	 * @see Chart#setAlignTicks(boolean)
	 */
	public void setAlignTicks(boolean alignTicks) {
		getChart().setAlignTicks(alignTicks);
	}

	/**
	 * Returns when using multiple axis, the ticks of two or more opposite axes
	 * will automatically be aligned by adding ticks to the axis or axes with
	 * the least ticks. This can be prevented by setting <code>alignTicks</code>
	 * to false. If the grid lines look messy, it's a good idea to hide them for
	 * the secondary axis by setting <code>gridLineWidth</code> to 0.
	 * <p>
	 * Default: true.
	 * 
	 * @see Chart#isAlignTicks()
	 */
	public boolean isAlignTicks() {
		return getChart().isAlignTicks();
	}

	/**
	 * Animation can be disabled throughout the chart by setting it to false
	 * here. It can be overridden for each individual API method as a function
	 * parameter. The only animation not affected by this option is the initial
	 * series animation, see {@link SeriesPlotOptions#setAnimation(boolean)}.
	 * 
	 * @see Chart#setAnimation(boolean)
	 * @see Animation
	 */
	public void setAnimation(boolean animation) {
		getChart().setAnimation(animation);
	}

	/**
	 * Returns whether enable the animation
	 * <p>
	 * Default: true
	 */
	public boolean isAnimation() {
		return getChart().isAnimation();
	}

	/**
	 * Sets to ignore the hidden series, if true, the axes will scale to the
	 * remaining visible series once one series is hidden. If false, hiding and
	 * showing a series will not affect the axes or the other series. For
	 * stacks, once one series within the stack is hidden, the rest of the stack
	 * will close in around it even if the axis is not affected.
	 * 
	 * @param ignoreHiddenSeries
	 *            if true, the axes will scale to the remaining visible series
	 *            once one series is hidden
	 */
	public void setIgnoreHiddenSeries(boolean ignoreHiddenSeries) {
		getChart().setIgnoreHiddenSeries(ignoreHiddenSeries);
	}

	/**
	 * Returns if true, the axes will scale to the remaining visible series once
	 * one series is hidden. If false, hiding and showing a series will not
	 * affect the axes or the other series. For stacks, once one series within
	 * the stack is hidden, the rest of the stack will close in around it even
	 * if the axis is not affected.
	 * <p>
	 * Default: true.
	 */
	public boolean isIgnoreHiddenSeries() {
		return getChart().isIgnoreHiddenSeries();
	}

	/**
	 * Sets the background color for the outer chart area.
	 * 
	 * @param backgroundColor
	 *            the background color for the outer chart area
	 */
	public void setBackgroundColor(String backgroundColor) {
		getChart().setBackgroundColor(backgroundColor);
	}

	/**
	 * Sets the background gradient for the outer chart area.
	 */
	public void setBackgroundColor(LinearGradient gradient) {
		getChart().setBackgroundColor(gradient);
	}

	/**
	 * Sets the background gradient for the outer chart area.
	 */
	public void setBackgroundColor(RadialGradient gradient) {
		getChart().setBackgroundColor(gradient);
	}

	/**
	 * Sets the background gradient for the outer chart area.
	 */
	public void setBackgroundColor(Color color) {
		getChart().setBackgroundColor(color);
	}

	/**
	 * Returns the background color for the outer chart area.
	 * <p>
	 * Default: "#FFFFFF".
	 * 
	 * @see Chart#getBackgroundColor()
	 */
	public Color getBackgroundColor() {
		return getChart().getBackgroundColor();
	}

	/**
	 * Sets the color of the outer chart border. The border is painted using
	 * vector graphic techniques to allow rounded corners.
	 */
	public void setBorderColor(String borderColor) {
		getChart().setBorderColor(borderColor);
	}

	/**
	 * Sets the color of the outer chart border. The border is painted using
	 * vector graphic techniques to allow rounded corners.
	 */
	public void setBorderColor(LinearGradient borderColor) {
		getChart().setBorderColor(borderColor);
	}

	/**
	 * Sets the color of the outer chart border. The border is painted using
	 * vector graphic techniques to allow rounded corners.
	 */
	public void setBorderColor(RadialGradient borderColor) {
		getChart().setBorderColor(borderColor);
	}

	/**
	 * Sets the color of the outer chart border. The border is painted using
	 * vector graphic techniques to allow rounded corners.
	 */
	public void setBorderColor(Color borderColor) {
		getChart().setBorderColor(borderColor);
	}

	/**
	 * Returns the color of the outer chart border. The border is painted using
	 * vector graphic techniques to allow rounded corners.
	 * <p>
	 * Default: "#4572A7".
	 * 
	 * @see Chart#getBorderColor()
	 */
	public Color getBorderColor() {
		return getChart().getBorderColor();
	}

	/**
	 * Sets the corner radius of the outer chart border. In export, the radius
	 * defaults to 0.
	 * <p>
	 * Default: 5.
	 * 
	 * @param borderRadius
	 *            the corner radius of the outer chart border
	 * @see Chart#setBorderRadius(Number)
	 */
	public void setBorderRadius(Number borderRadius) {
		getChart().setBorderRadius(borderRadius);
	}

	/**
	 * Returns the corner radius of the outer chart border. In export, the
	 * radius defaults to 0.
	 * <p>
	 * Default: 5.
	 * 
	 * @see Chart#getBorderRadius()
	 */
	public Number getBorderRadius() {
		return getChart().getBorderRadius();
	}

	/**
	 * Sets the pixel width of the outer chart border. The border is painted
	 * using vector graphic techniques to allow rounded corners.
	 * 
	 * @see Chart#setBorderWidth(Number)
	 */
	public void setBorderWidth(Number borderWidth) {
		getChart().setBorderWidth(borderWidth);
	}

	/**
	 * Returns the pixel width of the outer chart border. The border is painted
	 * using vector graphic techniques to allow rounded corners.
	 * <p>
	 * Default: 0.
	 * 
	 * @see Chart#getBorderWidth()
	 */
	public Number getBorderWidth() {
		return getChart().getBorderWidth();
	}

	/**
	 * Sets an explicit width for the chart. By default the width is calculated
	 * from the offset width of the containing element.
	 * 
	 * @param width
	 *            an explicit width for the chart
	 * @see Chart#setWidth(Number)
	 */
	public void setWidth(Number width) {
		getChart().setWidth(width);
	}

	/**
	 * Returns an explicit width for the chart. By default the width is
	 * calculated from the offset width of the containing element.
	 * <p>
	 * Default: null.
	 * 
	 * @see Chart#getWidth()
	 */
	public Number getWidth() {
		return getChart().getWidth();
	}

	/**
	 * Sets an explicit height for the chart. By default the height is
	 * calculated from the offset height of the containing element, or 400
	 * pixels if the containing element's height is 0.
	 * 
	 * @param height
	 *            an explicit height for the chart
	 * @see Chart#setHeight(Number)
	 */
	public void setHeight(Number height) {
		getChart().setHeight(height);
	}

	/**
	 * Returns an explicit height for the chart. By default the height is
	 * calculated from the offset height of the containing element, or 400
	 * pixels if the containing element's height is 0.
	 * <p>
	 * Default: null.
	 * 
	 * @see Chart#getHeight()
	 */
	public Number getHeight() {
		return getChart().getHeight();
	}

	/**
	 * Sets whether to invert the axes so that the x axis is vertical and y axis
	 * is horizontal. When true, the x axis is reversed by default. If a bar
	 * series is present in the chart, it will be inverted automatically.
	 * 
	 * @see Chart#setInverted(boolean)
	 */
	public void setInverted(boolean inverted) {
		getChart().setInverted(inverted);
	}

	/**
	 * Returns whether to invert the axes so that the x axis is vertical and y
	 * axis is horizontal. When true, the x axis is reversed by default. If a
	 * bar series is present in the chart, it will be inverted automatically.
	 * <p>
	 * Default: false.
	 * 
	 * @see Chart#isInverted()
	 */
	public boolean isInverted() {
		return getChart().isInverted();
	}

	/**
	 * Sets the margin between the outer edge of the chart and the plot area.
	 * The numbers in the array designate top, right, bottom and left
	 * respectively. Use the methods of {@link #setMarginTop(Number)},
	 * {@link #setMarginRight(Number)}, {@link #setMarginBottom(Number)} and
	 * {@link #setMarginLeft(Number)} for shorthand setting of one option.
	 * <p>
	 * Default: null.
	 * 
	 * @see #setMargin(Number[])
	 */
	public void setMargin(String margins) {
		final int[] margin = Utils.stringToInts(margins, 0);
		Number[] result = new Number[margin.length];
		int i = 0;
		for (int m : margin) {
			result[i++] = m;
		}
		setMargin(result);
	}

	/**
	 * Sets the margin between the outer edge of the chart and the plot area.
	 * The numbers in the array designate top, right, bottom and left
	 * respectively. Use the methods of {@link #setMarginTop(Number)},
	 * {@link #setMarginRight(Number)}, {@link #setMarginBottom(Number)} and
	 * {@link #setMarginLeft(Number)} for shorthand setting of one option.
	 * <p>
	 * Default: null.
	 * 
	 * @see #setMargin(String)
	 * @see Chart#setMargin(Number[])
	 */
	public void setMargin(Number[] margin) {
		getChart().setMargin(margin);
	}

	/**
	 * Returns the margin between the outer edge of the chart and the plot area.
	 * The numbers in the array designate top, right, bottom and left
	 * respectively. Use the methods of {@link #setMarginTop(Number)},
	 * {@link #setMarginRight(Number)}, {@link #setMarginBottom(Number)} and
	 * {@link #setMarginLeft(Number)} for shorthand setting of one option.
	 * <p>
	 * Default: null.
	 * 
	 * @see Chart#getMargin()
	 */
	public Number[] getMargin() {
		return getChart().getMargin();
	}

	/**
	 * Sets the margin between the bottom outer edge of the chart and the plot
	 * area. Use this to set a fixed pixel value for the margin as opposed to
	 * the default dynamic margin. See also {@link #setSpacingBottom(Number)}.
	 * 
	 * @param marginBottom
	 *            the margin between the bottom outer edge of the chart and the
	 *            plot area
	 * @see Chart#setMarginBottom(Number)
	 */
	public void setMarginBottom(Number marginBottom) {
		getChart().setMarginBottom(marginBottom);
	}

	/**
	 * Returns the margin between the bottom outer edge of the chart and the
	 * plot area.
	 * <p>
	 * Default: null.
	 * 
	 * @see Chart#getMarginBottom()
	 */
	public Number getMarginBottom() {
		return getChart().getMarginBottom();
	}

	/**
	 * Sets the margin between the left outer edge of the chart and the plot
	 * area. Use this to set a fixed pixel value for the margin as opposed to
	 * the default dynamic margin. See also {@link #setSpacingLeft(Number)}
	 * 
	 * @param marginLeft
	 *            the margin between the left outer edge of the chart and the
	 *            plot area
	 * @see Chart#setMarginLeft(Number)
	 */
	public void setMarginLeft(Number marginLeft) {
		getChart().setMarginLeft(marginLeft);
	}

	/**
	 * Returns the margin between the left outer edge of the chart and the plot
	 * area.
	 * <p>
	 * Default: null.
	 * 
	 * @see Chart#getMarginLeft()
	 */
	public Number getMarginLeft() {
		return getChart().getMarginLeft();
	}

	/**
	 * Sets the margin between the right outer edge of the chart and the plot
	 * area. Use this to set a fixed pixel value for the margin as opposed to
	 * the default dynamic margin. See also {@link #setSpacingRight(Number)}.
	 * 
	 * @param marginRight
	 *            the margin between the right outer edge of the chart and the
	 *            plot area
	 * @see Chart#setMarginRight(Number)
	 */
	public void setMarginRight(Number marginRight) {
		getChart().setMarginRight(marginRight);
	}

	/**
	 * Returns the margin between the right outer edge of the chart and the plot
	 * area.
	 * <p>
	 * Default: null.
	 * 
	 * @see Chart#getMarginRight()
	 */
	public Number getMarginRight() {
		return getChart().getMarginRight();
	}

	/**
	 * Sets the margin between the top outer edge of the chart and the plot
	 * area. Use this to set a fixed pixel value for the margin as opposed to
	 * the default dynamic margin. See also {@link #setSpacingTop(Number)}.
	 * 
	 * @param marginTop
	 *            the margin between the top outer edge of the chart and the
	 *            plot area
	 * @see Chart#setMarginTop(Number)
	 */
	public void setMarginTop(Number marginTop) {
		getChart().setMarginTop(marginTop);
	}

	/**
	 * Returns the margin between the top outer edge of the chart and the plot
	 * area.
	 * <p>
	 * Default: null.
	 * 
	 * @see Chart#getMarginTop()
	 */
	public Number getMarginTop() {
		return getChart().getMarginTop();
	}

	/**
	 * Sets equivalent to {@link #setZoomType(String)}, but for multitouch
	 * gestures only. By default, the <code>pinchType</code> is the same as the
	 * <code>zoomType</code> setting. However, pinching can be enabled
	 * separately in some cases, for example in stock charts where a mouse drag
	 * pans the chart, while pinching is enabled.
	 * <p>
	 * Default: null.
	 * 
	 * @see #setZoomType(String)
	 * @see Chart#setPinchType(String)
	 */
	public void setPinchType(String pinchType) {
		getChart().setPinchType(pinchType);
	}

	/**
	 * Returns equivalent to {@link #getZoomType()}, but for multitouch gestures
	 * only. By default, the <code>pinchType</code> is the same as the
	 * <code>zoomType</code> setting. However, pinching can be enabled
	 * separately in some cases, for example in stock charts where a mouse drag
	 * pans the chart, while pinching is enabled.
	 * <p>
	 * Default: null.
	 * 
	 * @see Chart#getPinchType()
	 */
	public String getPinchType() {
		return getChart().getPinchType();
	}

	/**
	 * Sets the background color for the plot area.
	 */
	public void setPlotBackgroundColor(String plotBackgroundColor) {
		getChart().setPlotBackgroundColor(plotBackgroundColor);
	}

	/**
	 * Sets the background color for the plot area.
	 */
	public void setPlotBackgroundColor(LinearGradient plotBackgroundColor) {
		getChart().setPlotBackgroundColor(plotBackgroundColor);
	}

	/**
	 * Sets the background color for the plot area.
	 */
	public void setPlotBackgroundColor(RadialGradient plotBackgroundColor) {
		getChart().setPlotBackgroundColor(plotBackgroundColor);
	}

	/**
	 * Sets the background color for the plot area.
	 */
	public void setPlotBackgroundColor(Color plotBackgroundColor) {
		getChart().setPlotBackgroundColor(plotBackgroundColor);
	}

	/**
	 * Returns the background color for the plot area.
	 * <p>
	 * Default: null.
	 * 
	 * @see Chart#getPlotBackgroundColor()
	 */
	public Color getPlotBackgroundColor() {
		return getChart().getPlotBackgroundColor();
	}

	/**
	 * Sets the URL for an image to use as the plot background. To set an image
	 * as the background for the entire chart, set a CSS background image to the
	 * container element. Note that for the image to be applied to exported
	 * charts, its URL needs to be accessible by the export server.
	 * 
	 * @param plotBackgroundImage
	 *            the URL for an image to use as the plot background
	 * @see Chart#setPlotBackgroundImage(String)
	 */
	public void setPlotBackgroundImage(String plotBackgroundImage) {
		getChart().setPlotBackgroundImage(plotBackgroundImage);
	}

	/**
	 * Returns the URL for an image to use as the plot background.
	 * <p>
	 * Default: null.
	 * 
	 * @see Chart#getPlotBackgroundImage()
	 */
	public String getPlotBackgroundImage() {
		return getChart().getPlotBackgroundImage();
	}

	/**
	 * Sets the color of the inner chart or plot area border.
	 */
	public void setPlotBorderColor(String plotBorderColor) {
		getChart().setPlotBorderColor(plotBorderColor);
	}

	/**
	 * Sets the color of the inner chart or plot area border.
	 */
	public void setPlotBorderColor(LinearGradient plotBorderColor) {
		getChart().setPlotBorderColor(plotBorderColor);
	}

	/**
	 * Sets the color of the inner chart or plot area border.
	 */
	public void setPlotBorderColor(RadialGradient plotBorderColor) {
		getChart().setPlotBorderColor(plotBorderColor);
	}

	/**
	 * Sets the color of the inner chart or plot area border.
	 */
	public void setPlotBorderColor(Color plotBorderColor) {
		getChart().setPlotBorderColor(plotBorderColor);
	}

	/**
	 * Returns the color of the inner chart or plot area border.
	 * <p>
	 * Default: "#C0C0C0".
	 * 
	 * @see Chart#getPlotBorderColor()
	 */
	public Color getPlotBorderColor() {
		return getChart().getPlotBorderColor();
	}

	/**
	 * Sets the pixel width of the plot area border.
	 * 
	 * @param plotBorderWidth
	 *            the pixel width of the plot area border
	 * @see Chart#setPlotBorderWidth(Number)
	 */
	public void setPlotBorderWidth(Number plotBorderWidth) {
		getChart().setPlotBorderWidth(plotBorderWidth);
	}

	/**
	 * Returns the pixel width of the plot area border.
	 * <p>
	 * Default: 0.
	 * 
	 * @see Chart#getPlotBorderWidth()
	 */
	public Number getPlotBorderWidth() {
		return getChart().getPlotBorderWidth();
	}

	/**
	 * Sets whether to apply a drop shadow to the plot area. Requires that
	 * {@link #setPlotBackgroundColor(String)} be set.
	 * 
	 * @param plotShadow
	 *            whether to apply a drop shadow to the plot area
	 * @see Chart#setPlotShadow(boolean)
	 */
	public void setPlotShadow(boolean plotShadow) {
		getChart().setPlotShadow(plotShadow);
	}

	/**
	 * Returns whether to apply a drop shadow to the plot area. Requires that
	 * {@link #setPlotBackgroundColor(String)} be set.
	 * <p>
	 * Default: false.
	 * 
	 * @see Chart#isPlotShadow()
	 */
	public boolean isPlotShadow() {
		return getChart().isPlotShadow();
	}

	/**
	 * Sets to true to enable the cartesian charts like line, spline, area and
	 * column are transformed into the polar coordinate system.
	 * 
	 * @param polar
	 *            when true, cartesian charts like line, spline, area and column
	 *            are transformed into the polar coordinate system
	 * @see Chart#setPolar(boolean)
	 */
	public void setPolar(boolean polar) {
		getChart().setPolar(polar);
	}

	/**
	 * Returns whether transforms into the polar coordinate system
	 * <p>
	 * Default: false.
	 * 
	 * @see Chart#isPolar()
	 */
	public boolean isPolar() {
		return getChart().isPolar();
	}

	/**
	 * Sets whether to reflow the chart to fit the width of the container div on
	 * resizing the window.
	 * 
	 * @param reflow
	 *            whether to reflow the chart to fit the width of the container
	 *            div on resizing the window
	 * @see Chart#setReflow(boolean)
	 */
	public void setReflow(boolean reflow) {
		getChart().setReflow(reflow);
	}

	/**
	 * Returns whether to reflow the chart to fit the width of the container div
	 * on resizing the window.
	 * <p>
	 * Default: true.
	 * 
	 * @see Chart#isReflow()
	 */
	public boolean isReflow() {
		return getChart().isReflow();
	}

	/**
	 * Sets the background color of the marker square when selecting (zooming in
	 * on) an area of the chart.
	 */
	public void setSelectionMarkerFill(String selectionMarkerFill) {
		getChart().setSelectionMarkerFill(selectionMarkerFill);
	}

	/**
	 * Sets the background color of the marker square when selecting (zooming in
	 * on) an area of the chart.
	 */
	public void setSelectionMarkerFill(LinearGradient selectionMarkerFill) {
		getChart().setSelectionMarkerFill(selectionMarkerFill);
	}

	/**
	 * Sets the background color of the marker square when selecting (zooming in
	 * on) an area of the chart.
	 */
	public void setSelectionMarkerFill(RadialGradient selectionMarkerFill) {
		getChart().setSelectionMarkerFill(selectionMarkerFill);
	}

	/**
	 * Sets the background color of the marker square when selecting (zooming in
	 * on) an area of the chart.
	 */
	public void setSelectionMarkerFill(Color selectionMarkerFill) {
		getChart().setSelectionMarkerFill(selectionMarkerFill);
	}

	/**
	 * Returns the background color of the marker square when selecting (zooming
	 * in on) an area of the chart.
	 * <p>
	 * Default: "rgba(69,114,167,0.25)".
	 * 
	 * @see Chart#getSelectionMarkerFill()
	 */
	public Color getSelectionMarkerFill() {
		return getChart().getSelectionMarkerFill();
	}

	/**
	 * Sets whether to apply a drop shadow to the outer chart area. Requires
	 * that backgroundColor be set.
	 * 
	 * @param shadow
	 *            whether to apply a drop shadow to the outer chart area
	 * @see Chart#setShadow(boolean)
	 */
	public void setShadow(boolean shadow) {
		getChart().setShadow(shadow);
	}

	/**
	 * Returns whether to apply a drop shadow to the outer chart area. Requires
	 * that backgroundColor be set.
	 * <p>
	 * Default: false.
	 * 
	 * @see Chart#isShadow()
	 */
	public boolean isShadow() {
		return getChart().isShadow();
	}

	/**
	 * Sets whether to show the axes initially. This only applies to empty
	 * charts where series are added dynamically, as axes are automatically
	 * added to cartesian series.
	 * 
	 * @param showAxes
	 *            whether to show the axes initially
	 * @see Chart#setShowAxes(boolean)
	 */
	public void setShowAxes(boolean showAxes) {
		getChart().setShowAxes(showAxes);
	}

	/**
	 * Returns whether to show the axes initially. This only applies to empty
	 * charts where series are added dynamically, as axes are automatically
	 * added to cartesian series.
	 * <p>
	 * Default: false.
	 * 
	 * @see Chart#isShowAxes()
	 */
	public boolean isShowAxes() {
		return getChart().isShowAxes();
	}

	/**
	 * Sets the distance between the outer edge of the chart and the content,
	 * like title, legend, axis title or labels. The numbers in the array
	 * designate top, right, bottom and left respectively. Use the methods of
	 * {@link #setSpacingTop(Number)}, {@link #setSpacingRight(Number)},
	 * {@link #setSpacingBottom(Number)} and {@link #setSpacingLeft(Number)} for
	 * shorthand setting of one option.
	 * 
	 * @param spacings
	 *            the distance between the outer edge of the chart and the
	 *            content, like title, legend, axis title or labels
	 * @see #setSpacing(Number[])
	 */
	public void setSpacing(String spacings) {
		final int[] spacing = Utils.stringToInts(spacings, 0);
		int i = 0;
		Number[] result = new Number[spacing.length];
		for (int s : spacing) {
			result[i++] = s;
		}
		setSpacing(result);
	}

	/**
	 * Sets the distance between the outer edge of the chart and the content,
	 * like title, legend, axis title or labels. The numbers in the array
	 * designate top, right, bottom and left respectively. Use the methods of
	 * {@link #setSpacingTop(Number)}, {@link #setSpacingRight(Number)},
	 * {@link #setSpacingBottom(Number)} and {@link #setSpacingLeft(Number)} for
	 * shorthand setting of one option.
	 * 
	 * @param spacing
	 *            the distance between the outer edge of the chart and the
	 *            content, like title, legend, axis title or labels
	 * @see Chart#setSpacing(Number[])
	 */
	public void setSpacing(Number[] spacing) {
		getChart().setSpacing(spacing);
	}

	/**
	 * Returns the distance between the outer edge of the chart and the content,
	 * like title, legend, axis title or labels. The numbers in the array
	 * designate top, right, bottom and left respectively.
	 * <p>
	 * Default: [10, 10, 15, 10].
	 */
	public Number[] getSpacing() {
		return getChart().getSpacing();
	}

	/**
	 * Sets the space between the bottom edge of the chart and the content (plot
	 * area, axis title and labels, title, subtitle or legend in top position).
	 * </p>
	 * 
	 * @param spacingBottom
	 *            The space between the bottom edge of the chart and the content
	 *            (plot area, axis title and labels, title, subtitle or legend
	 *            in top position)
	 * @see Chart#setSpacingBottom(Number)
	 */
	public void setSpacingBottom(Number spacingBottom) {
		getChart().setSpacingBottom(spacingBottom);
	}

	/**
	 * Returns the space between the bottom edge of the chart and the content
	 * (plot area, axis title and labels, title, subtitle or legend in top
	 * position).
	 * <p>
	 * Default: 15.
	 * 
	 * @see Chart#getSpacingBottom()
	 */
	public Number getSpacingBottom() {
		return getChart().getSpacingBottom();
	}

	/**
	 * Sets the space between the left edge of the chart and the content (plot
	 * area, axis title and labels, title, subtitle or legend in top position).
	 * </p>
	 * 
	 * @param spacingLeft
	 *            The space between the left edge of the chart and the content
	 *            (plot area, axis title and labels, title, subtitle or legend
	 *            in top position)
	 * @see Chart#setSpacingLeft(Number)
	 */
	public void setSpacingLeft(Number spacingLeft) {
		getChart().setSpacingLeft(spacingLeft);
	}

	/**
	 * Returns the space between the left edge of the chart and the content
	 * (plot area, axis title and labels, title, subtitle or legend in top
	 * position).
	 * <p>
	 * Default: 10.
	 * 
	 * @see Chart#getSpacingLeft()
	 */
	public Number getSpacingLeft() {
		return getChart().getSpacingLeft();
	}

	/**
	 * Sets the space between the right edge of the chart and the content (plot
	 * area, axis title and labels, title, subtitle or legend in top position).
	 * 
	 * @param spacingRight
	 *            The space between the right edge of the chart and the content
	 *            (plot area, axis title and labels, title, subtitle or legend
	 *            in top position)
	 * @see Chart#setSpacingRight(Number)
	 */
	public void setSpacingRight(Number spacingRight) {
		getChart().setSpacingRight(spacingRight);
	}

	/**
	 * Returns the space between the right edge of the chart and the content
	 * (plot area, axis title and labels, title, subtitle or legend in top
	 * position).
	 * <p>
	 * Default: 10.
	 * 
	 * @see Chart#getSpacingRight()
	 */
	public Number getSpacingRight() {
		return getChart().getSpacingRight();
	}

	/**
	 * Sets the space between the top edge of the chart and the content (plot
	 * area, axis title and labels, title, subtitle or legend in top position).
	 * 
	 * @param spacingTop
	 *            The space between the top edge of the chart and the content
	 *            (plot area, axis title and labels, title, subtitle or legend
	 *            in top position)
	 * @see Chart#setSpacingTop(Number)
	 */
	public void setSpacingTop(Number spacingTop) {
		getChart().setSpacingTop(spacingTop);
	}

	/**
	 * Returns the space between the top edge of the chart and the content (plot
	 * area, axis title and labels, title, subtitle or legend in top position).
	 * <p>
	 * Default: 10.
	 * 
	 * @see Chart#getSpacingTop()
	 */
	public Number getSpacingTop() {
		return getChart().getSpacingTop();
	}

	/**
	 * Sets additional CSS styles to apply inline to the container
	 * <code>div</code>. Note that since the default font styles are applied in
	 * the renderer, it is ignorant of the individual chart options and must be
	 * set globally. Defaults to:
	 * 
	 * <pre>
	 * fontFamily: "Lucida Grande", "Lucida Sans Unicode", Verdana, Arial, Helvetica, sans-serif; fontSize: 12px
	 * </pre>
	 * 
	 * @param style
	 *            additional CSS styles to apply inline to the container
	 *            <code>div</code>
	 * @see Chart#setStyle(String)
	 */
	public void setStyle(String style) {
		getChart().setStyle(style);
	}

	/**
	 * Sets additional CSS styles to apply inline to the container
	 * <code>div</code>. Note that since the default font styles are applied in
	 * the renderer, it is ignorant of the individual chart options and must be
	 * set globally. Defaults to:
	 * 
	 * <pre>
	 * fontFamily: "Lucida Grande", "Lucida Sans Unicode", Verdana, Arial, Helvetica, sans-serif; fontSize: 12px
	 * </pre>
	 * 
	 * @param style
	 *            additional CSS styles to apply inline to the container
	 *            <code>div</code>
	 * @see Chart#setStyle(Map)
	 */
	public <K, V> void setStyle(Map<K, V> style) {
		getChart().setStyle(style);
	}

	/**
	 * Returns additional CSS styles to apply inline to the container
	 * <code>div</code>. Note that since the default font styles are applied in
	 * the renderer, it is ignorant of the individual chart options and must be
	 * set globally. Defaults to:
	 * 
	 * <pre>
	 * fontFamily: "Lucida Grande", "Lucida Sans Unicode", Verdana, Arial, Helvetica, sans-serif; fontSize: 12px
	 * </pre>
	 * <p>
	 * Default: null.
	 */
	public <K, V> Map<K, V> getStyle() {
		return getChart().getStyle();
	}

	/**
	 * Sets decides in what dimentions the user can zoom by dragging the mouse.
	 * Can be one of <code>x</code>, <code>y</code> or <code>xy</code>.
	 * 
	 * @param zoomType
	 *            decides in what dimentions the user can zoom by dragging the
	 *            mouse
	 * @see Chart#setZoomType(String)
	 */
	public void setZoomType(String zoomType) {
		getChart().setZoomType(zoomType);
	}

	/**
	 * Returns decides in what dimentions the user can zoom by dragging the
	 * mouse. Can be one of <code>x</code>, <code>y</code> or <code>xy</code>.
	 * <p>
	 * Default: null.
	 * 
	 * @see Chart#getZoomType()
	 */
	public String getZoomType() {
		return getChart().getZoomType();
	}

	/**
	 * mark a draw flag to inform that this Chart needs update.
	 */
	protected void smartDrawChart() {
		if (_smartDrawChart || _model == null) { // already mark smart draw
			return;
		}
		_smartDrawChart = true;
		
		// do it directly
		try {
			doSmartDraw();
		} finally {
			_smartDrawChart = false;
		}
	}

	/**
	 * Returns the Tooltip options, if any. Otherwise, create a new one.
	 * 
	 * @see Tooltip
	 */
	public Tooltip getTooltip() {
		return getPlotData().getTooltip();
	}

	/**
	 * Sets the Tooltip options
	 * 
	 * @param tooltip
	 */
	public void setTooltip(Tooltip tooltip) {
		getPlotData().setTooltip(tooltip);
	}

	/**
	 * Returns the Chart options, if any. Otherwise, create a new one.
	 * 
	 * @see Chart
	 */
	public Chart getChart() {
		return getPlotData().getChart();
	}

	/**
	 * Sets the no-data options
	 * @since 1.0.1
	 */
	public void setNoData(NoData noData) {
		getPlotData().setNoData(noData);
	}

	/**
	 * Returns the no-data options
	 * @since 1.0.1
	 */
	public NoData getNoData() {
		return getPlotData().getNoData();
	}
	/**
	 * Returns the Exporting options, if any. Otherwise, create a new one.
	 * 
	 * @see Exporting
	 */
	public Exporting getExporting() {
		return getPlotData().getExporting();
	}

	/**
	 * Sets the Exporting options.
	 * 
	 * @see Exporting
	 */
	public void setExporting(Exporting exporting) {
		getPlotData().setExporting(exporting);
	}

	/**
	 * Sets the Credits options
	 * 
	 * @param credits
	 */
	public void setCredits(Credits credits) {
		getPlotData().setCredits(credits);
	}

	/**
	 * Returns the Credits options, if any. Otherwise, create a new one.
	 * 
	 * @see Credits
	 */
	public Credits getCredits() {
		return getPlotData().getCredits();
	}

	/**
	 * Sets the Chart options
	 * 
	 * @param chart
	 */
	public void setChart(Chart chart) {
		getPlotData().setChart(chart);
	}

	/**
	 * Returns the pane at the index 0
	 */
	public Pane getPane() {
		return getPlotData().getPane();
	}

	/**
	 * Returns the pane from the given index
	 */
	public Pane getPane(int index) {
		return getPlotData().getPane(index);
	}

	/**
	 * Returns the size of the pane list
	 */
	public int getPaneSize() {
		return getPlotData().getPaneSize();
	}

	/**
	 * Returns the Legend options, if any. Otherwise, create a new one.
	 * 
	 * @see Legend
	 */
	public Legend getLegend() {
		return getPlotData().getLegend();
	}

	/**
	 * Returns the Loading options, if any. Otherwise, create a new one.
	 * 
	 * @see Loading
	 */
	public Loading getLoading() {
		return getPlotData().getLoading();
	}

	/**
	 * Sets the Legend options
	 * 
	 * @param legend
	 */
	public void setLegend(Legend legend) {
		getPlotData().setLegend(legend);
	}

	/**
	 * Returns the labels options
	 */
	public Labels getLabels() {
		return getPlotData().getLabels();
	}

	/**
	 * Sets the labels options
	 */
	public void setLabels(Labels labels) {
		getPlotData().setLabels(labels);
	}

	/**
	 * Sets the navigation options
	 */
	public void setNavigation(Navigation navigation) {
		getPlotData().setNavigation(navigation);
	}

	/**
	 * Returns the navigation options
	 */
	public Navigation getNavigation() {
		return getPlotData().getNavigation();
	}

	/**
	 * Returns the plotOptions option, if any. Otherwise, create a new one.
	 * 
	 * @see PlotOptions
	 */
	public PlotOptions getPlotOptions() {
		return getPlotData().getPlotOptions();
	}

	/**
	 * Sets the plotOptions options
	 * 
	 * @param plotOptions
	 */
	public void setPlotOptions(PlotOptions plotOptions) {
		getPlotData().setPlotOptions(plotOptions);
	}

	/**
	 * Returns the Series options, if any. Otherwise, create a new one.
	 * <p>
	 * Returns the first series (index 0) by default.
	 * 
	 * @see Series
	 * @see #getSeries(int)
	 */
	public Series getSeries() {
		return getPlotData().getSeries();
	}

	/**
	 * Add a series to the chart after render time. Note that this method should
	 * never be used when adding data synchronously at chart render time, as it
	 * adds expense to the calculations and rendering. When adding data at the
	 * same time as the chart is initiated, add the series as a configuration
	 * option instead.
	 * 
	 * @param series
	 */
	public void addSeries(Series series) {
		getPlotData().addSeries(series);
	}

	/**
	 * Add an axis to the chart after render time. Note that this method should
	 * never be used when adding data synchronously at chart render time, as it
	 * adds expense to the calculations and rendering. When adding data at the
	 * same time as the chart is initiated, add the axis as a configuration
	 * option instead.
	 * 
	 * @param axis
	 */
	public void addAxis(Axis axis) {
		if (axis instanceof XAxis)
			getPlotData().addXAxis((XAxis) axis);
		else if (axis instanceof YAxis)
			getPlotData().addYAxis((YAxis) axis);
	}

	/**
	 * Returns the size of series
	 */
	public int getSeriesSize() {
		return getPlotData().getSeriesSize();
	}

	/**
	 * Returns the Series options from the given index, if any. Otherwise,
	 * create a new one.
	 * 
	 * @see Series
	 */
	public Series getSeries(int index) {
		return getPlotData().getSeries(index);
	}

	/**
	 * Returns the XAxis options, if any. Otherwise, create a new one.
	 * <p>
	 * Returns the first xAxis (index 0) by default.
	 * 
	 * @see XAxis
	 * @see #getXAxis(int)
	 */
	public XAxis getXAxis() {
		return getPlotData().getXAxis();
	}

	/**
	 * Returns the XAxis options from the given index, if any. Otherwise, create
	 * a new one.
	 * 
	 * @see XAxis
	 */
	public XAxis getXAxis(int index) {
		return getPlotData().getXAxis(index);
	}

	/**
	 * Returns the size of xAxis
	 */
	public int getXAxisSize() {
		return getPlotData().getXAxisSize();
	}

	/**
	 * Returns the YAxis options, if any. Otherwise, create a new one.
	 * <p>
	 * Returns the first yAxis (index 0) by default.
	 * 
	 * @see YAxis
	 * @see #getYAxis(int)
	 */
	public YAxis getYAxis() {
		return getPlotData().getYAxis();
	}

	/**
	 * Returns the YAxis options from the given index, if any. Otherwise, create
	 * a new one.
	 * 
	 * @see YAxis
	 */
	public YAxis getYAxis(int index) {
		return getPlotData().getYAxis(index);
	}

	/**
	 * Returns the size of yAxis
	 */
	public int getyAxisSize() {
		return getPlotData().getYAxisSize();
	}

	/**
	 * Sets the chart title.
	 */
	public void setTitle(String title) {
		getTitle().setText(title);
	}

	/**
	 * Sets the Title options
	 * 
	 * @param title
	 */
	public void setTitle(Title title) {
		getPlotData().setTitle(title);
	}

	/**
	 * Returns the Title options, if any. Otherwise, create a new one.
	 * 
	 * @see Title
	 */
	public Title getTitle() {
		return getPlotData().getTitle();
	}

	/**
	 * Returns the Subtitle options, if any. Otherwise, create a new one.
	 * 
	 * @see Subtitle
	 */
	public Subtitle getSubtitle() {
		return getPlotData().getSubtitle();
	}

	/**
	 * Sets the chart subtitle
	 * 
	 * @param subtitle
	 */
	public void setSubtitle(String subtitle) {
		getSubtitle().setText(subtitle);
	}

	/**
	 * Sets the Subtitle options
	 * 
	 * @param subtitle
	 */
	public void setSubtitle(Subtitle subtitle) {
		getPlotData().setSubtitle(subtitle);
	}

	/**
	 * Sets whether selection event will zoom the chart to the selected area.
	 * 
	 * @see Chart#setEnableZoomSelection(boolean)
	 */
	public void setEnableZoomSelection(boolean enableZoomSelection) {
		getChart().setEnableZoomSelection(enableZoomSelection);
	}

	/**
	 * Returns whether selection event will zoom the chart to the selected area.
	 * <p>
	 * Default: true.
	 * 
	 * @see Chart#isEnableZoomSelection()
	 */
	public boolean isEnableZoomSelection() {
		return getChart().isEnableZoomSelection();
	}

	private static String WRAP_FUNC = "function () {var self = this;this.%1$s;}";
	private static String RETURN_FUNC = "(function () {return self.%1$s;})()";

	/**
	 * Add a series to the chart as drilldown from a specific point in the
	 * parent series. This method is used for async drilldown, when clicking a
	 * point in a series should result in loading and displaying a more
	 * high-resolution series. When not async, the setup is simpler using the
	 * drilldown.series options structure.
	 * 
	 * @param point
	 *            The existing Point object from which the drilldown will start.
	 * @param series
	 *            The series options, as documented under plotOptions.series and
	 *            under the plotOptions for each series type
	 */
	public void addSeriesAsDrilldown(Point point, Series series) {
		_drilldownLevels.addFirst(series);
		JSFunction returnFunc = new JSFunction();
		returnFunc
				.callArray("data", point.getSeries().getData().indexOf(point));
		returnFunc.callArray("series", 0); // in drilldown mode, at client only
											// has one series
		JSFunction func = new JSFunction();
		func.callFunction(
				"addSeriesAsDrilldown",
				new JavaScriptValue(String.format(RETURN_FUNC,
						returnFunc.reverse())), series);
		smartUpdate("eval",
				new JavaScriptValue(String.format(WRAP_FUNC, func)), true);
	}

	private void doSmartDraw() {
		if (!checkLock()) return;
		
		if (_model == null)
			throw new UiException("Chart Model cannot be null!");

		PlotData plotData = getPlotEngine().drawPlot(_model);
		if (!_plotData.equals(plotData))
			_plotData.merge(plotData);
		smartUpdate("plotData", _plotData);
	}

	private transient PlotEngine _chartImpl; // chart implementation

	private PlotEngine getPlotEngine() {
		if (_chartImpl != null)
			return _chartImpl;

		final String type = getType();
		if (LINE.equals(type)) {
			_chartImpl = new LinePlotImpl(this);
		} else if (AREA.equals(type)) {
			_chartImpl = new AreaPlotImpl(this);
		} else if (PIE.equals(type)) {
			_chartImpl = new PiePlotImpl(this);
		} else if (BAR.equals(type)) {
			_chartImpl = new BarPlotImpl(this);
		} else if (COLUMN.equals(type)) {
			_chartImpl = new ColumnPlotImpl(this);
		} else if (AREA_SPLINE.equals(type)) {
			_chartImpl = new AreaSplinePlotImpl(this);
		} else if (SCATTER.equals(type)) {
			_chartImpl = new ScatterPlotImpl(this);
		} else if (BUBBLE.equals(type)) {
			_chartImpl = new BubblePlotImpl(this);
		} else if (WATERFALL.equals(type)) {
			_chartImpl = new WaterfallPlotImpl(this);
		} else if (POLAR.equals(type)) {
			_chartImpl = new PolarPlotImpl(this);
		} /*
		 * TODO Unsupported Chart Model else if (BOX_PLOT.equals(type)) {
		 * _chartImpl = new BoxPlotImpl(this); }
		 */else if (SPLINE.equals(type)) {
			_chartImpl = new SplinePlotImpl(this);
		} else if (FUNNEL.equals(type)) {
			_chartImpl = new FunnelPlotImpl(this);
		} else if (AREA_RANGE.equals(type)) {
			_chartImpl = new AreaRangePlotImpl(this);
		} else if (AREA_SPLINE_RANGE.equals(type)) {
			_chartImpl = new AreaSplineRangePlotImpl(this);
		} else if (COLUMN_RANGE.equals(type)) {
			_chartImpl = new ColumnRangePlotImpl(this);
		} else if (ERROR_BAR.equals(type)) {
			_chartImpl = new ErrorBarPlotImpl(this);
		} else if (GAUGE.equals(type)) {
			_chartImpl = new GaugePlotImpl(this);
		} else {
			throw new IllegalArgumentException("Unsupported type: [" + type
					+ "]");
		}
		return _chartImpl;
	}

	private boolean _rendered;

	/**
	 * Returns whether the component has been rendered to client side. Most of
	 * the time this method is used for component developer only.
	 */
	public boolean isRendered() {
		return _rendered;
	}

	// override
	public void invalidate() {
		super.invalidate();
		_rendered = false; // reset
		if (_drilldownLevels != null)
			_drilldownLevels.clear();
	}

	// override
	protected void renderProperties(org.zkoss.zk.ui.sys.ContentRenderer renderer)
			throws java.io.IOException {
		_rendered = true;
		super.renderProperties(renderer);

		// we shoulld use _befInvoke at the first
		if (!_befInvoke.isEmpty()) {
			render(renderer, "eval", _befInvoke);
			_befInvoke.clear();
		}

		if (_plotData != null)
			render(renderer, "plotData", _plotData);

		if (_options != null)
			render(renderer, "options", _options); // just in case

		if (_theme != null)
			render(renderer, "theme", _theme);

		render(renderer, "zclass", _zclass);
	}

	private LinkedList<Series> _drilldownLevels;

	protected void smartUpdate(String attr, Object value, boolean append) {
		super.smartUpdate(attr, value, append);
	}

	@SuppressWarnings("rawtypes")
	public void service(AuRequest request, boolean everError) {
		final String cmd = request.getCommand();
		final Map data = request.getData();

		if (cmd.equals("dummyEcho")) {
			String attrId = (String) data.get("id");
			EventListener listener = (EventListener) removeAttribute(attrId);
			try {
				listener.onEvent(new Event("onCallBack", this, data
						.get("value")));
			} catch (Exception e) {
				e.printStackTrace();
			}
		} else if (cmd.startsWith("onPlot")) {
			if (ChartsEvents.ON_PLOT_DRILL_UP.equals(cmd)) {
				_drilldownLevels.pop();
				final Series first = _drilldownLevels.getFirst();
				ChartsEvent evt = new ChartsEvent(request.getCommand(),
						request.getComponent(), -1, -1, null, false) {
					public Series getSeries() {
						return first;
					}
				};
				Events.postEvent(evt);
			} else if (ChartsEvents.ON_PLOT_DRILL_DOWN.equals(cmd)) {
				int sIndex = AuRequests.getInt(data, "sIndex", -1);
				ChartsEvent evt = null;
				if (sIndex == 0) {// first time drilldown
					evt = ChartsEvent.getChartsEvent(request);
				} else {
					final Series first = _drilldownLevels.getFirst();
					evt = new ChartsEvent(request.getCommand(),
							request.getComponent(), -1, AuRequests.getInt(data,
									"pIndex", -1),
							(Comparable<?>) data.get("category"), false) {
						public Series getSeries() {
							return first;
						}
					};
				}

				if (_drilldownLevels == null) {
					_drilldownLevels = new LinkedList<Series>();
					_drilldownLevels.push(evt.getSeries());
				}
				final int size = _drilldownLevels.size();

				Events.postEvent(evt);
				// user may use drilldown options
				// so we need to put the drilldown series in the
				// _drilldownLevels list
				if (_drilldownLevels.size() == size) {
					final String name = evt.getPoint().getDrilldown();

					for (Series seri : getDrilldown().getSeries()) {
						if (name.equals(seri.getId())) {
							_drilldownLevels.addFirst(seri);
							break;
						}
					}
				}
			} else if (ChartsEvents.ON_PLOT_SELECT.equals(cmd)) {
				ChartsEvent event = ChartsEvent.getChartsEvent(request);
				syncSelectedPoints(event.getPoint(), true);
				Events.postEvent(event);
			} else if (ChartsEvents.ON_PLOT_UNSELECT.equals(cmd)) {
				ChartsEvent event = ChartsEvent.getChartsEvent(request);
				syncSelectedPoints(event.getPoint(), false);
				Events.postEvent(event);
			} else
				Events.postEvent(ChartsEvent.getChartsEvent(request));
		} else if (Events.ON_CLICK.equals(cmd)) {
			Events.postEvent(ChartsClickEvent.getChartsClickEvent(request));
		} else if (Events.ON_SELECTION.equals(cmd)) {
			Events.postEvent(ChartsSelectionEvent
					.getChartsSelectionEvent(request));
		} else {
			super.service(request, everError);
		}
	}
	
	private void syncSelectedPoints(Point selPoint, boolean selected) {
		disableClientUpdate(true);
		try {
			selPoint.setSelectedDirectly(selected);
			if (selected) {
				_selPoints.add(selPoint);
			} else {
				_selPoints.remove(selPoint);
			}
		} finally {
			disableClientUpdate(false);
		}
	}
	
	public void selectPoint(Point selPoint, boolean accumulate) {
		disableClientUpdate(true);
		try {
			_selPoints.remove(selPoint);
			if (!accumulate) {
				for (Point p: _selPoints) {
					p.setSelectedDirectly(false);
				}
				_selPoints.clear();
			}
			if (selPoint.isSelected())
				_selPoints.add(selPoint);
		} finally {
			disableClientUpdate(false);
		}
	}
	
	/**
	 * Returns a readonly set of all currently selected points in the chart.
	 * Points can be selected by {@link Point#setSelected(boolean)}, {@link Point#select(boolean, boolean)} or by clicking.
	 * <p>
	 * Note: the returned set is not live data. You need to call {@link #getSelectedPoints()} again to 
	 * retrieve the modified set when changing the point's selected attribute. 
	 */
	public Set<Point> getSelectedPoints() {
		return Collections.unmodifiableSet(new HashSet<Point>(_selPoints));
	}

	/**
	 * Clears the selected points in the chart.
	 */
	public void clearPointSelection() {
		if (_selPoints.isEmpty())
			return;
		for (Point p: getSelectedPoints()) {
			p.setSelected(false);
		}
	}

	/**
	 * Sets the zclass for this component
	 * 
	 * @param zclass
	 */
	public void setZclass(String zclass) {
		if (!(Objects.equals(_zclass, zclass))) {
			_zclass = zclass;
			smartUpdate("zclass", zclass);
		}
	}

	/**
	 * The default zclass is "z-charts"
	 */
	public String getZclass() {
		return (this._zclass != null ? this._zclass : "z-charts");
	}
}
