/** Chart.java.

	Purpose:
		
	Description:
		
	History:
		11:52:02 AM Jan 13, 2014, Created by RaymondChao

Copyright (C) 2014 Potix Corporation. All Rights Reserved.
 */

package org.zkoss.chart;

import java.util.LinkedHashMap;
import java.util.Map;

import org.zkoss.chart.plotOptions.SeriesPlotOptions;
import org.zkoss.chart.util.DynamicalAttribute;
import org.zkoss.chart.util.MapsHelper;
import org.zkoss.lang.Generics;

/**
 * An Chart options regarding the chart area and plot area as well as general
 * chart options.
 * 
 * <p> All the options in this class support {@link DynamicalAttribute}.
 * 
 * @author jumperchen
 * @author RaymondChao
 */
public class Chart extends Optionable {
	private enum Attrs implements PlotAttribute, DynamicalAttribute {
		alignTicks,
		animation,
		backgroundColor,
		borderColor,
		borderRadius,
		borderWidth,
		// className, Same as zclass
		// defaultSeriesType, Same as type
		// events, Same as onXXX
		height,
		ignoreHiddenSeries,
		inverted,
		margin,
		marginBottom,
		marginLeft,
		marginRight,
		marginTop,
		pinchType,
		plotBackgroundColor,
		plotBackgroundImage,
		plotBorderColor,
		plotBorderWidth,
		plotShadow,
		polar,
		reflow,
		// renderTo,
		resetZoomButton,
		selectionMarkerFill,
		shadow,
		showAxes,
		spacing,
		spacingBottom,
		spacingLeft,
		spacingRight,
		spacingTop,
		style,
		type,
		width,
		zoomType,
		enableZoomSelection
	}

	/**
	 * Returns when using multiple axis, the ticks of two or more opposite axes
	 * will automatically be aligned by adding ticks to the axis or axes with
	 * the least ticks. This can be prevented by setting <code>alignTicks</code>
	 * to false. If the grid lines look messy, it's a good idea to hide them for
	 * the secondary axis by setting <code>gridLineWidth</code> to 0.
	 * <p>
	 * Default: true.
	 */
	public boolean isAlignTicks() {
		return getAttr(Attrs.alignTicks, true).asBoolean();
	}

	/**
	 * Sets when using multiple axis, the ticks of two or more opposite axes
	 * will automatically be aligned by adding ticks to the axis or axes with
	 * the least ticks. This can be prevented by setting <code>alignTicks</code>
	 * to false. If the grid lines look messy, it's a good idea to hide them for
	 * the secondary axis by setting <code>gridLineWidth</code> to 0.
	 */
	public void setAlignTicks(boolean alignTicks) {
		setAttr(Attrs.alignTicks, alignTicks);
	}

	/**
	 * Returns the overall animation for all chart updating.
	 * <p>
	 * Default: an instance of {@link Animation}
	 */
	public Animation getAnimation() {
		if (containsKey(Attrs.animation)) {
			return (Animation) getAttr(Attrs.animation);
		} else {
			return (Animation) getAttr(Attrs.animation,
					org.zkoss.chart.Animation.class).asValue();
		}
	}

	/**
	 * Returns whether enable the animation
	 * <p>
	 * Default: true
	 */
	public boolean isAnimation() {
		return getAnimation() != Animation.NONE;
	}

	/**
	 * Animation can be disabled throughout the chart by setting it to false
	 * here. It can be overridden for each individual API method as a function
	 * parameter. The only animation not affected by this option is the initial
	 * series animation, see {@link SeriesPlotOptions#setAnimation(boolean)}.
	 * <p>
	 * The animation can either be set as a boolean or a configuration object.
	 * See {@link #setAnimation(Animation)}. If <code>true</code>, it will use
	 * the 'swing' jQuery easing and a duration of 500 ms. If used as an
	 * Animation object, the following properties are supported:
	 * </p>
	 * <dl>
	 * <dt>duration</dt>
	 * <dd>The duration of the animation in milliseconds.</dd>
	 * <dt>easing</dt>
	 * <dd>When using jQuery as the general framework, the easing can be set to
	 * <code>linear</code> or <code>swing</code>. More easing functions are
	 * available with the use of jQuery plug-ins, most notably the jQuery UI
	 * suite. See <a href="http://api.jquery.com/animate/">the jQuery docs</a>.
	 * When using MooTools as the general framework, use the property name
	 * <code>transition</code> instead of <code>easing</code>.</dd>
	 * </dl>
	 * <p>
	 * 
	 * @see #setAnimation(Animation)
	 * @see Animation
	 */
	public void setAnimation(boolean animation) {
		setAnimation(animation ? new Animation() : Animation.NONE);
	}

	/**
	 * Animation can be disabled throughout the chart by setting it to false by
	 * the method of {@link #setAnimation(boolean)}. It can be overridden for
	 * each individual API method as a function parameter. The only animation
	 * not affected by this option is the initial series animation, see
	 * {@link SeriesPlotOptions#setAnimation(boolean)}.
	 * <p>
	 * Properties are supported:
	 * <dl>
	 * <dt>duration</dt>
	 * <dd>The duration of the animation in milliseconds.</dd>
	 * <dt>easing</dt>
	 * <dd>When using jQuery as the general framework, the easing can be set to
	 * <code>linear</code> or <code>swing</code>. More easing functions are
	 * available with the use of jQuery plug-ins, most notably the jQuery UI
	 * suite. See <a href="http://api.jquery.com/animate/">the jQuery docs</a>.
	 * When using MooTools as the general framework, use the property name
	 * <code>transition</code> instead of <code>easing</code>.</dd>
	 * </dl>
	 * <p>
	 * 
	 * @see #setAnimation(boolean)
	 * @see Animation
	 */
	public void setAnimation(Animation animation) {
		setAttr(Attrs.animation, animation == null ? Animation.NONE
				: animation);
	}

	/**
	 * Returns the background color or gradient for the outer chart area.
	 * <p>
	 * Default: "#FFFFFF".
	 */
	public Color getBackgroundColor() {
		if (!containsKey(Attrs.backgroundColor)) {
			setBackgroundColor("#A0A0A0");
		}
		return (Color) getAttr(Attrs.backgroundColor);
	}

	/**
	 * Sets the background color or gradient for the outer chart area.
	 */
	public void setBackgroundColor(Color color) {
		setAttr(Attrs.backgroundColor, color, NOT_NULL_VALUE);
	}

	/**
	 * Sets the background color or gradient for the outer chart area.
	 */
	public void setBackgroundColor(String color) {
		setBackgroundColor(new Color(color));
	}

	/**
	 * Sets the background color or gradient for the outer chart area.
	 */
	public void setBackgroundColor(LinearGradient color) {
		setBackgroundColor(new Color(color));
	}

	/**
	 * Sets the background color or gradient for the outer chart area.
	 */
	public void setBackgroundColor(RadialGradient color) {
		setBackgroundColor(new Color(color));
	}

	/**
	 * Returns the color of the outer chart border. The border is painted using
	 * vector graphic techniques to allow rounded corners.
	 * <p>
	 * Default: "#4572A7".
	 */
	public Color getBorderColor() {
		if (!containsKey(Attrs.borderColor)) {
			setBorderColor("#4572A7");
		}
		return (Color) getAttr(Attrs.borderColor);
	}

	/**
	 * Sets the color of the outer chart border. The border is painted using
	 * vector graphic techniques to allow rounded corners.
	 */
	public void setBorderColor(Color color) {
		setAttr(Attrs.borderColor, color, NOT_NULL_VALUE);
	}

	/**
	 * Sets the color of the outer chart border. The border is painted using
	 * vector graphic techniques to allow rounded corners.
	 */
	public void setBorderColor(String color) {
		setBorderColor(new Color(color));
	}

	/**
	 * Sets the color of the outer chart border. The border is painted using
	 * vector graphic techniques to allow rounded corners.
	 */
	public void setBorderColor(LinearGradient color) {
		setBorderColor(new Color(color));
	}

	/**
	 * Sets the color of the outer chart border. The border is painted using
	 * vector graphic techniques to allow rounded corners.
	 */
	public void setBorderColor(RadialGradient color) {
		setBorderColor(new Color(color));
	}

	/**
	 * Returns the corner radius of the outer chart border. In export, the
	 * radius defaults to 0.
	 * <p>
	 * Default: 5.
	 */
	public Number getBorderRadius() {
		return getAttr(Attrs.borderRadius, 5).asNumber();
	}

	/**
	 * Sets the corner radius of the outer chart border. In export, the radius
	 * defaults to 0.
	 * <p>
	 * Default: 5.
	 * 
	 * @param borderRadius
	 *            the corner radius of the outer chart border
	 */
	public void setBorderRadius(Number borderRadius) {
		setAttr(Attrs.borderRadius, borderRadius, 5);
	}

	/**
	 * Returns the pixel width of the outer chart border. The border is painted
	 * using vector graphic techniques to allow rounded corners.
	 * <p>
	 * Default: 0.
	 */
	public Number getBorderWidth() {
		return getAttr(Attrs.borderWidth, 0).asNumber();
	}

	/**
	 * Sets the pixel width of the outer chart border. The border is painted
	 * using vector graphic techniques to allow rounded corners.
	 * 
	 * @param borderWidth
	 *            the pixel width of the outer chart border
	 */
	public void setBorderWidth(Number borderWidth) {
		setAttr(Attrs.borderWidth, borderWidth, 0);
	}

	/**
	 * Returns an explicit height for the chart. By default the height is
	 * calculated from the offset height of the containing element, or 400
	 * pixels if the containing element's height is 0.
	 * <p>
	 * Default: null.
	 */
	public Number getHeight() {
		return getAttr(Attrs.height, null).asNumber();
	}

	/**
	 * Sets an explicit height for the chart. By default the height is
	 * calculated from the offset height of the containing element, or 400
	 * pixels if the containing element's height is 0.
	 */
	public void setHeight(Number height) {
		setAttr(Attrs.height, height);
	}

	/**
	 * Returns if true, the axes will scale to the remaining visible series once
	 * one series is hidden. If false, hiding and showing a series will not
	 * affect the axes or the other series. For stacks, once one series within
	 * the stack is hidden, the rest of the stack will close in around it even
	 * if the axis is not affected.
	 * <p>
	 * Default: true.
	 */
	public boolean isIgnoreHiddenSeries() {
		return getAttr(Attrs.ignoreHiddenSeries, true).asBoolean();
	}

	/**
	 * Sets to ignore the hidden series, if true, the axes will scale to the
	 * remaining visible series once one series is hidden. If false, hiding and
	 * showing a series will not affect the axes or the other series. For
	 * stacks, once one series within the stack is hidden, the rest of the stack
	 * will close in around it even if the axis is not affected.
	 * 
	 * @param ignoreHiddenSeries
	 *            if true, the axes will scale to the remaining visible series
	 *            once one series is hidden
	 */
	public void setIgnoreHiddenSeries(boolean ignoreHiddenSeries) {
		setAttr(Attrs.ignoreHiddenSeries, ignoreHiddenSeries);
	}

	/**
	 * Returns whether to invert the axes so that the x axis is vertical and y
	 * axis is horizontal. When true, the x axis is reversed by default. If a
	 * bar series is present in the chart, it will be inverted automatically.
	 * <p>
	 * Default: false.
	 */
	public boolean isInverted() {
		return getAttr(Attrs.inverted, false).asBoolean();
	}

	/**
	 * Sets whether to invert the axes so that the x axis is vertical and y axis
	 * is horizontal. When true, the x axis is reversed by default. If a bar
	 * series is present in the chart, it will be inverted automatically.
	 * 
	 * @param inverted
	 *            whether to invert the axes so that the x axis is vertical and
	 *            y axis is horizontal
	 */
	public void setInverted(boolean inverted) {
		setAttr(Attrs.inverted, inverted);
	}

	/**
	 * Returns the margin between the outer edge of the chart and the plot area.
	 * The numbers in the array designate top, right, bottom and left
	 * respectively. Use the methods of {@link #setMarginTop(Number)},
	 * {@link #setMarginRight(Number)}, {@link #setMarginBottom(Number)} and
	 * {@link #setMarginLeft(Number)} for shorthand setting of one option.
	 * <p>
	 * Default: null.
	 */
	public Number[] getMargin() {
		return getAttr(Attrs.margin, null).asNumberArray();
	}

	/**
	 * Sets the margin between the outer edge of the chart and the plot area.
	 * The numbers in the array designate top, right, bottom and left
	 * respectively. Use the methods of {@link #setMarginTop(Number)},
	 * {@link #setMarginRight(Number)}, {@link #setMarginBottom(Number)} and
	 * {@link #setMarginLeft(Number)} for shorthand setting of one option.
	 * <p>
	 * Default: null.
	 */
	public void setMargin(Number[] margin) {
		setAttr(Attrs.margin, margin);
	}

	/**
	 * Returns the margin between the bottom outer edge of the chart and the
	 * plot area.
	 * <p>
	 * Default: null.
	 */
	public Number getMarginBottom() {
		return getAttr(Attrs.marginBottom, null).asNumber();
	}

	/**
	 * Sets the margin between the bottom outer edge of the chart and the plot
	 * area. Use this to set a fixed pixel value for the margin as opposed to
	 * the default dynamic margin. See also {@link #setSpacingBottom(Number)}.
	 * 
	 * @param marginBottom
	 *            the margin between the bottom outer edge of the chart and the
	 *            plot area
	 */
	public void setMarginBottom(Number marginBottom) {
		setAttr(Attrs.marginBottom, marginBottom);
	}

	/**
	 * Returns the margin between the left outer edge of the chart and the plot
	 * area.
	 * <p>
	 * Default: null.
	 */
	public Number getMarginLeft() {
		return getAttr(Attrs.marginLeft, null).asNumber();
	}

	/**
	 * Sets the margin between the left outer edge of the chart and the plot
	 * area. Use this to set a fixed pixel value for the margin as opposed to
	 * the default dynamic margin. See also {@link #setSpacingLeft(Number)}
	 * 
	 * @param marginLeft
	 *            the margin between the left outer edge of the chart and the
	 *            plot area
	 */
	public void setMarginLeft(Number marginLeft) {
		setAttr(Attrs.marginLeft, marginLeft);
	}

	/**
	 * Returns the margin between the right outer edge of the chart and the plot
	 * area.
	 * <p>
	 * Default: null
	 */
	public Number getMarginRight() {
		return getAttr(Attrs.marginRight, null).asNumber();
	}

	/**
	 * Sets the margin between the right outer edge of the chart and the plot
	 * area. Use this to set a fixed pixel value for the margin as opposed to
	 * the default dynamic margin. See also {@link #setSpacingRight(Number)}.
	 * 
	 * @param marginRight
	 *            the margin between the right outer edge of the chart and the
	 *            plot area
	 */
	public void setMarginRight(Number marginRight) {
		setAttr(Attrs.marginRight, marginRight);
	}

	/**
	 * Returns the margin between the top outer edge of the chart and the plot
	 * area.
	 * <p>
	 * Default: null.
	 */
	public Number getMarginTop() {
		return getAttr(Attrs.marginTop, null).asNumber();
	}

	/**
	 * Sets the margin between the top outer edge of the chart and the plot
	 * area. Use this to set a fixed pixel value for the margin as opposed to
	 * the default dynamic margin. See also {@link #setSpacingTop(Number)}.
	 * 
	 * @param marginTop
	 *            the margin between the top outer edge of the chart and the
	 *            plot area
	 */
	public void setMarginTop(Number marginTop) {
		setAttr(Attrs.marginTop, marginTop);
	}

	/**
	 * Returns equivalent to {@link #getZoomType()}, but for multitouch
	 * gestures only. By default, the <code>pinchType</code> is the same as the
	 * <code>zoomType</code> setting. However, pinching can be enabled
	 * separately in some cases, for example in stock charts where a mouse drag
	 * pans the chart, while pinching is enabled.
	 * <p>
	 * Default: null.
	 */
	public String getPinchType() {
		return getAttr(Attrs.pinchType, null).asString();
	}

	/**
	 * Sets equivalent to {@link #setZoomType(String)}, but for multitouch
	 * gestures only. By default, the <code>pinchType</code> is the same as the
	 * <code>zoomType</code> setting. However, pinching can be enabled
	 * separately in some cases, for example in stock charts where a mouse drag
	 * pans the chart, while pinching is enabled.
	 * <p>
	 * Default: null.
	 * 
	 * @see #setZoomType(String)
	 */
	public void setPinchType(String pinchType) {
		setAttr(Attrs.pinchType, pinchType);
	}

	/**
	 * Returns the background color for the plot area.
	 * <p>
	 * Default: null.
	 */
	public Color getPlotBackgroundColor() {
		return (Color) getAttr(Attrs.plotBackgroundColor, null).asValue();
	}

	/**
	 * Sets the background color for the plot area.
	 */
	public void setPlotBackgroundColor(Color color) {
		setAttr(Attrs.plotBackgroundColor, color);
	}

	/**
	 * Sets the background color for the plot area.
	 */
	public void setPlotBackgroundColor(String color) {
		setPlotBackgroundColor(new Color(color));
	}

	/**
	 * Sets the background color for the plot area.
	 */
	public void setPlotBackgroundColor(LinearGradient color) {
		setPlotBackgroundColor(new Color(color));
	}

	/**
	 * Sets the background color for the plot area.
	 */
	public void setPlotBackgroundColor(RadialGradient color) {
		setPlotBackgroundColor(new Color(color));
	}
	
	/**
	 * Returns the URL for an image to use as the plot background.
	 * <p>
	 * Default: null.
	 */
	public String getPlotBackgroundImage() {
		return getAttr(Attrs.plotBackgroundImage, null).asString();
	}

	/**
	 * Sets the URL for an image to use as the plot background. To set an image
	 * as the background for the entire chart, set a CSS background image to the
	 * container element. Note that for the image to be applied to exported
	 * charts, its URL needs to be accessible by the export server.
	 * 
	 * @param plotBackgroundImage
	 *            the URL for an image to use as the plot background
	 */
	public void setPlotBackgroundImage(String plotBackgroundImage) {
		setAttr(Attrs.plotBackgroundImage, plotBackgroundImage);
	}

	/**
	 * Returns the color of the inner chart or plot area border.
	 * <p>
	 * Default: "#C0C0C0".
	 */
	public Color getPlotBorderColor() {
		if (!containsKey(Attrs.plotBorderColor)) {
			setPlotBorderColor("#C0C0C0");
		}
		return (Color) getAttr(Attrs.plotBorderColor);
	}

	/**
	 * Sets the color of the inner chart or plot area border.
	 */
	public void setPlotBorderColor(Color color) {
		setAttr(Attrs.plotBorderColor, color, NOT_NULL_VALUE);
	}

	/**
	 * Sets the color of the inner chart or plot area border.
	 */
	public void setPlotBorderColor(String color) {
		setPlotBorderColor(new Color(color));
	}

	/**
	 * Sets the color of the inner chart or plot area border.
	 */
	public void setPlotBorderColor(LinearGradient color) {
		setPlotBorderColor(new Color(color));
	}

	/**
	 * Sets the color of the inner chart or plot area border.
	 */
	public void setPlotBorderColor(RadialGradient color) {
		setPlotBorderColor(new Color(color));
	}

	/**
	 * Returns the pixel width of the plot area border.
	 * <p>
	 * Default: 0.
	 */
	public Number getPlotBorderWidth() {
		return getAttr(Attrs.plotBorderWidth, 0).asNumber();
	}

	/**
	 * Sets the pixel width of the plot area border.
	 * 
	 * @param plotBorderWidth
	 *            the pixel width of the plot area border
	 */
	public void setPlotBorderWidth(Number plotBorderWidth) {
		setAttr(Attrs.plotBorderWidth, plotBorderWidth, 0);
	}

	/**
	 * Returns whether to apply a drop shadow to the plot area. Requires that
	 * {@link #setPlotBackgroundColor(String)} be set.
	 * <p>
	 * Default: false.
	 */
	public boolean isPlotShadow() {
		return getAttr(Attrs.plotShadow, false).asBoolean();
	}

	/**
	 * Sets whether to apply a drop shadow to the plot area. Requires that
	 * {@link #setPlotBackgroundColor(String)} be set.
	 * 
	 * @param plotShadow
	 *            whether to apply a drop shadow to the plot area
	 */
	public void setPlotShadow(boolean plotShadow) {
		setAttr(Attrs.plotShadow, plotShadow);
	}

	/**
	 * Returns whether transforms into the polar coordinate system
	 * <p>
	 * Default: false.
	 */
	public boolean isPolar() {
		return getAttr(Attrs.polar, false).asBoolean();
	}

	/**
	 * Sets to true to enable the cartesian charts like line, spline, area and
	 * column are transformed into the polar coordinate system.
	 * 
	 * @param polar
	 *            when true, cartesian charts like line, spline, area and column
	 *            are transformed into the polar coordinate system
	 */
	public void setPolar(boolean polar) {
		setAttr(Attrs.polar, polar);
	}

	/**
	 * Returns whether to reflow the chart to fit the width of the container div
	 * on resizing the window.
	 * <p>
	 * Default: true.
	 */
	public boolean isReflow() {
		return getAttr(Attrs.reflow, true).asBoolean();
	}

	/**
	 * Sets whether to reflow the chart to fit the width of the container div on
	 * resizing the window.
	 * 
	 * @param reflow
	 *            whether to reflow the chart to fit the width of the container
	 *            div on resizing the window
	 */
	public void setReflow(boolean reflow) {
		setAttr(Attrs.reflow, reflow);
	}

	/**
	 * Returns the button that appears after a selection zoom, allowing the user
	 * to reset zoom.
	 * <p>
	 * Default: null.
	 */
	public ResetZoomButton getResetZoomButton() {
		return (ResetZoomButton) getAttr(Attrs.resetZoomButton, null).asValue();
	}

	/**
	 * Sets the button that appears after a selection zoom, allowing the user to
	 * reset zoom.
	 * 
	 * @param resetZoomButton
	 *            the button that appears after a selection zoom, allowing the
	 *            user to reset zoom
	 */
	public void setResetZoomButton(ResetZoomButton resetZoomButton) {
		setAttr(Attrs.resetZoomButton, resetZoomButton);
	}

	/**
	 * Returns the background color of the marker square when selecting (zooming
	 * in on) an area of the chart.
	 * <p>
	 * Default: "rgba(69,114,167,0.25)".
	 */
	public Color getSelectionMarkerFill() {
		if (!containsKey(Attrs.selectionMarkerFill)) {
			setSelectionMarkerFill("rgba(69,114,167,0.25)");
		}
		return (Color) getAttr(Attrs.selectionMarkerFill);
	}

	/**
	 * Sets the background color of the marker square when selecting (zooming in
	 * on) an area of the chart.
	 */
	public void setSelectionMarkerFill(Color color) {
		setAttr(Attrs.selectionMarkerFill, color, NOT_NULL_VALUE);
	}

	/**
	 * Sets the background color of the marker square when selecting (zooming in
	 * on) an area of the chart.
	 */
	public void setSelectionMarkerFill(String color) {
		setSelectionMarkerFill(new Color(color));
	}

	/**
	 * Sets the background color of the marker square when selecting (zooming in
	 * on) an area of the chart.
	 */
	public void setSelectionMarkerFill(LinearGradient color) {
		setSelectionMarkerFill(new Color(color));
	}

	/**
	 * Sets the background color of the marker square when selecting (zooming in
	 * on) an area of the chart.
	 */
	public void setSelectionMarkerFill(RadialGradient color) {
		setSelectionMarkerFill(new Color(color));
	}

	/**
	 * Returns whether to apply a drop shadow to the outer chart area. Requires
	 * that backgroundColor be set.
	 * <p>
	 * Default: false.
	 * 
	 * @see Shadow
	 */
	public boolean isShadow() {
		return getShadow() != Shadow.NONE;
	}

	/**
	 * Returns whether to apply a drop shadow to the outer chart area. Requires
	 * that backgroundColor be set.
	 * <p>
	 * Default: {@link Shadow#NONE}.
	 * 
	 * @see Shadow
	 */
	public Shadow getShadow() {
		return (Shadow) getAttr(Attrs.shadow, Shadow.NONE).asValue();
	}

	/**
	 * Sets whether to apply a drop shadow to the outer chart area. Requires
	 * that backgroundColor be set.
	 */
	public void setShadow(boolean shadow) {
		setShadow(shadow ? new Shadow() : Shadow.NONE);
	}

	/**
	 * Sets whether to apply a drop shadow to the outer chart area. Requires
	 * that backgroundColor be set.
	 * 
	 * @see Shadow
	 */
	public void setShadow(Shadow shadow) {
		setAttr(Attrs.shadow, shadow);
	}

	/**
	 * Returns whether to show the axes initially. This only applies to empty
	 * charts where series are added dynamically, as axes are automatically
	 * added to cartesian series.
	 * <p>
	 * Default: false.
	 */
	public boolean isShowAxes() {
		return getAttr(Attrs.showAxes, false).asBoolean();
	}

	/**
	 * Sets whether to show the axes initially. This only applies to empty
	 * charts where series are added dynamically, as axes are automatically
	 * added to cartesian series.
	 * 
	 * @param showAxes
	 *            whether to show the axes initially
	 */
	public void setShowAxes(boolean showAxes) {
		setAttr(Attrs.showAxes, showAxes);
	}

	/**
	 * Returns the distance between the outer edge of the chart and the content,
	 * like title, legend, axis title or labels. The numbers in the array
	 * designate top, right, bottom and left respectively.
	 * <p>
	 * Default: [10, 10, 15, 10].
	 */
	public Number[] getSpacing() {
		return getAttr(Attrs.spacing, new Number[] { 10, 10, 15, 10 })
				.asNumberArray();
	}

	/**
	 * Sets the distance between the outer edge of the chart and the content,
	 * like title, legend, axis title or labels. The numbers in the array
	 * designate top, right, bottom and left respectively. Use the methods of
	 * {@link #setSpacingTop(Number)}, {@link #setSpacingRight(Number)},
	 * {@link #setSpacingBottom(Number)} and {@link #setSpacingLeft(Number)} for
	 * shorthand setting of one option.
	 * 
	 * @param spacing
	 *            the distance between the outer edge of the chart and the
	 *            content, like title, legend, axis title or labels
	 */
	public void setSpacing(Number[] spacing) {
		setAttr(Attrs.spacing, spacing, NOT_NULL_VALUE);
	}

	/**
	 * Returns the space between the bottom edge of the chart and the content
	 * (plot area, axis title and labels, title, subtitle or legend in top
	 * position).
	 * <p>
	 * Default: 15.
	 */
	public Number getSpacingBottom() {
		return getAttr(Attrs.spacingBottom, 15).asNumber();
	}

	/**
	 * Sets the space between the bottom edge of the chart and the content (plot
	 * area, axis title and labels, title, subtitle or legend in top position).
	 * </p>
	 * 
	 * @param spacingBottom
	 *            The space between the bottom edge of the chart and the content
	 *            (plot area, axis title and labels, title, subtitle or legend
	 *            in top position)
	 */
	public void setSpacingBottom(Number spacingBottom) {
		setAttr(Attrs.spacingBottom, spacingBottom, 15);
	}

	/**
	 * Returns the space between the left edge of the chart and the content
	 * (plot area, axis title and labels, title, subtitle or legend in top
	 * position).
	 * <p>
	 * Default: 10.
	 */
	public Number getSpacingLeft() {
		return getAttr(Attrs.spacingLeft, 10).asNumber();
	}

	/**
	 * Sets the space between the left edge of the chart and the content (plot
	 * area, axis title and labels, title, subtitle or legend in top position).
	 * </p>
	 * 
	 * @param spacingLeft
	 *            The space between the left edge of the chart and the content
	 *            (plot area, axis title and labels, title, subtitle or legend
	 *            in top position)
	 */
	public void setSpacingLeft(Number spacingLeft) {
		setAttr(Attrs.spacingLeft, spacingLeft, 10);
	}

	/**
	 * Returns the space between the right edge of the chart and the content
	 * (plot area, axis title and labels, title, subtitle or legend in top
	 * position).
	 * <p>
	 * Default: 10.
	 */
	public Number getSpacingRight() {
		return getAttr(Attrs.spacingRight, 10).asNumber();
	}

	/**
	 * Sets the space between the right edge of the chart and the content (plot
	 * area, axis title and labels, title, subtitle or legend in top position).
	 * 
	 * @param spacingRight
	 *            The space between the right edge of the chart and the content
	 *            (plot area, axis title and labels, title, subtitle or legend
	 *            in top position)
	 */
	public void setSpacingRight(Number spacingRight) {
		setAttr(Attrs.spacingRight, spacingRight, 10);
	}

	/**
	 * Returns the space between the top edge of the chart and the content (plot
	 * area, axis title and labels, title, subtitle or legend in top position).
	 * <p>
	 * Default: 10.
	 */
	public Number getSpacingTop() {
		return getAttr(Attrs.spacingTop, 10).asNumber();
	}

	/**
	 * Sets the space between the top edge of the chart and the content (plot
	 * area, axis title and labels, title, subtitle or legend in top position).
	 * 
	 * @param spacingTop
	 *            The space between the top edge of the chart and the content
	 *            (plot area, axis title and labels, title, subtitle or legend
	 *            in top position)
	 */
	public void setSpacingTop(Number spacingTop) {
		setAttr(Attrs.spacingTop, spacingTop, 10);
	}

	/**
	 * Returns additional CSS styles to apply inline to the container
	 * <code>div</code>. Note that since the default font styles are applied in
	 * the renderer, it is ignorant of the individual chart options and must be
	 * set globally. Defaults to:
	 * 
	 * <pre>
	 * fontFamily: "Lucida Grande", "Lucida Sans Unicode", Verdana, Arial, Helvetica, sans-serif; fontSize: 12px
	 * </pre>
	 * <p>
	 * Default: null.
	 */
	public <K, V> Map<K, V> getStyle() {
		if (!containsKey(Attrs.style)) {
			setStyle("fontFamily: \"Lucida Grande\", \"Lucida Sans Unicode\", Verdana, Arial, Helvetica, sans-serif; fontSize: 12px");
		}
		return Generics.cast(getAttr(Attrs.style));
	}

	/**
	 * Sets additional CSS styles to apply inline to the container
	 * <code>div</code>. Note that since the default font styles are applied in
	 * the renderer, it is ignorant of the individual chart options and must be
	 * set globally. Defaults to:
	 * 
	 * <pre>
	 * fontFamily: "Lucida Grande", "Lucida Sans Unicode", Verdana, Arial, Helvetica, sans-serif; fontSize: 12px
	 * </pre>
	 * 
	 * @param style
	 *            additional CSS styles to apply inline to the container
	 *            <code>div</code>
	 * @see #setStyle(Map)
	 */
	public void setStyle(String style) {
		setStyle(MapsHelper.parse(new LinkedHashMap<String, String>(), style, ':', ';', '\''));
	}

	/**
	 * Sets additional CSS styles to apply inline to the container
	 * <code>div</code>. Note that since the default font styles are applied in
	 * the renderer, it is ignorant of the individual chart options and must be
	 * set globally. Defaults to:
	 * 
	 * <pre>
	 * fontFamily: "Lucida Grande", "Lucida Sans Unicode", Verdana, Arial, Helvetica, sans-serif; fontSize: 12px
	 * </pre>
	 * 
	 * @param style
	 *            additional CSS styles to apply inline to the container
	 *            <code>div</code>
	 * @see #setStyle(String)
	 */
	public <K, V> void setStyle(Map<K, V> style) {
		setAttr(Attrs.style, style, NOT_NULL_VALUE);
	}

	/**
	 * Returns the default series type for the chart. Can be any of the chart
	 * types listed under {@link Charts}.
	 * <p>
	 * Default: "line".
	 */
	public String getType() {
		return getAttr(Attrs.type, "line").asString();
	}

	/**
	 * Sets the default series type for the chart. Can be any of the chart types
	 * listed under {@link Charts}.
	 * 
	 * @param type
	 *            the default series type for the chart
	 */
	public void setType(String type) {
		setAttr(Attrs.type, type, "line");
	}

	/**
	 * Returns an explicit width for the chart. By default the width is
	 * calculated from the offset width of the containing element.
	 * <p>
	 * Default: null.
	 */
	public Number getWidth() {
		return getAttr(Attrs.width, null).asNumber();
	}

	/**
	 * Sets an explicit width for the chart. By default the width is calculated
	 * from the offset width of the containing element.
	 * 
	 * @param width
	 *            an explicit width for the chart
	 */
	public void setWidth(Number width) {
		setAttr(Attrs.width, width);
	}

	/**
	 * Returns decides in what dimensions the user can zoom by dragging the
	 * mouse. Can be one of <code>x</code>, <code>y</code> or <code>xy</code>.
	 * <p>
	 * Default: null.
	 */
	public String getZoomType() {
		return getAttr(Attrs.zoomType, null).asString();
	}

	/**
	 * Sets decides in what dimensions the user can zoom by dragging the mouse.
	 * Can be one of <code>x</code>, <code>y</code> or <code>xy</code>.
	 * 
	 * @param zoomType
	 *            decides in what dimensions the user can zoom by dragging the
	 *            mouse
	 */
	public void setZoomType(String zoomType) {
		if (!(zoomType == null || "x".equals(zoomType) || "y".equals(zoomType) || "xy"
				.equals(zoomType)))
			throw new IllegalArgumentException("Unsupported zoomType: ["
					+ zoomType + "]");
		setAttr(Attrs.zoomType, zoomType);
	}
	
	/**
	 * Returns whether selection event will zoom the chart to the selected area.
	 * <p>
	 * Default: true.
	 * 
	 */
	public Boolean isEnableZoomSelection() {
		return getAttr(Attrs.enableZoomSelection, true).asBoolean();
	}
	
	/**
	 * Sets whether selection event will zoom the chart to the selected area.
	 * 
	 */
	public void setEnableZoomSelection(boolean enableZoomSelection) {
		setAttr(Attrs.enableZoomSelection, enableZoomSelection);
	}
}