/** AxisTitle.java.

	Purpose:
		
	Description:
		
	History:
		09:50:02 AM Jan 13, 2014, Created by RaymondChao

Copyright (C) 2014 Potix Corporation. All Rights Reserved.
 */

package org.zkoss.chart;

import java.util.LinkedHashMap;
import java.util.Map;

import org.zkoss.chart.util.DynamicalAttribute;
import org.zkoss.chart.util.MapsHelper;
import org.zkoss.lang.Generics;

/**
 * An option for axis' title
 * 
 * <p> All the options in this class support {@link DynamicalAttribute}.
 * 
 * @author jumperchen
 */
public class AxisTitle extends Optionable {
	private enum Attrs implements PlotAttribute, DynamicalAttribute {
		align, margin, offset, rotation, style, text, x, y
	}

	/**
	 * Returns alignment of the title relative to the axis values. Possible
	 * values are "low", "middle" or "high".
	 * <p>
	 * Default: "middle".
	 */
	public String getAlign() {
		return getAttr(Attrs.align, "middle").asString();
	}

	/**
	 * Sets alignment of the title relative to the axis values. Possible values
	 * are "low", "middle" or "high".
	 */
	public void setAlign(String align) {
		if (!("low".equals(align) || "middle".equals(align) || "high".equals(align)))
			throw new IllegalArgumentException("Unsupported align: [" + align + "]");
		setAttr(Attrs.align, align, "middle");
	}

	/**
	 * Returns the pixel distance between the axis labels or line and the title.
	 * Defaults to 0 for horizontal axes, 10 for vertical
	 * <p>
	 * Default: 0.
	 */
	public Number getMargin() {
		return getAttr(Attrs.margin, 0).asNumber();
	}

	/**
	 * Sets the pixel distance between the axis labels or line and the title.
	 * Defaults to 0 for horizontal axes, 10 for vertical
	 */
	public void setMargin(Number margin) {
		setAttr(Attrs.margin, margin, 0);
	}

	/**
	 * Returns the distance of the axis title from the axis line. By default,
	 * this distance is computed from the offset width of the labels, the
	 * labels' distance from the axis and the title's margin. However when the
	 * offset option is set, it overrides all this.
	 * <p>
	 * Default: null.
	 */
	public Number getOffset() {
		return getAttr(Attrs.offset, null).asNumber();
	}

	/**
	 * Sets the distance of the axis title from the axis line. By default, this
	 * distance is computed from the offset width of the labels, the labels'
	 * distance from the axis and the title's margin. However when the offset
	 * option is set, it overrides all this.
	 */
	public void setOffset(Number offset) {
		setAttr(Attrs.offset, offset);
	}

	/**
	 * Returns the rotation of the text in degrees. 0 is horizontal, 270 is
	 * vertical reading from bottom to top.
	 * <p>
	 * Default: 0.
	 * 
	 * @return rotation the rotation of the text in degrees
	 */
	public Number getRotation() {
		return getAttr(Attrs.rotation, 0).asNumber();
	}

	/**
	 * Sets the rotation of the text in degrees. 0 is horizontal, 270 is
	 * vertical reading from bottom to top.
	 */
	public void setRotation(Number rotation) {
		setAttr(Attrs.rotation, rotation, 0);
	}

	/**
	 * Returns CSS styles for the title. When titles are rotated they are
	 * rendered using vector graphic techniques and not all styles are
	 * applicable.
	 * <p>
	 * Default:
	 * <pre>
	 * color: #6D869F; fontWeight: bold;
	 * </pre>
	 */
	public <K, V> Map<K, V> getStyle() {
		if (!containsKey(Attrs.style)) {
			setStyle("color: #6D869F; fontWeight: bold;");
		}
		return Generics.cast(getAttr(Attrs.style));
	}
	/**
	 * Sets CSS styles for the title. When titles are rotated they are
	 * rendered using vector graphic techniques and not all styles are
	 * applicable.
	 * <p>
	 * Default:
	 * <pre>
	 * color: #6D869F; fontWeight: bold;
	 * </pre>
	 */
	public void setStyle(String style) {
		setStyle(MapsHelper.parse(new LinkedHashMap<String, String>(), style, ':', ';', '\''));
	}
	
	/**
	 * Sets CSS styles for the title. When titles are rotated they are
	 * rendered using vector graphic techniques and not all styles are
	 * applicable.
	 * <p>
	 * Default:
	 * <pre>
	 * color: #6D869F; fontWeight: bold;
	 * </pre>
	 */
	public <K, V> void setStyle(Map<K, V> style) {
		setAttr(Attrs.style, style, NOT_NULL_VALUE);
	}

	/**
	 * Returns the actual text of the axis title. It can contain basic HTML text
	 * markup like &lt;b&gt;, &lt;i&gt; and spans with style.
	 * <p>
	 * Default: null.
	 */
	public String getText() {
		return getAttr(Attrs.text, null).asString();
	}

	/**
	 * Sets the actual text of the axis title. It can contain basic HTML text
	 * markup like &lt;b&gt;, &lt;i&gt; and spans with style.
	 */
	public void setText(String text) {
		setAttr(Attrs.text, text);
	}

	/**
	 * Returns the x offset.
	 */
	public Number getX() {
		return getAttr(Attrs.x, 0).asNumber();
	}

	/**
	 * Sets the x offset.
	 */
	public void setX(Number x) {
		setAttr(Attrs.x, x, 0);
	}

	/**
	 * Returns the y offset.
	 */
	public Number getY() {
		return getAttr(Attrs.y, 0).asNumber();
	}

	/**
	 * Sets the y offset.
	 */
	public void setY(Number y) {
		setAttr(Attrs.y, y, 0);
	}
}