/** AxisLabels.java.

	Purpose:
		
	Description:
		
	History:
		17:52:02 PM Jan 13, 2014, Created by 

Copyright (C) 2014 Potix Corporation. All Rights Reserved.
 */
package org.zkoss.chart;

import java.util.LinkedHashMap;
import java.util.Map;

import org.zkoss.chart.util.DynamicalAttribute;
import org.zkoss.chart.util.MapsHelper;
import org.zkoss.lang.Generics;

/**
 * The axis labels show the number or category for each tick.
 * 
 * <p> All the options in this class support {@link DynamicalAttribute}.
 * @author jumperchen
 * @author RaymondChao
 */
public class AxisLabels extends Optionable {
	private enum Attrs implements PlotAttribute, DynamicalAttribute {
		align,
		distance,
		enabled,
		format,
		formatter,
		maxStaggerLines,
		overflow,
		rotation,
		staggerLines,
		step,
		style,
		useHTML,
		x,
		y,
		zIndex
	}

	/**
	 * Returns what part of the string the given position is anchored to. Can be
	 * one of <code>left</code>, <code>center</code> or <code>right</code>.
	 * <p> Defaults: <code>center</code>.
	 */
	public String getAlign() {
		return getAttr(Attrs.align, "center").asString();
	}

	/**
	 * Sets what part of the string the given position is anchored to. Can be
	 * one of <code>left</code>, <code>center</code> or <code>right</code>.
	 * <p> Defaults: <code>center</code>.
	 */
	public void setAlign(String align) {
		if (!("center".equals(align) || "left".equals(align) || "right".equals(align)))
			throw new IllegalArgumentException("Unsupported align [" + align + "]");
		setAttr(Attrs.align, align, "center");
	}

	/**
	 * Returns the distance of axis labels for Gauge chart type only.
	 * <p> Default: 15
	 */
	public Number getDistance() {
		return getAttr(Attrs.distance, 15).asNumber();
	}

	/**
	 * Sets the distance of axis labels for Gauge chart type only.
	 */
	public void setDistance(Number distance) {
		setAttr(Attrs.distance, distance, 15);
	}

	/**
	 * Returns whether to enable or disable the axis labels.
	 * <p>
	 * Default: true.
	 */
	public boolean isEnabled() {
		return getAttr(Attrs.enabled, true).asBoolean();
	}

	/**
	 * Sets whether to enable or disable the axis labels.
	 */
	public void setEnabled(boolean enabled) {
		setAttr(Attrs.enabled, enabled);
	}

	/**
	 * Returns a <a href="http://docs.highcharts.com#formatting">format
	 * string</a> for the axis label.
	 * <p>
	 * Default: <code>{value}</code>.
	 */
	public String getFormat() {
		return getAttr(Attrs.format, "{value}").asString();
	}

	/**
	 * Sets a <a href="http://docs.highcharts.com#formatting">format string</a>
	 * for the axis label.
	 */
	public void setFormat(String format) {
		setAttr(Attrs.format, format, "{value}");
	}

	/**
	 * Returns the maximum lines of the to add to automatically avoid
	 * overlapping X labels. When <code>staggerLines</code> is not set,
	 * <code>maxStaggerLines</code> defines how many lines the axis is allowed
	 * to add to automatically avoid overlapping X labels. Set to <code>1</code>
	 * to disable overlap detection.
	 * <p>
	 * Default: 5.
	 */
	public Number getMaxStaggerLines() {
		return getAttr(Attrs.maxStaggerLines, 5).asNumber();
	}

	/**
	 * Sets the maximum lines of the to add to automatically avoid overlapping X
	 * labels. When <code>staggerLines</code> is not set,
	 * <code>maxStaggerLines</code> defines how many lines the axis is allowed
	 * to add to automatically avoid overlapping X labels. Set to <code>1</code>
	 * to disable overlap detection.
	 */
	public void setMaxStaggerLines(Number maxStaggerLines) {
		setAttr(Attrs.maxStaggerLines, maxStaggerLines, 5);
	}

	/**
	 * Returns how to handle overflowing labels on horizontal axis. Can be
	 * undefined or "justify". If "justify", labels will not render outside the
	 * plot area. If there is room to move it, it will be aligned to the edge,
	 * else it will be removed.
	 * <p>
	 * Default: null.
	 */
	public String getOverflow() {
		return getAttr(Attrs.overflow, null).asString();
	}

	/**
	 * Sets how to handle overflowing labels on horizontal axis. Can be
	 * undefined or "justify". If "justify", labels will not render outside the
	 * plot area. If there is room to move it, it will be aligned to the edge,
	 * else it will be removed.
	 * 
	 * @param overflow
	 *            how to handle overflowing labels on horizontal axis
	 */
	public void setOverflow(String overflow) {
		setAttr(Attrs.overflow, overflow);
	}

	/**
	 * Returns rotation of the labels in degrees.
	 * <p>
	 * Default: 0.
	 */
	public Object getRotation() {
		return getAttr(Attrs.rotation, 0).asValue();
	}

	/**
	 * Sets rotation of the labels in degrees.
	 * 
	 * @param rotation
	 *            rotation of the labels in degrees
	 */
	public void setRotation(Number rotation) {
		setAttr(Attrs.rotation, rotation, 0);
	}

	/**
	 * Sets rotation of the labels in degrees or <code>auto</code>.
	 * 
	 * @param rotation
	 *            rotation of the labels in degrees
	 */
	public void setRotation(String rotation) {
		setAttr(Attrs.rotation, rotation);
	}

	/**
	 * Returns the number of lines to spread the labels over to make room or
	 * tighter labels.
	 * <p>
	 * Default: 0
	 * 
	 * @return staggerLines the number of lines to spread the labels
	 */
	public Number getStaggerLines() {
		return getAttr(Attrs.staggerLines, 0).asNumber();
	}

	/**
	 * Sets the number of lines to spread the labels over to make room or
	 * tighter labels.
	 */
	public void setStaggerLines(Number staggerLines) {
		setAttr(Attrs.staggerLines, staggerLines, 0);
	}

	/**
	 * Returns to show only every <em>n</em>'th label on the axis, set the step to
	 * <em>n</em>. Setting the step to 2 shows every other label.
	 * </p>
	 * <p>
	 * By default, on a vertical axis, the step is calculated automatically to
	 * avoid overlap. To prevent this, set it to 1. This usually only happens on
	 * a category axis, and is often a sign that you have chosen the wrong axis
	 * type. Read more at <a
	 * href="http://www.highcharts.com/docs/chart-concepts/axes">Axis docs</a>
	 * => What axis should I use?
	 * </p>
	 * <p>
	 * Default: null.
	 */
	public Number getStep() {
		return getAttr(Attrs.step, null).asNumber();
	}

	/**
	 * Sets to show only every <em>n</em>'th label on the axis, set the step to
	 * <em>n</em>. Setting the step to 2 shows every other label.
	 * </p>
	 * <p>
	 * By default, on a vertical axis, the step is calculated automatically to
	 * avoid overlap. To prevent this, set it to 1. This usually only happens on
	 * a category axis, and is often a sign that you have chosen the wrong axis
	 * type. Read more at <a
	 * href="http://www.highcharts.com/docs/chart-concepts/axes">Axis docs</a>
	 * => What axis should I use?
	 * </p>
	 */
	public void setStep(Number step) {
		setAttr(Attrs.step, step);
	}

	/**
	 * Returns CSS styles for the label. When labels are rotated they are
	 * rendered using vector graphic techniques and not all styles are
	 * applicable.
	 * <p>
	 * Default:
	 * <pre>
	 * color: #6D869F; fontWeight: bold;
	 * </pre>
	 * 
	 */
	public <K, V> Map<K, V> getStyle() {
		if (!containsKey(Attrs.style)) {
			setStyle("color: #6D869F; fontWeight: bold;");
		}
		return Generics.cast(getAttr(Attrs.style));
	}

	/**
	 * Sets CSS styles for the label. When labels are rotated they are rendered
	 * using vector graphic techniques and not all styles are applicable.
	 * <p>
	 * Default:
	 * <pre>
	 * color: #6D869F; fontWeight: bold;
	 * </pre>
	 * @see #setStyle(Map)
	 */
	public void setStyle(String style) {
		setStyle(MapsHelper.parse(new LinkedHashMap<String, String>(), style, ':', ';', '\''));
	}

	/**
	 * Sets CSS styles for the label. When labels are rotated they are rendered
	 * using vector graphic techniques and not all styles are applicable.
	 * <p>
	 * Default:
	 * <pre>
	 * color: #6D869F; fontWeight: bold;
	 * </pre>
	 * @see #setStyle(String)
	 */
	public <K, V> void setStyle(Map<K, V> style) {
		setAttr(Attrs.style, style, NOT_NULL_VALUE);
	}

	/**
	 * Returns whether to <a
	 * href="http://docs.highcharts.com/#formatting$html">use HTML</a> to render
	 * the labels.
	 * <p>
	 * Default: false.
	 */
	public boolean isUseHTML() {
		return getAttr(Attrs.useHTML, false).asBoolean();
	}

	/**
	 * Sets whether to <a href="http://docs.highcharts.com/#formatting$html">use
	 * HTML</a> to render the labels.
	 */
	public void setUseHTML(boolean useHTML) {
		setAttr(Attrs.useHTML, useHTML);
	}

	/**
	 * Returns the x position offset of the label relative to the tick position
	 * on the axis.
	 * <p> Default: 0
	 */
	public Number getX() {
		return getAttr(Attrs.x, 0).asNumber();
	}

	/**
	 * Sets the x position offset of the label relative to the tick position on
	 * the axis.
	 * <p> Default: 0
	 */
	public void setX(Number x) {
		setAttr(Attrs.x, x, 0);
	}

	/**
	 * Returns the y position offset of the label relative to the tick position
	 * on the axis.
	 * <p> Default: 0
	 */
	public Number getY() {
		return getAttr(Attrs.y, 0).asNumber();
	}

	/**
	 * Sets the y position offset of the label relative to the tick position on
	 * the axis.
	 * 
	 * <p> Default: 0
	 */
	public void setY(Number y) {
		// we don't use 0 here to solve HC JS inconsistent issue
		setAttr(Attrs.y, y, NOT_NULL_VALUE);
	}

	/**
	 * Returns the Z index for the axis labels.
	 * <p>
	 * Default: 7.
	 */
	public Number getZIndex() {
		return getAttr(Attrs.zIndex, 7).asNumber();
	}

	/**
	 * Sets the Z index for the axis labels.
	 */
	public void setZIndex(Number zIndex) {
		setAttr(Attrs.zIndex, zIndex, 7);
	}

}