/** Axis.java.

	Purpose:
		
	Description:
		
	History:
		17:51:32 PM Jan 13, 2014, Created by RaymondChao

Copyright (C) 2014 Potix Corporation. All Rights Reserved.
 */
package org.zkoss.chart;

import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

import org.zkoss.chart.OptionDataEvent.EventType;
import org.zkoss.chart.util.DeferredCall;
import org.zkoss.chart.util.DynamicalAttribute;
import org.zkoss.chart.util.JSFunction;
import org.zkoss.json.JSONObject;
import org.zkoss.lang.Generics;
import org.zkoss.lang.Objects;

/**
 * A chart can have from 0 axes (pie chart) to multiples. In a normal, single
 * series cartesian chart, there is one X axis and one Y axis. The X axis or
 * axes are referenced by {@link Charts#getXAxis(int)}. If there is only one
 * axis, it can be referenced through {@link Charts#getXAxis()}, and
 * multiple axes have increasing indices. The same pattern goes for Y axes.
 * <p>
 * All the options in this class support {@link DynamicalAttribute}.
 * 
 * @author jumperchen
 * @author RaymondChao
 */
public class Axis extends Optionable {
	private enum DAttrs implements PlotAttribute, DynamicalAttribute {
		allowDecimals,
		alternateGridColor,
		categories,
		dateTimeLabelFormats,
		endOnTick,
		events,
		gridLineColor,
		gridLineDashStyle,
		gridLineInterpolation,
		gridLineWidth,
		id,
		labels,
		lineColor,
		lineWidth,
		linkedTo,
		max,
		maxPadding,
		// maxZoom, Deprecated
		min,
		minPadding,
		minRange,
		minTickInterval,
		minorGridLineColor,
		minorGridLineDashStyle,
		minorGridLineWidth,
		minorTickColor,
		minorTickInterval,
		minorTickLength,
		minorTickPosition,
		minorTickWidth,
		offset,
		opposite,
		pane,
		reversed,
		showEmpty,
		showFirstLabel,
		showLastLabel,
		startOfWeek,
		startOnTick,
		tickColor,
		tickInterval,
		tickLength,
		tickPixelInterval,
		tickPosition,
		tickPositioner,
		tickPositions,
		tickWidth,
		tickmarkPlacement,
		title,
		type
	}

	private enum Attrs implements PlotAttribute {
		plotBands, plotLines,

		// internal use
		plotband,
		plotLine
	}

	// TODO correct javadoc

	/**
	 * Sets the minimum and maximum of the axes after render time. If the
	 * startOnTick and endOnTick options are true, the minimum and maximum
	 * values are rounded off to the nearest tick. To prevent this, these
	 * options can be set to false before calling setExtremes. Also, setExtremes
	 * will not allow a range lower than the minRange option, which by default
	 * is the range of five points.
	 * 
	 * @param min
	 * @param max
	 */
	public void setExtremes(final Number min, final Number max,
			final boolean redraw, final Animation animation) {
		fireEvent(EventType.INVOKE, "extremes", null, new DeferredCall() {
			public void execute(JSFunction func) {
				func.callFunction("setExtremes", min, max, redraw, animation);
			}
		});
	}

	public void setExtremes(Number min, Number max) {
		setExtremes(min, max, true, new Animation());
	}

	/**
	 * Returns whether to allow decimals in this axis' ticks. When counting
	 * integers, like persons or hits on a web page, decimals must be avoided in
	 * the axis tick labels.
	 * <p>
	 * Default: true.
	 */
	public boolean isAllowDecimals() {
		return getAttr(DAttrs.allowDecimals, true).asBoolean();
	}

	/**
	 * Sets whether to allow decimals in this axis' ticks. When counting
	 * integers, like persons or hits on a web page, decimals must be avoided in
	 * the axis tick labels.
	 */
	public void setAllowDecimals(boolean allowDecimals) {
		setAttr(DAttrs.allowDecimals, allowDecimals);
	}
	/**
	 * Returns when using an alternate grid color, a band is painted across the
	 * plot area between every other grid line.
	 * <p>
	 * Default: null.
	 */
	public Color getAlternateGridColor() {
		return (Color) getAttr(DAttrs.alternateGridColor, null).asValue();
	}

	/**
	 * Sets when using an alternate grid color, a band is painted across the
	 * plot area between every other grid line.
	 */
	public void setAlternateGridColor(Color color) {
		setAttr(DAttrs.alternateGridColor, color);
	}

	/**
	 * Sets when using an alternate grid color, a band is painted across the
	 * plot area between every other grid line.
	 */
	public void setAlternateGridColor(String color) {
		setAlternateGridColor(new Color(color));
	}

	/**
	 * Sets when using an alternate grid color, a band is painted across the
	 * plot area between every other grid line.
	 */
	public void setAlternateGridColor(LinearGradient color) {
		setAlternateGridColor(new Color(color));
	}

	/**
	 * Sets when using an alternate grid color, a band is painted across the
	 * plot area between every other grid line.
	 */
	public void setAlternateGridColor(RadialGradient color) {
		setAlternateGridColor(new Color(color));
	}

	/**
	 * Returns the categories in a list.
	 * <p>
	 * Default: null
	 */
	public <T> List<T> getCategories() {
		return Generics.cast(getAttr(DAttrs.categories, null).asValue());
	}

	/**
	 * Sets categories are present for the Axis, names are used instead of
	 * numbers for that axis.
	 * <p>
	 * Defaults: null
	 * 
	 * @see #setCategories(String...)
	 */
	public <T> void setCategories(List<T> categories) {
		setAttr(DAttrs.categories, categories);
	}

	/**
	 * Sets categories are present for the Axis, names are used instead of
	 * numbers for that axis.
	 * <p>
	 * Defaults: null
	 * 
	 * @see #setCategories(List)
	 */
	public void setCategories(String... categories) {
		setAttr(DAttrs.categories, categories);
	}

	/**
	 * Returns for a datetime axis, the scale will automatically adjust to the
	 * appropriate unit. This member gives the default string representations
	 * used for each unit. For an overview of the replacement codes, see
	 * dateFormat. Defaults to:
	 * 
	 * <pre>
	 * {
	 * 	millisecond: '%H:%M:%S.%L',
	 * 	second: '%H:%M:%S',
	 * 	minute: '%H:%M',
	 * 	hour: '%H:%M',
	 * 	day: '%e. %b',
	 * 	week: '%e. %b',
	 * 	month: '%b \'%y',
	 * 	year: '%Y'
	 * }
	 * </pre>
	 */
	public DateTimeLabelFormats getDateTimeLabelFormats() {
		DateTimeLabelFormats dateTimeLabelFormats = (DateTimeLabelFormats) getAttr(DAttrs.dateTimeLabelFormats);
		if (dateTimeLabelFormats == null) {
			dateTimeLabelFormats = new DateTimeLabelFormats();
			setDateTimeLabelFormats(dateTimeLabelFormats);
		}
		return dateTimeLabelFormats;
	}

	/**
	 * Sets for a datetime axis, the scale will automatically adjust to the
	 * appropriate unit. This member gives the default string representations
	 * used for each unit. For an overview of the replacement codes, see
	 * dateFormat. Defaults to:
	 * 
	 * <pre>
	 * {
	 * 	millisecond: '%H:%M:%S.%L',
	 * 	second: '%H:%M:%S',
	 * 	minute: '%H:%M',
	 * 	hour: '%H:%M',
	 * 	day: '%e. %b',
	 * 	week: '%e. %b',
	 * 	month: '%b \'%y',
	 * 	year: '%Y'
	 * }
	 * </pre>
	 */
	public void setDateTimeLabelFormats(
			DateTimeLabelFormats dateTimeLabelFormats) {
		setAttr(DAttrs.dateTimeLabelFormats, dateTimeLabelFormats);
	}

	/**
	 * Returns whether to force the axis to end on a tick. Use this option with
	 * the <code>maxPadding</code> option to control the axis end.
	 * <p>
	 * Default: false.
	 * 
	 * @return endOnTick whether to force the axis to end on a tick
	 */
	public boolean isEndOnTick() {
		return getAttr(DAttrs.endOnTick, false).asBoolean();
	}

	/**
	 * Sets whether to force the axis to end on a tick. Use this option with the
	 * <code>maxPadding</code> option to control the axis end.
	 * 
	 * @param endOnTick
	 *            whether to force the axis to end on a tick
	 */
	public void setEndOnTick(boolean endOnTick) {
		setAttr(DAttrs.endOnTick, endOnTick);
	}

	/**
	 * Returns color of the grid lines extending the ticks across the plot area.
	 * <p>
	 * Default: #C0C0C0.
	 */
	public Color getGridLineColor() {
		if (!containsKey(DAttrs.gridLineColor)) {
			setGridLineColor("#C0C0C0");
		}
		return (Color) getAttr(DAttrs.gridLineColor);
	}

	/**
	 * Sets color of the grid lines extending the ticks across the plot area.
	 */
	public void setGridLineColor(Color color) {
		setAttr(DAttrs.gridLineColor, color, NOT_NULL_VALUE);
	}

	/**
	 * Sets color of the grid lines extending the ticks across the plot area.
	 */
	public void setGridLineColor(String color) {
		setGridLineColor(new Color(color));
	}

	/**
	 * Sets color of the grid lines extending the ticks across the plot area.
	 */
	public void setGridLineColor(LinearGradient color) {
		setGridLineColor(new Color(color));
	}

	/**
	 * Sets color of the grid lines extending the ticks across the plot area.
	 */
	public void setGridLineColor(RadialGradient color) {
		setGridLineColor(new Color(color));
	}

	/**
	 * Returns the dash or dot style of the grid lines. For possible values,
	 * <code>Solid</code>, <code>ShortDash</code>, <code>ShortDot</code>,
	 * <code>ShortDashDot</code>, <code>ShortDashDotDot</code>, <code>Dot</code>
	 * , <code>Dash</code>, <code>LongDash</code>, <code>DashDot</code>,
	 * <code>LongDashDot</code>, and <code>LongDashDotDot</code>.
	 * <p>
	 * Default: "Solid".
	 */
	public String getGridLineDashStyle() {
		return getAttr(DAttrs.gridLineDashStyle, "Solid").asString();
	}

	/**
	 * Sets the dash or dot style of the grid lines. For possible values,
	 * <code>Solid</code>, <code>ShortDash</code>, <code>ShortDot</code>,
	 * <code>ShortDashDot</code>, <code>ShortDashDotDot</code>, <code>Dot</code>
	 * , <code>Dash</code>, <code>LongDash</code>, <code>DashDot</code>,
	 * <code>LongDashDot</code>, and <code>LongDashDotDot</code>.
	 */
	public void setGridLineDashStyle(String gridLineDashStyle) {
		if (!("Solid".equalsIgnoreCase(gridLineDashStyle)
				|| "ShortDash".equalsIgnoreCase(gridLineDashStyle)
				|| "ShortDot".equalsIgnoreCase(gridLineDashStyle)
				|| "ShortDashDot".equalsIgnoreCase(gridLineDashStyle)
				|| "ShortDashDotDot".equalsIgnoreCase(gridLineDashStyle)
				|| "Dot".equalsIgnoreCase(gridLineDashStyle)
				|| "Dash".equalsIgnoreCase(gridLineDashStyle)
				|| "LongDash".equalsIgnoreCase(gridLineDashStyle)
				|| "DashDot".equalsIgnoreCase(gridLineDashStyle)
				|| "LongDashDot".equalsIgnoreCase(gridLineDashStyle) || "LongDashDotDot"
					.equalsIgnoreCase(gridLineDashStyle)))
			throw new IllegalArgumentException("Unsupported style: ["
					+ gridLineDashStyle + "]");
		setAttr(DAttrs.gridLineDashStyle, gridLineDashStyle, "Solid");
	}

	public String getGridLineInterpolation() {
		return getAttr(DAttrs.gridLineInterpolation, null).asString();
	}

	public void setGridLineInterpolation(String gridLineInterpolation) {
		setAttr(DAttrs.gridLineInterpolation, gridLineInterpolation);
	}

	/**
	 * Returns the width of the grid lines extending the ticks across the plot
	 * area.
	 * <p>
	 * Default: 0.
	 */
	public Number getGridLineWidth() {
		return getAttr(DAttrs.gridLineWidth, 0).asNumber();
	}

	/**
	 * Sets the width of the grid lines extending the ticks across the plot
	 * area.
	 */
	public void setGridLineWidth(Number gridLineWidth) {
		setAttr(DAttrs.gridLineWidth, gridLineWidth, 0);
	}

	/**
	 * Returns an id for the axis.
	 * <p>
	 * Default: null.
	 */
	public String getId() {
		return getAttr(DAttrs.id, null).asString();
	}

	/**
	 * Sets an id for the axis.
	 */
	public void setId(String id) {
		setAttr(DAttrs.id, id);
	}

	/**
	 * Returns the axis labels show the number or category for each tick.
	 */
	public AxisLabels getLabels() {
		AxisLabels labels = (AxisLabels) getAttr(DAttrs.labels);
		if (labels == null) {
			labels = new AxisLabels();
			setLabels(labels);
		}
		return labels;
	}

	/**
	 * Sets the axis labels show the number or category for each tick.
	 * 
	 * @param labels
	 *            the axis labels show the number or category for each tick
	 */
	public void setLabels(AxisLabels labels) {
		setAttr(DAttrs.labels, labels);
	}

	/**
	 * Returns the color of the line marking the axis itself.
	 * <p>
	 * Default: "#C0D0E0".
	 */
	public Color getLineColor() {
		if (!containsKey(DAttrs.lineColor)) {
			setLineColor("#C0D0E0");
		}
		return (Color) getAttr(DAttrs.lineColor);
	}

	/**
	 * Sets the color of the line marking the axis itself.
	 */
	public void setLineColor(Color color) {
		setAttr(DAttrs.lineColor, color, NOT_NULL_VALUE);
	}

	/**
	 * Sets the color of the line marking the axis itself.
	 */
	public void setLineColor(String color) {
		setLineColor(new Color(color));
	}

	/**
	 * Sets the color of the line marking the axis itself.
	 */
	public void setLineColor(LinearGradient color) {
		setLineColor(new Color(color));
	}

	/**
	 * Sets the color of the line marking the axis itself.
	 */
	public void setLineColor(RadialGradient color) {
		setLineColor(new Color(color));
	}

	/**
	 * Returns the width of the line marking the axis itself.
	 * <p>
	 * Default: 1.
	 */
	public Number getLineWidth() {
		return getAttr(DAttrs.lineWidth, 1).asNumber();
	}

	/**
	 * Sets the width of the line marking the axis itself.
	 */
	public void setLineWidth(Number lineWidth) {
		setAttr(DAttrs.lineWidth, lineWidth, 1);
	}

	/**
	 * Returns index of another axis that this axis is linked to. When an axis
	 * is linked to a master axis, it will take the same extremes as the master,
	 * but as assigned by min or max or by setExtremes. It can be used to show
	 * additional info, or to ease reading the chart by duplicating the scales.
	 * <p>
	 * Default: null.
	 */
	public Number getLinkedTo() {
		return getAttr(DAttrs.linkedTo, null).asNumber();
	}

	/**
	 * Sets index of another axis that this axis is linked to. When an axis is
	 * linked to a master axis, it will take the same extremes as the master,
	 * but as assigned by min or max or by setExtremes. It can be used to show
	 * additional info, or to ease reading the chart by duplicating the scales.
	 */
	public void setLinkedTo(Number linkedTo) {
		setAttr(DAttrs.linkedTo, linkedTo);
	}

	/**
	 * Returns the maximum value of the axis. If <code>-1</code>, the max value
	 * is automatically calculated. If the <code>endOnTick</code> option is
	 * true, the <code>max</code> value might be rounded up. The actual maximum
	 * value is also influenced by {@link Chart#isAlignTicks()}.
	 * <p>
	 * Default: null
	 */
	public Number getMax() {
		return getAttr(DAttrs.max, null).asNumber();
	}

	/**
	 * Sets the maximum value of the axis. If <code>-1</code>, the max value is
	 * automatically calculated. If the <code>endOnTick</code> option is true,
	 * the <code>max</code> value might be rounded up. The actual maximum value
	 * is also influenced by {@link Chart#isAlignTicks()}.
	 */
	public void setMax(Number max) {
		setAttr(DAttrs.max, max);
	}

	/**
	 * Returns padding of the max value relative to the length of the axis. A
	 * padding of 0.05 will make a 100px axis 5px longer. This is useful when
	 * you don't want the highest data value to appear on the edge of the plot
	 * area. When the {@link #setMax(Number)} option is set or a max extreme is
	 * set using {@link #setExtremes(Number, Number)}, the maxPadding will be
	 * ignored.
	 * <p>
	 * Default: 0.01.
	 */
	public Number getMaxPadding() {
		return getAttr(DAttrs.maxPadding, 0.01).asNumber();
	}

	/**
	 * Sets padding of the max value relative to the length of the axis. A
	 * padding of 0.05 will make a 100px axis 5px longer. This is useful when
	 * you don't want the highest data value to appear on the edge of the plot
	 * area. When the {@link #setMax(Number)} option is set or a max extreme is
	 * set using {@link #setExtremes(Number, Number)}, the maxPadding will be
	 * ignored.
	 */
	public void setMaxPadding(Number maxPadding) {
		setAttr(DAttrs.maxPadding, maxPadding, 0.01);
	}

	/**
	 * Returns the minimum value of the axis. If <code>null</code> the min value
	 * is automatically calculated. If the {@link #setStartOnTick(boolean)}
	 * option is true, the <code>min</code> value might be rounded down.
	 * <p>
	 * Default: null.
	 */
	public Number getMin() {
		return getAttr(DAttrs.min, null).asNumber();
	}

	/**
	 * Sets the minimum value of the axis. If <code>null</code> the min value is
	 * automatically calculated. If the {@link #setStartOnTick(boolean)} option
	 * is true, the <code>min</code> value might be rounded down.
	 */
	public void setMin(Number min) {
		setAttr(DAttrs.min, min);
	}

	/**
	 * Returns padding of the min value relative to the length of the axis. A
	 * padding of 0.05 will make a 100px axis 5px longer. This is useful when
	 * you don't want the lowest data value to appear on the edge of the plot
	 * area. When the {@link #setMin(Number)} option is set or a min extreme is
	 * set using {@link #setExtremes(Number, Number)}, the minPadding will be
	 * ignored.
	 * <p>
	 * Default: 0.01.
	 */
	public Number getMinPadding() {
		return getAttr(DAttrs.minPadding, 0.01).asNumber();
	}

	/**
	 * Sets padding of the min value relative to the length of the axis. A
	 * padding of 0.05 will make a 100px axis 5px longer. This is useful when
	 * you don't want the lowest data value to appear on the edge of the plot
	 * area. When the {@link #setMin(Number)} option is set or a min extreme is
	 * set using {@link #setExtremes(Number, Number)}, the minPadding will be
	 * ignored.
	 */
	public void setMinPadding(Number minPadding) {
		setAttr(DAttrs.minPadding, minPadding, 0.01);
	}

	/**
	 * Returns the minimum range to display on this axis. The entire axis will
	 * not be allowed to span over a smaller interval than this. For example,
	 * for a datetime axis the main unit is milliseconds. If minRange is set to
	 * 3600000, you can't zoom in more than to one hour. </p>
	 * <p>
	 * The default minRange for the x axis is five times the smallest interval
	 * between any of the data points.
	 * </p>
	 * <p>
	 * On a logarithmic axis, the unit for the minimum range is the power. So a
	 * minRange of 1 means that the axis can be zoomed to 10-100, 100-1000,
	 * 1000-10000 etc.
	 * </p>
	 * <p>
	 * Default: null
	 */
	public Number getMinRange() {
		return getAttr(DAttrs.minRange, null).asNumber();
	}

	/**
	 * Sets the minimum range to display on this axis. The entire axis will not
	 * be allowed to span over a smaller interval than this. For example, for a
	 * datetime axis the main unit is milliseconds. If minRange is set to
	 * 3600000, you can't zoom in more than to one hour. </p>
	 * <p>
	 * The default minRange for the x axis is five times the smallest interval
	 * between any of the data points.
	 * </p>
	 * <p>
	 * On a logarithmic axis, the unit for the minimum range is the power. So a
	 * minRange of 1 means that the axis can be zoomed to 10-100, 100-1000,
	 * 1000-10000 etc.
	 * </p>
	 * <p>
	 * Default: null.
	 */
	public void setMinRange(Number minRange) {
		setAttr(DAttrs.minRange, minRange);
	}

	/**
	 * Returns the minimum tick interval allowed in axis values. For example on
	 * zooming in on an axis with daily data, this can be used to prevent the
	 * axis from showing hours.
	 * <p>
	 * Default: null
	 */
	public Number getMinTickInterval() {
		return getAttr(DAttrs.minTickInterval, null).asNumber();
	}

	/**
	 * Sets the minimum tick interval allowed in axis values. For example on
	 * zooming in on an axis with daily data, this can be used to prevent the
	 * axis from showing hours.
	 * <p>
	 * Default: null
	 */
	public void setMinTickInterval(Number minTickInterval) {
		setAttr(DAttrs.minTickInterval, minTickInterval);
	}

	/**
	 * Returns color of the minor, secondary grid lines.
	 * <p>
	 * Default: "#E0E0E0".
	 */
	public Color getMinorGridLineColor() {
		if (!containsKey(DAttrs.minorGridLineColor)) {
			setMinorGridLineColor("#E0E0E0");
		}
		return (Color) getAttr(DAttrs.minorGridLineColor);
	}

	/**
	 * Sets color of the minor, secondary grid lines.
	 */
	public void setMinorGridLineColor(Color color) {
		setAttr(DAttrs.minorGridLineColor, color, NOT_NULL_VALUE);
	}

	/**
	 * Sets color of the minor, secondary grid lines.
	 */
	public void setMinorGridLineColor(String color) {
		setMinorGridLineColor(new Color(color));
	}

	/**
	 * Sets color of the minor, secondary grid lines.
	 */
	public void setMinorGridLineColor(LinearGradient color) {
		setMinorGridLineColor(new Color(color));
	}

	/**
	 * Sets color of the minor, secondary grid lines.
	 */
	public void setMinorGridLineColor(RadialGradient color) {
		setMinorGridLineColor(new Color(color));
	}

	/**
	 * Returns the dash or dot style of the minor grid lines. For possible
	 * values, <code>Solid</code>, <code>ShortDash</code>, <code>ShortDot</code>
	 * , <code>ShortDashDot</code>, <code>ShortDashDotDot</code>,
	 * <code>Dot</code> , <code>Dash</code>, <code>LongDash</code>,
	 * <code>DashDot</code>, <code>LongDashDot</code>, and
	 * <code>LongDashDotDot</code>.
	 * <p>
	 * Default: "Solid".
	 */
	public String getMinorGridLineDashStyle() {
		return getAttr(DAttrs.minorGridLineDashStyle, "Solid").asString();
	}

	/**
	 * Sets the dash or dot style of the minor grid lines. For possible values,
	 * <code>Solid</code>, <code>ShortDash</code>, <code>ShortDot</code>,
	 * <code>ShortDashDot</code>, <code>ShortDashDotDot</code>, <code>Dot</code>
	 * , <code>Dash</code>, <code>LongDash</code>, <code>DashDot</code>,
	 * <code>LongDashDot</code>, and <code>LongDashDotDot</code>.
	 */
	public void setMinorGridLineDashStyle(String minorGridLineDashStyle) {
		if (!("Solid".equalsIgnoreCase(minorGridLineDashStyle)
				|| "ShortDash".equalsIgnoreCase(minorGridLineDashStyle)
				|| "ShortDot".equalsIgnoreCase(minorGridLineDashStyle)
				|| "ShortDashDot".equalsIgnoreCase(minorGridLineDashStyle)
				|| "ShortDashDotDot".equalsIgnoreCase(minorGridLineDashStyle)
				|| "Dot".equalsIgnoreCase(minorGridLineDashStyle)
				|| "Dash".equalsIgnoreCase(minorGridLineDashStyle)
				|| "LongDash".equalsIgnoreCase(minorGridLineDashStyle)
				|| "DashDot".equalsIgnoreCase(minorGridLineDashStyle)
				|| "LongDashDot".equalsIgnoreCase(minorGridLineDashStyle) || "LongDashDotDot"
					.equalsIgnoreCase(minorGridLineDashStyle)))
			throw new IllegalArgumentException("Unsupported style: ["
					+ minorGridLineDashStyle + "]");
		setAttr(DAttrs.minorGridLineDashStyle, minorGridLineDashStyle, "Solid");
	}

	/**
	 * Returns width of the minor, secondary grid lines.
	 * <p>
	 * Default: 1.
	 */
	public Number getMinorGridLineWidth() {
		return getAttr(DAttrs.minorGridLineWidth, 1).asNumber();
	}

	/**
	 * Sets width of the minor, secondary grid lines.
	 */
	public void setMinorGridLineWidth(Number minorGridLineWidth) {
		setAttr(DAttrs.minorGridLineWidth, minorGridLineWidth, 1);
	}

	/**
	 * Returns color for the minor tick marks.
	 * <p>
	 * Default: "#A0A0A0".
	 */
	public Color getMinorTickColor() {
		if (!containsKey(DAttrs.minorTickColor)) {
			setMinorTickColor("#A0A0A0");
		}
		return (Color) getAttr(DAttrs.minorTickColor);
	}

	/**
	 * Sets color for the minor tick marks.
	 */
	public void setMinorTickColor(Color color) {
		setAttr(DAttrs.minorTickColor, color, NOT_NULL_VALUE);
	}

	/**
	 * Sets color for the minor tick marks.
	 */
	public void setMinorTickColor(String color) {
		setMinorTickColor(new Color(color));
	}

	/**
	 * Sets color for the minor tick marks.
	 */
	public void setMinorTickColor(LinearGradient color) {
		setMinorTickColor(new Color(color));
	}

	/**
	 * Sets color for the minor tick marks.
	 */
	public void setMinorTickColor(RadialGradient color) {
		setMinorTickColor(new Color(color));
	}

	/**
	 * Returns the tick interval in scale units for the minor ticks. On a linear
	 * axis, if <code>"auto"</code>, the minor tick interval is calculated as a
	 * fifth of the tickInterval. If <code>null</code>, minor ticks are not
	 * shown. </p>
	 * <p>
	 * On logarithmic axes, the unit is the power of the value. For example,
	 * setting the minorTickInterval to 1 puts one tick on each of 0.1, 1, 10,
	 * 100 etc. Setting the minorTickInterval to 0.1 produces 9 ticks between 1
	 * and 10, 10 and 100 etc. A minorTickInterval of "auto" on a log axis
	 * results in a best guess, attempting to enter approximately 5 minor ticks
	 * between each major tick.
	 * </p>
	 * <p>
	 * On axes using {@link #setCategories(List)}, minor ticks are not
	 * supported.
	 * </p>
	 * <p>
	 * Default: null.
	 */
	public Number getMinorTickInterval() {
		return getAttr(DAttrs.minorTickInterval, null).asNumber();
	}

	/**
	 * Sets tick interval in scale units for the minor ticks. On a linear axis,
	 * if <code>"auto"</code>, the minor tick interval is calculated as a fifth
	 * of the tickInterval. If <code>null</code>, minor ticks are not shown.
	 * </p>
	 * <p>
	 * On logarithmic axes, the unit is the power of the value. For example,
	 * setting the minorTickInterval to 1 puts one tick on each of 0.1, 1, 10,
	 * 100 etc. Setting the minorTickInterval to 0.1 produces 9 ticks between 1
	 * and 10, 10 and 100 etc. A minorTickInterval of "auto" on a log axis
	 * results in a best guess, attempting to enter approximately 5 minor ticks
	 * between each major tick.
	 * </p>
	 * <p>
	 * On axes using {@link #setCategories(List)}, minor ticks are not
	 * supported.
	 * </p>
	 */
	public void setMinorTickInterval(Number minorTickInterval) {
		setAttr(DAttrs.minorTickInterval, minorTickInterval);
	}

	/**
	 * Sets tick interval in scale units for the minor ticks. On a linear axis,
	 * if <code>"auto"</code>, the minor tick interval is calculated as a fifth
	 * of the tickInterval. If <code>null</code>, minor ticks are not shown.
	 * </p>
	 * <p>
	 * On logarithmic axes, the unit is the power of the value. For example,
	 * setting the minorTickInterval to 1 puts one tick on each of 0.1, 1, 10,
	 * 100 etc. Setting the minorTickInterval to 0.1 produces 9 ticks between 1
	 * and 10, 10 and 100 etc. A minorTickInterval of "auto" on a log axis
	 * results in a best guess, attempting to enter approximately 5 minor ticks
	 * between each major tick.
	 * </p>
	 * <p>
	 * On axes using {@link #setCategories(List)}, minor ticks are not
	 * supported.
	 * </p>
	 */
	public void setMinorTickInterval(String minorTickInterval) {
		setAttr(DAttrs.minorTickInterval, minorTickInterval);
	}

	/**
	 * Returns the pixel length of the minor tick marks.
	 * <p>
	 * Default: 2.
	 * 
	 * @return minorTickLength the pixel length of the minor tick marks
	 */
	public Number getMinorTickLength() {
		return getAttr(DAttrs.minorTickLength, 2).asNumber();
	}

	/**
	 * Sets the pixel length of the minor tick marks.
	 */
	public void setMinorTickLength(Number minorTickLength) {
		setAttr(DAttrs.minorTickLength, minorTickLength, 2);
	}

	/**
	 * Returns the position of the minor tick marks relative to the axis line.
	 * Can be one of <code>inside</code> and <code>outside</code>.
	 * <p>
	 * Default: "outside".
	 * 
	 * @return minorTickPosition the position of the minor tick marks relative
	 *         to the axis line
	 */
	public String getMinorTickPosition() {
		return getAttr(DAttrs.minorTickPosition, "outside").asString();
	}

	/**
	 * Sets the position of the minor tick marks relative to the axis line. Can
	 * be one of <code>inside</code> and <code>outside</code>.
	 * 
	 * @param minorTickPosition
	 *            the position of the minor tick marks relative to the axis line
	 */
	public void setMinorTickPosition(String minorTickPosition) {
		if (!("inside".equals(minorTickPosition) || "outside"
				.equals(minorTickPosition)))
			throw new IllegalArgumentException("Unsupported position: ["
					+ minorTickPosition + "]");
		setAttr(DAttrs.minorTickPosition, minorTickPosition, NOT_NULL_VALUE);
	}

	/**
	 * Returns the pixel width of the minor tick mark.
	 * <p>
	 * Default: 0.
	 */
	public Number getMinorTickWidth() {
		return getAttr(DAttrs.minorTickWidth, 0).asNumber();
	}

	/**
	 * Sets the pixel width of the minor tick mark.
	 */
	public void setMinorTickWidth(Number minorTickWidth) {
		setAttr(DAttrs.minorTickWidth, minorTickWidth, 0);
	}

	/**
	 * Returns the distance in pixels from the plot area to the axis line. A
	 * positive offset moves the axis with it's line, labels and ticks away from
	 * the plot area. This is typically used when two or more axes are displayed
	 * on the same side of the plot.
	 * <p>
	 * Default: 0.
	 */
	public Number getOffset() {
		return getAttr(DAttrs.offset, 0).asNumber();
	}

	/**
	 * Sets the distance in pixels from the plot area to the axis line. A
	 * positive offset moves the axis with it's line, labels and ticks away from
	 * the plot area. This is typically used when two or more axes are displayed
	 * on the same side of the plot.
	 */
	public void setOffset(Number offset) {
		setAttr(DAttrs.offset, offset, 0);
	}

	/**
	 * Returns whether to display the axis on the opposite side of the normal.
	 * The normal is on the left side for vertical axes and bottom for
	 * horizontal, so the opposite sides will be right and top respectively.
	 * This is typically used with dual or multiple axes.
	 * <p>
	 * Default: false.
	 */
	public boolean isOpposite() {
		return getAttr(DAttrs.opposite, false).asBoolean();
	}

	/**
	 * Sets whether to display the axis on the opposite side of the normal. The
	 * normal is on the left side for vertical axes and bottom for horizontal,
	 * so the opposite sides will be right and top respectively. This is
	 * typically used with dual or multiple axes.
	 */
	public void setOpposite(boolean opposite) {
		setAttr(DAttrs.opposite, opposite);
	}

	/**
	 * Returns the index related to the pane
	 * <p>
	 * This option is applied to Gauge chart type only.
	 * <p>
	 * Default: 0
	 */
	public Number getPane() {
		return getAttr(DAttrs.pane, 0).asNumber();
	}

	/**
	 * Sets the axis to link to the index of the pane.
	 * <p>
	 * This option is applied to Gauge chart type only.
	 */
	public void setPane(Number pane) {
		setAttr(DAttrs.pane, pane, 0);
	}

	/**
	 * Returns a list of colored bands stretching across the plot area marking
	 * an interval on the axis.
	 * <p>
	 * In a gauge, a plot band on the Y axis (value axis) will stretch along the
	 * perimiter of the gauge.
	 * </p>
	 * <p>
	 * Default: null.
	 */
	public List<PlotBand> getPlotBands() {
		return Generics.cast(getAttr(Attrs.plotBands, null).asValue());
	}

	/**
	 * Sets a list of colored bands stretching across the plot area marking an
	 * interval on the axis.
	 * <p>
	 * In a gauge, a plot band on the Y axis (value axis) will stretch along the
	 * perimiter of the gauge.
	 * </p>
	 * <p>
	 * Default: null.
	 */
	public void setPlotBands(List<PlotBand> plotBands) {
		List<PlotBand> oldPlotBands = getPlotBands();
		if (oldPlotBands != null) {
			List<PlotBand> copy = new CopyOnWriteArrayList<PlotBand>(
					oldPlotBands);
			for (PlotBand plotBand : copy) {
				removePlotBand(plotBand);
			}
		}
		for (PlotBand plotBand : plotBands)
			addPlotBand(plotBand);
	}

	/**
	 * Adds a plot band after render time.
	 * 
	 * @param plotBand
	 */
	public void addPlotBand(final PlotBand plotBand) {
		List<PlotBand> plotBands = getPlotBands();
		if (plotBands == null) {
			plotBands = new LinkedList<PlotBand>();
		}
		plotBand.addOptionDataListener(this);
		plotBands.add(plotBand);
		setAttr(Attrs.plotBands, plotBands);

		fireEvent(EventType.ADDED, Attrs.plotband.toString(), plotBand,
				new DeferredCall() {
					public void execute(JSFunction func) {
						func.callFunction("addPlotBand", plotBand);
					}
				});
	}

	/**
	 * Removes a plot band with given id after render time.
	 */
	public void removePlotBand(final String id) {
		List<PlotBand> plotBands = getPlotBands();
		if (plotBands == null)
			return; // nothing to do

		for (PlotBand plotBand : plotBands) {
			if (Objects.equals(plotBand.getId(), id)) {
				removePlotBand(plotBand);
				return;
			}
		}
	}

	/**
	 * Removes a plot band after render time.
	 */
	public void removePlotBand(final PlotBand plotBand) {
		List<PlotBand> plotBands = getPlotBands();
		if (plotBands == null)
			return; // nothing to do
		if (plotBand != null) {
			plotBands.remove(plotBand);
			plotBand.removeOptionDataListener(this);
			fireEvent(EventType.REMOVED, Attrs.plotband.toString(), plotBand,
					new DeferredCall() {
						public void execute(JSFunction func) {
							func.callFunction("removePlotBand",
									plotBand.getId());
						}
					});
		}
	}

	/**
	 * Returns a list of lines streching across the plot area, marking a
	 * specific value on one of the axes.
	 * <p>
	 * Default: null.
	 */
	public List<PlotLine> getPlotLines() {
		return Generics.cast(getAttr(Attrs.plotLines, null).asValue());
	}

	/**
	 * Sets a list of lines streching across the plot area, marking a specific
	 * value on one of the axes.
	 */
	public void setPlotLines(List<PlotLine> plotLines) {
		List<PlotLine> oldPlotLines = getPlotLines();
		if (oldPlotLines != null) {
			List<PlotLine> copy = new CopyOnWriteArrayList<PlotLine>(
					oldPlotLines);
			for (PlotLine plotLine : copy) {
				removePlotLine(plotLine);
			}
		}
		for (PlotLine plotLine : plotLines)
			addPlotLine(plotLine);
	}

	/**
	 * Adds a plot line after render time.
	 */
	public void addPlotLine(final PlotLine plotLine) {
		List<PlotLine> plotLines = getPlotLines();
		if (plotLines == null) {
			plotLines = new LinkedList<PlotLine>();
		}
		plotLine.addOptionDataListener(this);
		plotLines.add(plotLine);
		setAttr(Attrs.plotLines, plotLines);

		fireEvent(EventType.ADDED, Attrs.plotLine.toString(), plotLine,
				new DeferredCall() {
					public void execute(JSFunction func) {
						func.callFunction("addPlotLine", plotLine);
					}
				});
	}

	/**
	 * Removes a plot line with given id after render time.
	 */
	public void removePlotLine(final String id) {
		List<PlotLine> plotLines = getPlotLines();
		if (plotLines == null)
			return; // nothiing to do

		PlotLine target = null;
		for (PlotLine plotLine : plotLines) {
			if (Objects.equals(plotLine.getId(), id)) {
				target = plotLine;
				break;
			}
		}
		if (target != null) {
			target.removeOptionDataListener(this);
			plotLines.remove(target);
			fireEvent(EventType.REMOVED, Attrs.plotLine.toString(), target,
					new DeferredCall() {
						public void execute(JSFunction func) {
							func.callFunction("removePlotLine", id);
						}
					});
		}
	}

	/**
	 * Removes a plot line after render time.
	 */
	public void removePlotLine(final PlotLine plotLine) {
		List<PlotLine> plotLines = getPlotLines();
		if (plotLines == null)
			return; // nothing to do
		if (plotLine != null) {
			plotLines.remove(plotLine);
			plotLine.removeOptionDataListener(this);
			fireEvent(EventType.REMOVED, Attrs.plotband.toString(), plotLine,
					new DeferredCall() {
						public void execute(JSFunction func) {
							func.callFunction("removePlotBand",
									plotLine.getId());
						}
					});
		}
	}

	/**
	 * Returns whether to reverse the axis so that the highest number is closest
	 * to the origin. If the chart is inverted, the x axis is reversed by
	 * default.
	 * <p>
	 * Default: false.
	 */
	public boolean isReversed() {
		return getAttr(DAttrs.reversed, false).asBoolean();
	}

	/**
	 * Sets whether to reverse the axis so that the highest number is closest to
	 * the origin. If the chart is inverted, the x axis is reversed by default.
	 */
	public void setReversed(boolean reversed) {
		setAttr(DAttrs.reversed, reversed);
	}

	/**
	 * Returns whether to show the axis line and title when the axis has no
	 * data.
	 * <p>
	 * Default: true.
	 */
	public boolean isShowEmpty() {
		return getAttr(DAttrs.showEmpty, true).asBoolean();
	}

	/**
	 * Sets whether to show the axis line and title when the axis has no data.
	 */
	public void setShowEmpty(boolean showEmpty) {
		setAttr(DAttrs.showEmpty, showEmpty);
	}

	/**
	 * Returns whether to show the first tick label.
	 * <p>
	 * Default: true.
	 */
	public boolean isShowFirstLabel() {
		return getAttr(DAttrs.showFirstLabel, true).asBoolean();
	}

	/**
	 * Sets whether to show the first tick label.
	 */
	public void setShowFirstLabel(boolean showFirstLabel) {
		setAttr(DAttrs.showFirstLabel, showFirstLabel);
	}

	/**
	 * Returns whether to show the last tick label.
	 * <p>
	 * Default: false.
	 */
	public boolean isShowLastLabel() {
		return getAttr(DAttrs.showLastLabel, false).asBoolean();
	}

	/**
	 * Sets whether to show the last tick label.
	 */
	public void setShowLastLabel(boolean showLastLabel) {
		setAttr(DAttrs.showLastLabel, showLastLabel);
	}

	/**
	 * Returns for datetime axes, this decides where to put the tick between
	 * weeks. 0 = Sunday, 1 = Monday.
	 * <p>
	 * Default: 1.
	 */
	public Number getStartOfWeek() {
		return getAttr(DAttrs.startOfWeek, 1).asNumber();
	}

	/**
	 * Sets for datetime axes, this decides where to put the tick between weeks.
	 * 0 = Sunday, 1 = Monday.
	 */
	public void setStartOfWeek(Number startOfWeek) {
		setAttr(DAttrs.startOfWeek, startOfWeek, 1);
	}

	/**
	 * Returns whether to force the axis to start on a tick. Use this option
	 * with the <code>maxPadding</code> option to control the axis start.
	 * <p>
	 * Default: false.
	 */
	public boolean isStartOnTick() {
		return getAttr(DAttrs.startOnTick, false).asBoolean();
	}

	/**
	 * Sets whether to force the axis to start on a tick. Use this option with
	 * the {@link #setMaxPadding(Number)} option to control the axis start.
	 */
	public void setStartOnTick(boolean startOnTick) {
		setAttr(DAttrs.startOnTick, startOnTick);
	}

	/**
	 * Returns color for the main tick marks.
	 * <p>
	 * Default: "#C0D0E0".
	 */
	public String getTickColor() {
		return getAttr(DAttrs.tickColor, "#C0D0E0").asString();
	}

	/**
	 * Sets color for the main tick marks.
	 */
	public void setTickColor(String tickColor) {
		setAttr(DAttrs.tickColor, tickColor, "#C0D0E0");
	}

	/**
	 * Returns the interval of the tick marks in axis units. When
	 * <code>null</code>, the tick interval is computed to approximately follow
	 * the {@link #setTickPixelInterval(Number)} on linear and datetime axes. On
	 * categorized axes, a <code>null</code> tickInterval will default to 1, one
	 * category. Note that datetime axes are based on milliseconds, so for
	 * example an interval of one day is expressed as
	 * <code>24 * 3600 * 1000</code>.
	 * <p>
	 * On logarithmic axes, the tickInterval is based on powers, so a
	 * tickInterval of 1 means one tick on each of 0.1, 1, 10, 100 etc. A
	 * tickInterval of 2 means a tick of 0.1, 10, 1000 etc. A tickInterval of
	 * 0.2 puts a tick on 0.1, 0.2, 0.4, 0.6, 0.8, 1, 2, 4, 6, 8, 10, 20, 40
	 * etc.
	 * </p>
	 * <p>
	 * Default: null.
	 */
	public Number getTickInterval() {
		return getAttr(DAttrs.tickInterval, null).asNumber();
	}

	/**
	 * Sets the interval of the tick marks in axis units. When <code>null</code>
	 * , the tick interval is computed to approximately follow the
	 * {@link #setTickPixelInterval(Number)} on linear and datetime axes. On
	 * categorized axes, a <code>null</code> tickInterval will default to 1, one
	 * category. Note that datetime axes are based on milliseconds, so for
	 * example an interval of one day is expressed as
	 * <code>24 * 3600 * 1000</code>.
	 * <p>
	 * On logarithmic axes, the tickInterval is based on powers, so a
	 * tickInterval of 1 means one tick on each of 0.1, 1, 10, 100 etc. A
	 * tickInterval of 2 means a tick of 0.1, 10, 1000 etc. A tickInterval of
	 * 0.2 puts a tick on 0.1, 0.2, 0.4, 0.6, 0.8, 1, 2, 4, 6, 8, 10, 20, 40
	 * etc.
	 * </p>
	 * <p>
	 * Default: null.
	 */
	public void setTickInterval(Number tickInterval) {
		setAttr(DAttrs.tickInterval, tickInterval);
	}

	/**
	 * Returns the pixel length of the main tick marks.
	 * <p>
	 * Default: 5.
	 */
	public Number getTickLength() {
		return getAttr(DAttrs.tickLength, 5).asNumber();
	}

	/**
	 * Sets the pixel length of the main tick marks.
	 */
	public void setTickLength(Number tickLength) {
		setAttr(DAttrs.tickLength, tickLength, NOT_NULL_VALUE);
	}

	/**
	 * Returns if tickInterval is <code>null</code> this option sets the
	 * approximate pixel interval of the tick marks. Not applicable to
	 * categorized axis. Defaults to <code>72</code> for the Y axis and
	 * <code>100</code> for the X axis.
	 * <p>
	 * Default: null.
	 */
	public Number getTickPixelInterval() {
		return getAttr(DAttrs.tickPixelInterval, null).asNumber();
	}

	/**
	 * Sets if tickInterval is <code>null</code> this option sets the
	 * approximate pixel interval of the tick marks. Not applicable to
	 * categorized axis. Defaults to <code>72</code> for the Y axis and
	 * <code>100</code> for the X axis.
	 * <p>
	 * Default: null.
	 */
	public void setTickPixelInterval(Number tickPixelInterval) {
		setAttr(DAttrs.tickPixelInterval, tickPixelInterval);
	}

	/**
	 * Returns the position of the major tick marks relative to the axis line.
	 * Can be one of <code>inside</code> and <code>outside</code>.
	 * <p>
	 * Default: "outside".
	 * 
	 * @return tickPosition the position of the major tick marks relative to the
	 *         axis line
	 */
	public String getTickPosition() {
		return getAttr(DAttrs.tickPosition, "outside").asString();
	}

	/**
	 * Sets the position of the major tick marks relative to the axis line. Can
	 * be one of <code>inside</code> and <code>outside</code>.
	 */
	public void setTickPosition(String tickPosition) {
		if (!("inside".equals(tickPosition) || "outside".equals(tickPosition)))
			throw new IllegalArgumentException("Unsupported position: ["
					+ tickPosition + "]");
		setAttr(DAttrs.tickPosition, tickPosition, NOT_NULL_VALUE);
	}

	/**
	 * Returns a list defining where the ticks are laid out on the axis. This
	 * overrides the default behaviour of {@link #setTickPixelInterval(Number)}
	 * and {@link #setTickInterval(Number)}
	 * <p>
	 * Default: null.
	 */
	public List<Number> getTickPositions() {
		return Generics.cast(getAttr(DAttrs.tickPositions, null).asValue());
	}

	/**
	 * Sets a list defining where the ticks are laid out on the axis. This
	 * overrides the default behaviour of {@link #setTickPixelInterval(Number)}
	 * and {@link #setTickInterval(Number)}
	 * <p>
	 * Default: null.
	 */
	public void setTickPositions(List<Number> tickPositions) {
		setAttr(DAttrs.tickPositions, tickPositions);
	}

	/**
	 * Returns the pixel width of the major tick marks.
	 * <p>
	 * Default: 1.
	 */
	public Number getTickWidth() {
		return getAttr(DAttrs.tickWidth, 1).asNumber();
	}

	/**
	 * Sets the pixel width of the major tick marks.
	 */
	public void setTickWidth(Number tickWidth) {
		setAttr(DAttrs.tickWidth, tickWidth, 1);
	}

	/**
	 * Returns for categorized axes only. If "on" the tick mark is placed in the
	 * center of the category, if "between" the tick mark is placed between
	 * categories.
	 * <p>
	 * Default: "between".
	 */
	public String getTickmarkPlacement() {
		return getAttr(DAttrs.tickmarkPlacement, "between").asString();
	}

	/**
	 * Sets for categorized axes only. If "on" the tick mark is placed in the
	 * center of the category, if "between" the tick mark is placed between
	 * categories.
	 */
	public void setTickmarkPlacement(String tickmarkPlacement) {
		if (!("on".equals(tickmarkPlacement) || "between"
				.equals(tickmarkPlacement)))
			throw new IllegalArgumentException(
					"Unsupported tickmark placement: [" + tickmarkPlacement
							+ "]");
		setAttr(DAttrs.tickmarkPlacement, tickmarkPlacement, "between");
	}

	/**
	 * Returns the axis title, showing next to the axis line.
	 * <p>
	 * Default: an instance of {@link AxisTitle}.
	 */
	public AxisTitle getTitle() {
		AxisTitle title = (AxisTitle) getAttr(DAttrs.title);
		if (title == null) {
			title = new AxisTitle();
			setTitle(title);
		}
		return title;
	}

	/**
	 * Sets the axis title, showing next to the axis line.
	 */
	public void setTitle(AxisTitle title) {
		setAttr(DAttrs.title, title);
	}

	/**
	 * Sets the title text of the axis.
	 */
	public void setTitle(String title) {
		getTitle().setText(title);
	}

	/**
	 * Returns the type of axis. Can be one of <code>"linear"</code>,
	 * <code>"logarithmic"</code>, <code>"datetime"</code> or
	 * <code>"category"</code>. In a datetime axis, the numbers are given in
	 * milliseconds, and tick marks are placed on appropriate values like full
	 * hours or days. In a category axis, the <a href="#series.data">point
	 * names</a> of the chart's series are used for categories, if not a <a
	 * href="#Axis.categories">categories</a> array is defined.
	 * <p>
	 * Default: "linear".
	 */
	public String getType() {
		return getAttr(DAttrs.type, "linear").asString();
	}

	/**
	 * Sets the type of axis. Can be one of <code>"linear"</code>,
	 * <code>"logarithmic"</code>, <code>"datetime"</code> or
	 * <code>"category"</code>. In a datetime axis, the numbers are given in
	 * milliseconds, and tick marks are placed on appropriate values like full
	 * hours or days. In a category axis, the <a href="#series.data">point
	 * names</a> of the chart's series are used for categories, if not a <a
	 * href="#Axis.categories">categories</a> array is defined.
	 */
	public void setType(String type) {
		if (!("linear".equals(type) || "logarithmic".equals(type)
				|| "datetime".equals(type) || "category".equals(type)))
			throw new IllegalArgumentException("Unsupported type: [" + type
					+ "]");
		setAttr(DAttrs.type, type, "linear");
	}

	/**
	 * Removes the axis.
	 */
	public void remove() {
		fireEvent(EventType.DESTROYED, "axis", this, new DeferredCall() {
			public void execute(JSFunction func) {
				func.callFunction("remove");
			}
		});
		clearOptonDataListener();
	}

	// override
	public void onChange(OptionDataEvent event) {
		final Optionable target = event.getTarget();
		event.setCurrentTarget(this);
		event.addJSFunctionCall(new DeferredCall() {
			public void execute(JSFunction func) {
				if (target instanceof AxisLabels) {
					JSONObject json = new JSONObject();
					json.put("labels", target);
					func.callFunction("update", json);
				} else if (target instanceof AxisTitle) {
					func.callFunction("setTitle", target);
				} else if (target instanceof DateTimeLabelFormats) {
					JSONObject json = new JSONObject();
					json.put("dateTimeLabelFormats", target);
					func.callFunction("update", json);
				} else {
					func.callFunction("update", target);
				}
			}
		});
		fireEvent(event);
	}
}