'use client';

import React from 'react';

import type { DropdownMenuProps } from '@radix-ui/react-dropdown-menu';

import { setAlign } from '@udecode/plate-alignment';
import {TablePlugin} from '@udecode/plate-table/react';
import {useEditorRef, useEditorSelector, usePluginOption} from '@udecode/plate/react';
import {
  AlignCenterIcon,
  AlignJustifyIcon,
  AlignLeftIcon,
  AlignRightIcon,
} from 'lucide-react';

import { STRUCTURAL_TYPES } from './transforms';

import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuRadioGroup,
  DropdownMenuRadioItem,
  DropdownMenuTrigger,
  useOpenState,
} from './dropdown-menu';
import { ToolbarButton } from './toolbar';

const items = [
  {
    icon: AlignLeftIcon,
    value: 'left',
  },
  {
    icon: AlignCenterIcon,
    value: 'center',
  },
  {
    icon: AlignRightIcon,
    value: 'right',
  },
  {
    icon: AlignJustifyIcon,
    value: 'justify',
  },
];

export function AlignDropdownMenu({ children, ...props }: DropdownMenuProps) {
  const editor = useEditorRef();
  /* Potix: Fix toolbar value after grid cell selection */
  const selectedCells = usePluginOption(TablePlugin, 'selectedCells');
  const value = useEditorSelector(editor => {
    const fragment = selectedCells && selectedCells.length > 0
        ? selectedCells.flatMap(cell => editor.api.fragment<TElement>(cell, {structuralTypes: STRUCTURAL_TYPES}))
        : editor.api.fragment<TElement>(editor.selection, {structuralTypes: STRUCTURAL_TYPES});
    return editor.api.prop({defaultValue: 'start', getProp: node => node.align, nodes: fragment});
  }, [selectedCells]);

  const openState = useOpenState();
  const IconValue =
      items.find((item) => item.value === value)?.icon ?? AlignLeftIcon;

  return (
      <DropdownMenu modal={false} {...openState} {...props}>
        <DropdownMenuTrigger asChild>
          <ToolbarButton pressed={openState.open} tooltip="Align" isDropdown>
            <IconValue />
          </ToolbarButton>
        </DropdownMenuTrigger>

        <DropdownMenuContent className="min-w-0" align="start">
          <DropdownMenuRadioGroup
              value={value}
              onValueChange={(value: any) => {
                /* Potix: Fix toolbar action after grid cell selection */
                if (selectedCells && selectedCells.length > 0) {
                  selectedCells.forEach((cell) => {
                    const cellPath = editor.api.findPath(cell);
                    if (cellPath) setAlign(editor, {setNodesOptions: {at: cellPath}, value: value});
                  });
                } else setAlign(editor, {value: value});
                editor.tf.focus();
              }}
          >
            {items.map(({ icon: Icon, value: itemValue }) => (
                <DropdownMenuRadioItem key={itemValue} value={itemValue} hideIcon>
                  <Icon />
                </DropdownMenuRadioItem>
            ))}
          </DropdownMenuRadioGroup>
        </DropdownMenuContent>
      </DropdownMenu>
  );
}
