import {EditableProps, ElementStatic, LeafStatic, NodeComponents, pipeDecorate, SlateEditor} from '@udecode/plate';
import {AnyPlatePlugin} from '@udecode/plate/react';
import {type DecoratedRange, type Descendant, ElementApi, RangeApi, type Value,} from '@udecode/slate';
import React from 'react';
import {NodeEntry} from 'slate';

const defaultDecorate: (entry: NodeEntry) => DecoratedRange[] = () => [];

function Children({
                      children = [],
                      components,
                      decorate = defaultDecorate,
                      decorations = [],
                      editor,
                  }: {
    children: Descendant[];
    components: NodeComponents;
    decorate: EditableProps['decorate'];
    decorations: DecoratedRange[];
    editor: SlateEditor;
}): React.ReactElement {
    return (
        <React.Fragment>
            {children.map((child, i) => {
                const p = editor.api.findPath(child);

                let ds: DecoratedRange[] = [];

                if (p) {
                    const range = editor.api.range(p)!;
                    ds = decorate([child, p]);

                    for (const dec of decorations) {
                        const d = RangeApi.intersection(dec, range);

                        if (d) {
                            ds.push(d);
                        }
                    }
                }

                return ElementApi.isElement(child) ? (
                    <ElementStatic
                        key={i}
                        components={components}
                        decorate={decorate}
                        decorations={ds}
                        editor={editor}
                        element={child}
                    />
                ) : (
                    <LeafStatic
                        key={i}
                        components={components}
                        decorations={ds}
                        editor={editor}
                        leaf={child}
                    />
                );
            })}
        </React.Fragment>
    );
}

export type PlateStaticProps = {
    /** Node components to render. */
    components: NodeComponents;
    /** Editor instance. */
    editor: SlateEditor;
    style?: React.CSSProperties;
    /** Controlled value. Alias to `editor.children`. */
    value?: Value;
} & React.HTMLAttributes<HTMLDivElement>;

export function PlateStatic(props: PlateStaticProps): React.ReactElement {
    const {components, editor, value} = props;

    if (value) {
        editor.children = value;
    }

    const decorate = pipeDecorate(editor);

    let afterEditable: React.ReactNode = undefined,
        beforeEditable: React.ReactNode = undefined;

    editor.pluginList.forEach((plugin) => {
        const {
            render: {afterEditable: AfterEditable, beforeEditable: BeforeEditable},
        } = plugin as AnyPlatePlugin;

        if (AfterEditable) {
            afterEditable = (
                <>
                    {afterEditable}
                    <AfterEditable/>
                </>
            );
        }
        if (BeforeEditable) {
            beforeEditable = (
                <>
                    {beforeEditable}
                    <BeforeEditable/>
                </>
            );
        }
    });

    const content = (
        /* Potix: remove the slate-editor div */
        // <div
        //     className={clsx('slate-editor', className)}
        //     data-slate-editor
        //     data-slate-node="value"
        //     {...rest}
        // >
        <Children
            components={components}
            decorate={decorate}
            decorations={[]}
            editor={editor}
        >
            {editor.children}
        </Children>
        // </div>
    );

    let aboveEditable: React.ReactNode = (
        <>
            {beforeEditable}
            {content}
            {afterEditable}
        </>
    );

    editor.pluginList.forEach((plugin) => {
        const {
            render: {aboveEditable: AboveEditable},
        } = plugin as AnyPlatePlugin;

        if (AboveEditable) {
            aboveEditable = <AboveEditable>{aboveEditable}</AboveEditable>;
        }
    });

    return aboveEditable;
}
